# SPDX-License-Identifier: BSD-3-Clause
import time
import os
import requests
import requests_unixsocket

requests_unixsocket.monkeypatch()


class AdariClient:
    """
    Utility class to make communicate with the ADARI server

    Parameters
    ----------
    url : str
        The URL where the server can be contacted
    """

    def __init__(self, url):
        self._url = url

    def connect(self, timeout: int):
        """
        Ping the server to check if it is alive

        Parameters
        ----------
        timeout : int
        The amount of seconds to wait until no more attempts are done.
        """
        connected = False
        now = time.time()
        # Busy poll until the server is available.
        # In most of the cases this should be a fraction of a second
        while not connected and ((time.time() - now) < timeout):
            try:
                res_req = requests.get(self._url + "/ping")
                connected = True
            except Exception:
                pass
        # This will be executed in the case the timeout was reached,
        # so that the proper exception is raised.
        # Also in the case timeout <= 0, in which case the loop is ommited.
        if not connected:
            res_req = requests.get(self._url + "/ping")
            res_req.raise_for_status()

    def loadreport(self, report_name: str):
        """
        Ask the server to load a report

        Parameters:
        -----------
        report_name : str
            The name of the report to load
        """
        res_req = requests.get(self._url + "/loadreport/" + report_name)
        res_req.raise_for_status()
        if res_req.text == "null":  # Default content if OK
            return
        elif int(res_req.text):
            return int(res_req.text)

    def runreport(self, report_name: str, inputs: str, outputs: str):
        """
        Ask the server to run a report

        Parameters:
        -----------
        report_name : str
            The name of the report to run
        inputs : str
            The set of frames with the input filenames and categories
        outputs : str
            A JSON filename with the created artifacts and its metadata
        """
        # Ensure that the file with the inputs has an absolute path
        inputs_abspath = os.path.abspath(inputs)

        # Now we put all requests for runreport through as json payload post request
        res_req = requests.post(
            self._url + "/runreport",
            json={"report_name": report_name, "sof": inputs_abspath, "output": outputs},
        )
        res_req.raise_for_status()
        if res_req.text == "null":  # Default content if OK
            return
        elif int(res_req.text):
            return int(res_req.text)

    def server_supports_driver(self, driver: str):
        """
        Ask the server whether a given driver is supported.
        See documentation on AdariServerController for a description of driver

        Parameters:
        -----------
        driver : str
            Driver name. Possible options are 'png', 'desktop', 'web'.
        """
        res_req = requests.get(self._url + "/driver_support/" + driver)
        res_req.raise_for_status()
        return res_req
