# SPDX-License-Identifier: BSD-3-Clause
import numpy as np
import os
from adari_core.plots.panel import Panel
from adari_core.plots.images import ImagePlot, CentralImagePlot
from adari_core.plots.points import LinePlot
from adari_core.plots.text import TextPlot
from adari_core.report import AdariReportBase


class MasterAstrometryReport(AdariReportBase):
    """Master report for Astrometry panels"""

    data_extensions = None
    default_extension = 0
    image_category = None
    fov_per_panel = 24
    plot_zoom_fov = False
    panel_kwargs = dict(x=6, y=5, height_ratios=[1, 4, 4, 4, 4])

    def __init__(self, name):
        super().__init__(name)

        if self.image_category is None:
            raise NotImplementedError("Provide a value for the FoV image category")

    def parse_sof(self):
        raise NotImplementedError(
            "MasterAstrometryReport is a template only, "
            "the child Report is responsible for "
            "defining parse_sof"
        )

    def image_plot(
        self, hdu, img_kwargs={}, zoom_in=False, zoom_in_extent=100, title_num=None
    ):
        """Create a plot from a given HDU image.

        Description
        -----------
        Given an image HDU, this function will create an ImagePlot and a zoom-in plot
        of the central region (if applicable, default is None).
        """
        zoom_img_plot = None
        if hdu.data is not None:
            img_plot = ImagePlot(
                **img_kwargs,
                title=f"{title_num}",
            )
            img_plot.add_data(hdu.data)
            if zoom_in:
                if "title" not in img_kwargs.keys():
                    img_kwargs["title"] = (
                        f"{title_num} Central ({zoom_in_extent}x{zoom_in_extent})"
                    )
                else:
                    img_kwargs[
                        "title"
                    ] += f"{title_num} Central ({zoom_in_extent}x{zoom_in_extent})"
                zoom_img_plot = CentralImagePlot(
                    img_plot, **img_kwargs, extent=zoom_in_extent
                )
        else:
            img_plot = LinePlot(
                legend=False,
                y_label="y",
                x_label="x",
                title=f"{title_num} (NO DATA)",
            )
            if zoom_in:
                zoom_img_plot = LinePlot(
                    legend=False,
                    y_label="y",
                    x_label="x",
                    title=f"{title_num}NO DATA Central ({zoom_in_extent}x{zoom_in_extent})",
                )
        return img_plot, zoom_img_plot

    def img_grid_panel(
        self,
        hdu_list,
        textrow=True,
        zoom_in=False,
        zoom_in_extent=100,
        img_kw_list=None,
        img_ext=(1, 1),
    ):
        """Create a panel of images from a given list of HDU.

        Description
        -----------
        Given a list of HDU, this function will create a panel that
        contains a mosaic of all images (including zoom-in plots if applicable).

        Parameters
        ----------
        - hdu_list: list
            List of HDU images that will be used to compute the plots. If the HDU
            does not contain data, an empty placeholder LinePlot will be created instead.
            The order of HDU in the list will be mapped to a rectangular grid row by row.
        - textrow: bool, default=True
            If true, the panel will include an extra row at the top of the panel to include
            metadata information.
        - zoom_in: bool, defaul=False
            If true, an extra panel for every HDU image will be created on the right of every
            image in the final panel. This will be considered when accounting for the number
            of row and columns required.
        - zoom_in_extent: int, default=100
            Size of the zoom-in cutout image in pixels.
        - img_kw_list: dict, default=None
            A list of dicitonaries that will be used as parameters for the image (and zoom image)
            plots.
        - img_ext: 2D tuple, default=(1,2)
            Plot image extension in the Panel grid. This might be useful to enlarge the size of the images.
            The number of rows and columns is increased accordingly by the function.

        Returns
        -------
        - panel: Panel
            A panel containing all the HDU images plots and zoom-in plots.

        Notes
        -----
        This function will make use of the report attribute panel_kwargs, which corresponds to a dictionary
        containing the basic information to build the panel.

        """
        # Get the number of rows and columns containing data in the panel
        nrow, ncol = self.panel_kwargs["y"], self.panel_kwargs["x"]

        # Add extra plots to include text and/or zoom-in plots
        if textrow:
            self.panel_kwargs["y"] += 1
            # TODO: It would be good if this is corrected automatically
            # self.panel_kwargs['heigh'] = ...
        if zoom_in:
            self.panel_kwargs["x"] *= 2

        # Account for image extensions
        self.panel_kwargs["x"] *= img_ext[0]
        self.panel_kwargs["y"] = (self.panel_kwargs["y"] - int(textrow)) * img_ext[
            1
        ] + int(textrow)
        if "height_ratios" in self.panel_kwargs:
            self.panel_kwargs["height_ratios"] = [
                max(self.panel_kwargs["height_ratios"])
            ] * self.panel_kwargs["y"]
            if textrow:
                self.panel_kwargs["height_ratios"][0] = 1

        # Build the empty panel
        panel = Panel(**self.panel_kwargs)
        # Get the number of images to plot
        n_images = len(hdu_list)
        # Check image metadata
        if img_kw_list is None:
            img_kw_list = [{}] * n_images

        if n_images != len(img_kw_list):
            raise ValueError(
                "The length of the input HDU list must be the same as the extensions and image keywords lists"
            )
        # Fill the panel
        for idx in range(n_images):
            ifu_num = []
            for ext in self.data_extensions:
                ifu_num.append(ext)
            py, px = np.unravel_index(idx, (nrow, ncol))
            px = px * (1 + zoom_in) * img_ext[0]
            py = py * img_ext[1] + 1 * int(textrow)
            hdu = hdu_list[idx]
            img, zoom_img = self.image_plot(
                hdu,
                img_kwargs=img_kw_list[idx],
                zoom_in=zoom_in,
                zoom_in_extent=zoom_in_extent,
                title_num=ifu_num[idx],
            )
            panel.assign_plot(img, px, py, xext=img_ext[0], yext=img_ext[1])
            if zoom_in:
                panel.assign_plot(
                    zoom_img, px + img_ext[0], py, xext=img_ext[0], yext=img_ext[1]
                )
        return panel

    def set_data_extensions(self):
        """Set the default extensions to be used for each HDUL."""
        self.data_extensions = [self.default_extension] * len(self.hdus)

    def generate_panels(self, **img_grid_kwargs):
        panels = {}
        # Check if data extensions are provided
        if self.data_extensions is None:
            self.set_data_extensions()
        for hdu_dict in self.hdus:
            hdul = hdu_dict[self.image_category]
            list_of_hdu = []
            for ext in self.data_extensions:
                list_of_hdu.append(hdul[ext])
            panel = self.img_grid_panel(list_of_hdu, **img_grid_kwargs)
            # Metadata
            text_col = (
                str(hdul["PRIMARY"].header.get("INSTRUME")),
                "PRO CATG: " + str(hdul["PRIMARY"].header.get("HIERARCH ESO PRO CATG")),
                "FILE NAME: " + os.path.basename(hdul.filename()),
                "RAW1 NAME: "
                + str(hdul["PRIMARY"].header.get("HIERARCH ESO PRO REC1 RAW1 NAME")),
            )
            text = TextPlot(columns=1, v_space=0.3)
            text.add_data(text_col)
            panel.assign_plot(text, 0, 0, xext=3)

            input_files = [hdul.filename()]

            addme = {
                "report_name": f"{self.name}",
                "report_description": f"{self.name}",
                "report_tags": [],
                "ext": f"{self.data_extensions}",
                "input_files": input_files,
            }
            panels[panel] = addme

        return panels
