# SPDX-License-Identifier: BSD-3-Clause
"""
This module specifies a TextPlot class for displaying textual information.
"""

from .plot import Plot


class TextPlot(Plot):
    """
    This Plot type is designed to display textual information.

    Parameters
    ----------
    columns : int, optional
        The number of columns to show the various text lines in.
        Defaults to 1.
    v_space : float, optional
        The spacing between the base location of each text line
        in the y-direction, expressed in units of plot height.
        Defaults to 0.1 (i.e., one-tenth of
        the Plot height).
    fontsize : float, optional
        Size of the printed font.
        Defaults to 10.
    halign : str, optional
        Horizontal alignment of the (xref, yref) position relative to the
        string.
        Options are 'left', 'center', 'right'.
        Defaults to 'left'.
    valign : str, optional
        Vertical alignment of the (xref, yref) position relative to the
        string.
        Options are 'top', 'center', 'bottom'.
        Defaults to 'baseline' (between bottom and center).
    xref : float, optional
        Reference point in x-axis for text alignment in plot. Nominal
        plot boundaries are 0 to 1.
        Defaults to 0 (left edge).
    yref : float, optional
        Reference point in y-axis for text alignment in plot. Nominal
        plot boundaries are 0 (bottom) to 1 (top).
        Defaults to 1 (top edge).
    """

    def __init__(self, *args, columns=1, v_space=0.1, **kwargs):
        self.columns = columns
        self.v_space = v_space
        super().__init__(*args, **kwargs)
        self._fontsize = 10
        self._halign = "left"
        self._valign = "baseline"
        self._xref = None
        self._yref = None
        self.fontsize = kwargs.get("fontsize")
        self.halign = kwargs.get("halign")
        self.valign = kwargs.get("valign")
        self.xref = kwargs.get("xref")
        self.yref = kwargs.get("yref")
        if self._data is None:
            self._data = []

    def __str__(self):
        return "TextPlot"

    @property
    def columns(self):
        """int : The number of columns of text."""
        return self._columns

    @columns.setter
    def columns(self, c):
        try:
            assert isinstance(c, int), "Number of columns must be an int"
            assert c > 0, "Number of columns must be > 0"
        except AssertionError as e:
            raise ValueError(str(e))
        self._columns = c

    @property
    def v_space(self):
        """float : The vertical spacing between text lines in the same
        column."""
        return self._v_space

    @v_space.setter
    def v_space(self, f):
        try:
            assert isinstance(f, float), "v_space must be a float"
            assert 0 < f < 1, "v_space must be 0 < v < 1"
        except AssertionError as e:
            raise ValueError(str(e))
        self._v_space = f

    def _is_data_valid(self, data, fail_on_none=True, error_on_fail=True):
        s = super()._is_data_valid(
            data, fail_on_none=fail_on_none, error_on_fail=error_on_fail
        )
        if s is not None:
            return s

        try:
            if isinstance(data, str):
                return True
            tmp_data = list(data)
            for i, s in enumerate(tmp_data):
                assert isinstance(s, str), "{} is not a string".format(str(s))
                return True
        except TypeError:
            raise ValueError(
                "Unable to convert the input iterable into a " "list for checking"
            )
        except AssertionError as e:
            raise ValueError("Invalid data passed to TextPlot: {}".format(str(e)))

    def add_data(
        self,
        data,
        fontsize=None,
        halign=None,
        valign=None,
        xref=None,
        yref=None,
    ):
        """
        Add text data to this Plot.

        Data added is appended to any string data already existing in this
        TextPlot.

        Blank lines can be inserted by adding the empty string.

        Parameters
        ----------
        data : str, or iterable of str
            Either a string to add to this TextPlot's data, or an iterable
            of strings to be added.

        Raises
        -------
        ValueError
            If the data is invalid.
        """
        self._is_data_valid(data, fail_on_none=True, error_on_fail=True)
        if self._data is None:
            self._data = []
        # Append the data, then flatten it in case a list of strings was
        # passed in
        if isinstance(data, str):
            self._data.append(data)
        else:
            self._data += list(data)
        if fontsize is not None:
            self._fontsize = fontsize
        if halign is not None:
            if halign in ["center", "left", "right"]:
                self._halign = halign
            else:
                del self._data  # Don't leave half-baked definition
                raise ValueError(
                    f"{halign} is not a valid halign parameter; "
                    "pass one of 'left', 'center', 'right'"
                )
        if valign is not None:
            if valign in ["center", "top", "bottom"]:
                self._valign = valign
            else:
                del self._data  # Don't leave half-baked definition
                raise ValueError(
                    f"{valign} is not a valid valign parameter; "
                    "pass one of 'top', 'center', 'bottom'"
                )
        if xref is not None:
            self._xref = xref
        if yref is not None:
            self._yref = yref

    def metadata(self, master_im, ext):
        instru_name = str(master_im["PRIMARY"].header.get("INSTRUME"))
        ext_name = "EXTNAME: " + str(
            master_im[ext].header.get("EXTNAME", "Name not found")
        )

        procatg_value = "PRO CATG: " + str(
            master_im["PRIMARY"].header.get("HIERARCH ESO PRO CATG")
        )
        filename = "FILE NAME: " + str(master_im["PRIMARY"].header.get("PIPEFILE"))
        ins_mode = "INS MODE: " + str(
            master_im["PRIMARY"].header.get("HIERARCH ESO INS MODE")
        )
        raw_name = "RAW1 NAME: " + str(
            master_im["PRIMARY"].header.get("HIERARCH ESO PRO REC1 RAW1 NAME")
        )
        binx = "BIN X: " + str(master_im["PRIMARY"].header.get("HIERARCH ESO DET BINX"))
        biny = "BIN Y: " + str(master_im["PRIMARY"].header.get("HIERARCH ESO DET BINY"))

        return (
            instru_name,
            ext_name,
            procatg_value,
            raw_name,
            filename,
            ins_mode,
            binx,
            biny,
            "",
        )
