# SPDX-License-Identifier: BSD-3-Clause
import logging
import logging.config

from multiprocessing import Process

logger = logging.getLogger(__name__)


class AdariServerController:
    """
    Utility class to make communicate with the ADARI server

    A server is configured with a driver and a renderer.
    The driver is a user-facing parameter that specifies, from the user
    point of view, where the report will "appear". The defined drivers
    are 'png', which will create a static PNG file for each panel.
    'pdf', which will create a static PDF file with all the panels, one
    per page.
    'desktop', which will make the reports appearing into a desktop
    GUI interface and 'web', which wil make the report appear in a web
    GUI interface.

    A renderer is the specific technology used to display it. The
    current ones are matplotlib and bokeh. The renderer is internal to the
    server whereas the driver is a parameter the user sets. There is no
    one-to-one correspondence between driver and renderer. For instance,
    the 'png' and 'desktop' drivers can use both the matplotlib renderer.
    But they could also be implemented with bokeh (it has capabilities to
    save to PNG and also can be used in a desktop application with electron).

    In the current design a server has a single renderer

    Parameters
    ----------
    driver : str
        The driver that the user has specified
    renderer_config: map
        The configuration of the renderer. This map is interpreted by
        each server derived class based on the expected configurations
        for the renderer.

    """

    def __init__(self, driver: str, renderer_config: map, **kwargs):
        self.driver = driver
        self.renderer_config = renderer_config
        self.server = None
        self.uds = kwargs.get("uds", None)

    def start(self):
        logger.info("AdariServerController:start")
        self.server_launch_process = Process(
            target=self.run,
            kwargs={
                "log_level": logger.getEffectiveLevel(),
                "renderer_config": self.renderer_config,
                "uds": self.uds,
            },
        )
        self.server_launch_process.start()

    def run(self, **kwargs):
        if self.driver == "png":
            from adari_core.server.server_png import AdariServerPng

            self.server = AdariServerPng(kwargs["renderer_config"], uds=kwargs["uds"])
        if self.driver == "pdf":
            from adari_core.server.server_pdf import AdariServerPdf

            self.server = AdariServerPdf(kwargs["renderer_config"], uds=kwargs["uds"])
        if self.driver == "json":
            from adari_core.server.server_json import AdariServerJSON

            self.server = AdariServerJSON(kwargs["renderer_config"], uds=kwargs["uds"])
        # May want to pass some tcp related info to AdariServerBokeh here (cannot use uds)
        #        if self.driver == "bokeh":
        #            self.server = AdariServerBokeh(self.renderer_config)
        self.server.run(**kwargs)

    def stop(self):
        if hasattr(self, "server_launch_process"):
            logger.info("Sending terminate signal to adari server process")
            self.server_launch_process.terminate()
            self.server_launch_process.join(2)
            if self.server_launch_process.is_alive():
                self.server_launch_process.kill()
