import logging
import threading
from concurrent.futures.thread import ThreadPoolExecutor
from datetime import timedelta
from typing import Callable, List
from uuid import UUID

from edps.config.configuration import Configuration
from edps.interfaces.JobsRepository import JobsRepository


class Cleaner:
    def __init__(self, config: Configuration, delete_job: Callable[[UUID], List[UUID]], repository: JobsRepository):
        self.cleanup_older_than = timedelta(seconds=config.cleanup_older_than_seconds)
        self.cleanup_check_period_seconds = config.cleanup_check_period_seconds
        self.delete_job = delete_job
        self.repository = repository
        self.executor_pool = ThreadPoolExecutor(max_workers=1)
        self.tick = threading.Event()
        self.logger = logging.getLogger('Cleaner')
        if config.cleanup_enabled:
            self.executor_pool.submit(self.cleanup_round)

    def cleanup_round(self):
        if not self.tick.wait(self.cleanup_check_period_seconds):
            self.logger.debug("Periodic cleaner loop")
            jobs_to_remove = self.repository.find_jobs_to_remove(self.cleanup_older_than)
            self.logger.info("Periodic cleaner removes %s", jobs_to_remove)
            while jobs_to_remove:
                job_id = jobs_to_remove.pop()
                try:
                    deleted_cascade = self.delete_job(job_id)
                    jobs_to_remove = [job_id for job_id in jobs_to_remove if job_id not in deleted_cascade]
                except Exception as e:
                    self.logger.warning("Cleanup round failed: %s", str(e))
            self.logger.debug("Periodic cleaner done")
            self.executor_pool.submit(self.cleanup_round)

    def shutdown(self):
        self.logger.info("Shutting down cleaner...")
        self.executor_pool.shutdown(wait=False)
        self.tick.set()
        self.logger.info("Cleaner shut down")
