from typing import List, Optional

from pydantic import BaseModel


class AssociationEntry(BaseModel):
    name: Optional[str] = None
    category: Optional[str] = None
    mjdobs: Optional[float] = None

    @classmethod
    def from_dict(cls, d):
        return cls(name=d['name'],
                   category=d['category'],
                   mjdobs=d['mjdobs'])

    class Config:
        json_schema_extra = {
            "example": {
                "name": "/some/path/file.fits",
                "category": "BIAS",
                "mjdobs": 56292.5
            }
        }

    def __repr__(self):
        return f'{self.name} {self.category} {self.mjdobs}'

    def __str__(self):
        return repr(self)


class AssociationResultDTO(BaseModel):
    files: Optional[List[AssociationEntry]] = None
    jobs: Optional[List[AssociationEntry]] = None
    complete: Optional[bool] = None
    optional: Optional[bool] = None
    level: Optional[float] = None
    task_name: Optional[str] = None

    @classmethod
    def from_dict(cls, d):
        return cls(
            files=[AssociationEntry.from_dict(f) for f in d.get('files', [])],
            jobs=[AssociationEntry.from_dict(f) for f in d.get('jobs', [])],
            complete=d['complete'],
            optional=d['optional'],
            level=d['level'],
            task_name=d['task_name'],
        )

    class Config:
        json_schema_extra = {
            "example": {
                "files": [AssociationEntry.Config.json_schema_extra['example']],
                "jobs": [AssociationEntry.Config.json_schema_extra['example']],
                "complete": True,
                "optional": False,
                "level": 1,
                "task_name": "bias",
            }
        }

    def __repr__(self):
        result = [f'Association name: {self.task_name}',
                  f'Association level: {self.level}',
                  f'Complete: {self.complete}',
                  f'Optional: {self.optional}']
        for f in self.files:
            result.append(str(f))
        for job in self.jobs:
            result.append(str(job))
        return '\n'.join(result)

    def __str__(self):
        return repr(self)
