from datetime import datetime
from enum import Enum
from typing import List, Dict, Optional

from pydantic import BaseModel

from edps.client.FitsFile import FitsFile
from edps.client.ProcessingJob import LogEntry, ReportEntry
from edps.client.ProcessingJobStatus import ProcessingJobStatus
from edps.interfaces.JobsRepository import ExecutionResult


class LoopRequestDTO(BaseModel):
    job_id: str
    parameters: Optional[Dict[str, str]] = None
    inputs: Optional[List[FitsFile]] = None

    @classmethod
    def from_dict(cls, d):
        return cls(
            job_id=d['job_id'],
            parameters=d.get("parameters", None),
            inputs=d.get("inputs", None),
        )

    def __repr__(self):
        return self.model_dump_json()

    class Config:
        json_schema_extra = {
            "example": {
                "job_id": "5c04c6ce-5dd0-4a32-a9b8-41b55f66319c",
                "parameters": {"a": "b"},
                "inputs": [FitsFile.Config.json_schema_extra['example']]
            }
        }


class LoopOnceRequestDTO(BaseModel):
    job_ids: List[str] = []
    parameters: Dict[str, str] = {}

    @classmethod
    def from_dict(cls, d):
        return cls(
            job_ids=d.get("job_ids", []),
            parameters=d.get("parameters", {})
        )

    def __repr__(self):
        return self.model_dump_json()

    class Config:
        json_schema_extra = {
            "example": {
                "job_ids": ["5c04c6ce-5dd0-4a32-a9b8-41b55f66319c"],
                "parameters": {"a": "b"}
            }
        }


class BreakpointedJobState(str, Enum):
    RUNNING = "RUNNING"
    STOPPED_ON_BREAKPOINT = "STOPPED_ON_BREAKPOINT"

    def __dict__(self):
        return {self.name: self.value}


class BreakpointStateDTO(BaseModel):
    job_id: str
    state: BreakpointedJobState
    next_parameters: Optional[Dict[str, str]] = None
    next_inputs: Optional[List[FitsFile]] = None

    @classmethod
    def from_dict(cls, d):
        return cls(
            job_id=d["job_id"],
            state=BreakpointedJobState(d["state"]),
            next_parameters=d.get("next_parameters", {}),
            next_inputs=d.get("next_inputs", [])
        )

    def __repr__(self):
        return self.model_dump_json()

    class Config:
        json_schema_extra = {
            "example": {
                "job_id": "5c04c6ce-5dd0-4a32-a9b8-41b55f66319c",
                "state": "STOPPED_ON_BREAKPOINT",
                "next_parameters": {"a": "b"},
                "next_inputs": [FitsFile.Config.json_schema_extra['example']]
            }
        }


class BreakpointsStateDTO(BaseModel):
    breakpoints: List[BreakpointStateDTO] = []

    @classmethod
    def from_dict(cls, d):
        return cls(
            breakpoints=[BreakpointStateDTO.from_dict(brk) for brk in d.get("breakpoints", [])]
        )

    def __repr__(self):
        return self.model_dump_json()

    class Config:
        json_schema_extra = {
            "example": {
                "breakpoints": [BreakpointStateDTO.Config.json_schema_extra['example']]
            }
        }


class BreakpointInspectResultDTO(BaseModel):
    input_files: List[FitsFile] = []
    output_files: List[FitsFile] = []
    logs: List[LogEntry] = []
    reports: List[ReportEntry] = []
    recipe_parameters: Dict[str, str] = {}
    completion_date: Optional[str] = None
    status: ProcessingJobStatus

    @classmethod
    def from_dict(cls, d):
        return cls(
            completion_date=d['completion_date'],
            status=ProcessingJobStatus(d['status']),
            input_files=[FitsFile.from_dict(file) for file in d.get('input_files', [])],
            output_files=[FitsFile.from_dict(file) for file in d.get('output_files', [])],
            reports=[ReportEntry.from_dict(report) for report in d.get('reports', [])],
            logs=[LogEntry.from_dict(log) for log in d.get('logs', [])],
            recipe_parameters=d.get('recipe_parameters', {})
        )

    @classmethod
    def from_execution_result(cls, result: ExecutionResult):
        return cls(
            completion_date=result.completion_date,
            status=result.status,
            input_files=result.input_files,
            output_files=result.output_files,
            reports=result.reports,
            logs=result.logs,
            recipe_parameters=result.recipe_parameters
        )

    def __repr__(self):
        return self.model_dump_json()

    class Config:
        json_schema_extra = {
            "example": {
                "completion_date": datetime.now().isoformat(),
                "status": ProcessingJobStatus.COMPLETED,
                "input_files": [FitsFile.Config.json_schema_extra['example']],
                "output_files": [FitsFile.Config.json_schema_extra['example']],
                "reports": [ReportEntry.Config.json_schema_extra['example']],
                "logs": [LogEntry.Config.json_schema_extra['example']],
                "recipe_parameters": {"a": "b"}
            }
        }
