import json
from typing import List, Callable, TypeVar, Dict
from uuid import UUID

import requests

from edps.client.BreakpointsStateDTO import BreakpointsStateDTO, LoopOnceRequestDTO, LoopRequestDTO, \
    BreakpointInspectResultDTO
from edps.client.CalselectorRequest import CalselectorRequest
from edps.client.Error import RestError
from edps.client.FitsFile import FitsFile
from edps.client.FlatOrganization import DatasetsDTO
from edps.client.GraphType import GraphType
from edps.client.JobInfo import JobInfo
from edps.client.ProcessingJob import ProcessingJob
from edps.client.ProcessingJobStatus import JobStatus
from edps.client.ProcessingRequest import ProcessingRequest
from edps.client.ProcessingResponse import ProcessingResponse
from edps.client.Rejection import Rejection
from edps.client.RunReportsRequestDTO import RunReportsRequestDTO
from edps.client.WorkflowDTO import WorkflowDTO, CalselectorJob
from edps.client.WorkflowStateDTO import WorkflowStateDTO
from edps.client.monad import Either
from edps.client.search import SearchFilter
from edps.phase3.phase3 import Phase3Configuration
from edps.utils import as_json

T = TypeVar('T')


class EDPSClient(object):
    REQUESTS_PATH = "/requests"
    CLASSIFY_PATH = REQUESTS_PATH + "/classify"
    CALSELECTOR_PATH = REQUESTS_PATH + "/calselector"
    ORGANISE_PATH = REQUESTS_PATH + "/organise"
    ORGANISE_FLAT_PATH = REQUESTS_PATH + "/organise_flattened"
    JOBS_PATH = "/jobs"
    JOBS_FILTER_PATH = JOBS_PATH + "/filter"
    SCHEDULED_JOBS_PATH = JOBS_PATH + "/scheduled"
    JOB_DETAILS_PATH = JOBS_PATH + "/{job_id}"
    JOB_STATUS_PATH = JOBS_PATH + "/{job_id}/status"
    REPORT_PATH = JOBS_PATH + "/{job_id}/reports/{report_name}"
    LOG_PATH = JOBS_PATH + "/{job_id}/logs/{log_name}"
    WORKFLOWS_PATH = "/workflows"
    WORKFLOW_PATH = WORKFLOWS_PATH + "/{workflow_name}"
    WORKFLOW_DIR_PATH = WORKFLOWS_PATH + "/{workflow_name}/dir"
    WORKFLOW_STATE_PATH = WORKFLOWS_PATH + "/{workflow_name}/state"
    WORKFLOW_JOBS_PATH = WORKFLOWS_PATH + "/{workflow_name}/jobs"
    WORKFLOW_JOB_PATH = WORKFLOW_JOBS_PATH + "/{job_id}"
    GRAPH_PATH = WORKFLOWS_PATH + "/{workflow_name}/graph"
    ASSOC_MAP_PATH = WORKFLOWS_PATH + "/{workflow_name}/associationMap"
    RESET_PATH = WORKFLOWS_PATH + "/{workflow_name}/reset"
    UPDATE_CALIB_PATH = WORKFLOWS_PATH + "/{workflow_name}/updateCalibDb"
    TARGETS_PATH = WORKFLOWS_PATH + "/{workflow_name}/targets"
    PARAMETER_SETS_PATH = WORKFLOWS_PATH + "/{workflow_name}/parameterSets"
    DEFAULT_PARAMS_PATH = WORKFLOWS_PATH + "/{workflow_name}/{task_name}/defaultParameters"
    RECIPE_PARAMS_PATH = WORKFLOWS_PATH + "/{workflow_name}/{task_name}/recipeParameters"
    REJECT_JOB_PATH = JOBS_PATH + "/{job_id}/reject"
    DELETE_JOB_PATH = JOBS_PATH + "/{job_id}/delete"
    DELETE_JOBS_PATH = JOBS_PATH + "/delete"
    RESUBMIT_JOB_PATH = JOBS_PATH + "/{job_id}/resubmit"
    ASSOC_REPORT_PATH = JOBS_PATH + "/{job_id}/associationReport"

    BREAKPOINTS_PATH = "/breakpoints"
    LOOP_EXECUTION = BREAKPOINTS_PATH + "/loop"
    LOOP_EXECUTION_ONCE = BREAKPOINTS_PATH + "/loop_once"
    CREATE_BREAKPOINT = BREAKPOINTS_PATH + "/{job_id}"
    REMOVE_BREAKPOINT = BREAKPOINTS_PATH + "/{job_id}"
    CONTINUE_EXECUTION = BREAKPOINTS_PATH + "/{job_id}/continue"
    INSPECT_BREAKPOINT = BREAKPOINTS_PATH + "/{job_id}/inspect"

    STOP_PROCESSING_PATH = "/killall"
    SHUTDOWN_PATH = "/shutdown"
    VERSION_PATH = "/version"

    PIPELINES_PATH = "/pipelines"

    REPORT_EXECUTION_PATH = "/report"

    PHASE3_PACKAGE_PATH = "/phase3"

    def __init__(self, host, port, scheme="http", root_path=""):
        self.endpoint = scheme + "://" + host + ":" + str(port) + root_path

    def submit_processing_request(self, payload: ProcessingRequest) -> Either[ProcessingResponse, RestError]:
        payload = as_json(payload)
        r = requests.post(self.endpoint + EDPSClient.REQUESTS_PATH, data=payload,
                          headers={'Content-type': 'application/json'})
        return EDPSClient.handle_error(r, lambda response: ProcessingResponse.from_dict(json.loads(response.content)))

    def submit_to_classify(self, payload: ProcessingRequest) -> Either[List[FitsFile], RestError]:
        payload = as_json(payload)
        r = requests.post(self.endpoint + EDPSClient.CLASSIFY_PATH, data=payload,
                          headers={'Content-type': 'application/json'})
        return EDPSClient.handle_error(r, lambda resp: [FitsFile.from_dict(x) for x in json.loads(resp.content)])

    def submit_to_calselector(self, payload: CalselectorRequest) -> Either[List[CalselectorJob], RestError]:
        payload = as_json(payload)
        r = requests.post(self.endpoint + EDPSClient.CALSELECTOR_PATH, data=payload,
                          headers={'Content-type': 'application/json'})
        return EDPSClient.handle_error(r, lambda response: [CalselectorJob.from_dict(x) for x in
                                                            json.loads(response.content)])

    def submit_to_organise(self, payload: ProcessingRequest) -> Either[List[JobInfo], RestError]:
        payload = as_json(payload)
        r = requests.post(self.endpoint + EDPSClient.ORGANISE_PATH, data=payload,
                          headers={'Content-type': 'application/json'})
        return EDPSClient.handle_error(r, lambda response: [JobInfo.from_dict(x) for x in json.loads(response.content)])

    def submit_to_organise_flat(self, payload: ProcessingRequest) -> Either[DatasetsDTO, RestError]:
        payload = as_json(payload)
        r = requests.post(self.endpoint + EDPSClient.ORGANISE_FLAT_PATH, data=payload,
                          headers={'Content-type': 'application/json'})
        return EDPSClient.handle_error(r, lambda response: DatasetsDTO.from_dict(json.loads(response.content)))

    def get_jobs_list(self, search=".*", offset=0, limit=50) -> Either[List[ProcessingJob], RestError]:
        r = requests.get(self.endpoint + EDPSClient.JOBS_PATH,
                         params={"search": search, "skip": offset, "limit": limit})
        return EDPSClient.handle_error(r, lambda response: [ProcessingJob.from_dict(x) for x in
                                                            json.loads(response.content)])

    def get_jobs_list_filter(self, search_filter: SearchFilter) -> Either[List[ProcessingJob], RestError]:
        r = requests.get(self.endpoint + EDPSClient.JOBS_FILTER_PATH, params={
            "instrument": search_filter.instrument,
            "targets": search_filter.targets,
            "meta_targets": search_filter.meta_targets,
            "completion_time_from": search_filter.completion_time_from,
            "completion_time_to": search_filter.completion_time_to,
            "mjdobs_from": search_filter.mjdobs_from,
            "mjdobs_to": search_filter.mjdobs_to,
            "submission_time_from": search_filter.submission_time_from,
            "submission_time_to": search_filter.submission_time_to
        })
        return EDPSClient.handle_error(r, lambda response: [ProcessingJob.from_dict(x) for x in
                                                            json.loads(response.content)])

    def get_job_details(self, job_id: str) -> Either[ProcessingJob, RestError]:
        r = requests.get(self.endpoint + EDPSClient.JOB_DETAILS_PATH.format(job_id=job_id))
        return EDPSClient.handle_error(r, lambda response: ProcessingJob.from_dict(json.loads(response.content)))

    def get_job_status(self, job_id: str) -> Either[JobStatus, RestError]:
        r = requests.get(self.endpoint + EDPSClient.JOB_STATUS_PATH.format(job_id=job_id))
        return EDPSClient.handle_error(r, lambda response: JobStatus.from_dict(json.loads(response.content)))

    def get_job_report(self, job_id: str, panel_name: str) -> Either[str, RestError]:
        r = requests.get(self.endpoint + EDPSClient.REPORT_PATH.format(job_id=job_id, report_name=panel_name))
        return EDPSClient.handle_error(r, lambda response: response.text)

    def get_job_log(self, job_id: str, log_name: str) -> Either[str, RestError]:
        r = requests.get(self.endpoint + EDPSClient.LOG_PATH.format(job_id=job_id, log_name=log_name))
        return EDPSClient.handle_error(r, lambda response: response.text)

    def get_workflow_graph(self, workflow_name: str, graph_type: GraphType = GraphType.SIMPLE) -> Either[
        str, RestError]:
        r = requests.get(self.endpoint + EDPSClient.GRAPH_PATH.format(workflow_name=workflow_name),
                         params={"graph_type": graph_type})
        return EDPSClient.handle_error(r, lambda response: response.text)

    def get_assoc_map(self, workflow_name: str) -> Either[str, RestError]:
        r = requests.get(self.endpoint + EDPSClient.ASSOC_MAP_PATH.format(workflow_name=workflow_name))
        return EDPSClient.handle_error(r, lambda response: response.text)

    def list_workflows(self) -> Either[List[str], RestError]:
        r = requests.get(self.endpoint + EDPSClient.WORKFLOWS_PATH)
        return EDPSClient.handle_error(r, lambda response: json.loads(response.content))

    def get_workflow(self, workflow_name: str) -> Either[WorkflowDTO, RestError]:
        r = requests.get(self.endpoint + EDPSClient.WORKFLOW_PATH.format(workflow_name=workflow_name))
        return EDPSClient.handle_error(r, lambda response: WorkflowDTO.from_dict(json.loads(response.content)))

    def get_workflow_dir(self, workflow_name: str) -> Either[str, RestError]:
        r = requests.get(self.endpoint + EDPSClient.WORKFLOW_DIR_PATH.format(workflow_name=workflow_name))
        return EDPSClient.handle_error(r, lambda response: response.text)

    def get_workflow_state(self, workflow_name: str) -> Either[WorkflowStateDTO, RestError]:
        r = requests.get(self.endpoint + EDPSClient.WORKFLOW_STATE_PATH.format(workflow_name=workflow_name))
        return EDPSClient.handle_error(r, lambda response: WorkflowStateDTO.from_dict(json.loads(response.content)))

    def get_workflow_jobs(self, workflow_name: str) -> Either[List[JobInfo], RestError]:
        r = requests.get(self.endpoint + EDPSClient.WORKFLOW_JOBS_PATH.format(workflow_name=workflow_name))
        return EDPSClient.handle_error(r, lambda response: [JobInfo.from_dict(x) for x in json.loads(response.content)])

    def get_workflow_job(self, workflow_name: str, job_id: str) -> Either[JobInfo, RestError]:
        r = requests.get(self.endpoint + EDPSClient.WORKFLOW_JOB_PATH.format(workflow_name=workflow_name,
                                                                             job_id=job_id))
        return EDPSClient.handle_error(r, lambda response: JobInfo.from_dict(json.loads(response.content)))

    def reset_workflow(self, workflow_name: str) -> Either[str, RestError]:
        r = requests.post(self.endpoint + EDPSClient.RESET_PATH.format(workflow_name=workflow_name))
        return EDPSClient.handle_error(r, lambda response: response.text)

    def get_targets(self, workflow_name: str, targets: List[str],
                    meta_targets: List[str]) -> Either[str, RestError]:
        r = requests.get(self.endpoint + EDPSClient.TARGETS_PATH.format(workflow_name=workflow_name),
                         params={"targets": targets, "meta_targets": meta_targets})
        return EDPSClient.handle_error(r, lambda response: response.text)

    def get_parameter_sets(self, workflow_name: str) -> Either[str, RestError]:
        r = requests.get(self.endpoint + EDPSClient.PARAMETER_SETS_PATH.format(workflow_name=workflow_name))
        return EDPSClient.handle_error(r, lambda response: response.text)

    def get_default_params(self, workflow_name: str, task_name: str) -> Either[str, RestError]:
        r = requests.get(
            self.endpoint + EDPSClient.DEFAULT_PARAMS_PATH.format(workflow_name=workflow_name, task_name=task_name))
        return EDPSClient.handle_error(r, lambda response: response.text)

    def get_recipe_params(self, workflow_name: str, task_name: str, parameter_set: str) -> Either[str, RestError]:
        r = requests.get(
            self.endpoint + EDPSClient.RECIPE_PARAMS_PATH.format(workflow_name=workflow_name, task_name=task_name),
            params={"parameter_set": parameter_set})
        return EDPSClient.handle_error(r, lambda response: response.text)

    def update_calib_db(self, workflow_name: str) -> Either[str, RestError]:
        r = requests.post(self.endpoint + EDPSClient.UPDATE_CALIB_PATH.format(workflow_name=workflow_name))
        return EDPSClient.handle_error(r, lambda response: response.text)

    def reject_job(self, job_id: UUID) -> Either[Rejection, RestError]:
        r = requests.post(self.endpoint + EDPSClient.REJECT_JOB_PATH.format(job_id=job_id))
        return EDPSClient.handle_error(r, lambda response: Rejection.from_dict(json.loads(response.content)))

    def delete_job_cascade(self, job_id: UUID) -> Either[List[UUID], RestError]:
        r = requests.post(self.endpoint + EDPSClient.DELETE_JOB_PATH.format(job_id=job_id))
        return EDPSClient.handle_error(r, lambda response: [UUID(job_id) for job_id in json.loads(response.content)])

    def delete_independent_jobs_subset(self, job_ids: List[str]) -> Either[List[UUID], RestError]:
        r = requests.post(self.endpoint + EDPSClient.DELETE_JOBS_PATH, json=job_ids)
        return EDPSClient.handle_error(r, lambda response: [UUID(job_id) for job_id in json.loads(response.content)])

    def get_job_association_report(self, job_id: UUID) -> Either[bytes, RestError]:
        r = requests.get(self.endpoint + EDPSClient.ASSOC_REPORT_PATH.format(job_id=job_id))
        return EDPSClient.handle_error(r, lambda response: response.content)

    def resubmit_job(self, job_id: UUID) -> Either[UUID, RestError]:
        r = requests.post(self.endpoint + EDPSClient.RESUBMIT_JOB_PATH.format(job_id=job_id))
        return EDPSClient.handle_error(r, lambda response: response.content)

    def create_breakpoint(self, job_id: UUID) -> Either[str, RestError]:
        r = requests.post(self.endpoint + EDPSClient.CREATE_BREAKPOINT.format(job_id=job_id),
                          headers={'Content-type': 'application/json'})
        return EDPSClient.handle_error(r, lambda response: response.content)

    def remove_breakpoint(self, job_id: UUID) -> Either[str, RestError]:
        r = requests.delete(self.endpoint + EDPSClient.REMOVE_BREAKPOINT.format(job_id=job_id),
                            headers={'Content-type': 'application/json'})
        return EDPSClient.handle_error(r, lambda response: response.content)

    def loop_execution(self, job_id: UUID, parameters: Dict[str, str]) -> Either[UUID, RestError]:
        payload = as_json(LoopRequestDTO(job_id=str(job_id), parameters=parameters))
        r = requests.post(self.endpoint + EDPSClient.LOOP_EXECUTION, data=payload,
                          headers={'Content-type': 'application/json'})
        return EDPSClient.handle_error(r, lambda response: response.content)

    def loop_execution_once(self, request: LoopOnceRequestDTO) -> Either[str, RestError]:
        payload = as_json(request)
        r = requests.post(self.endpoint + EDPSClient.LOOP_EXECUTION_ONCE, data=payload,
                          headers={'Content-Type': 'application/json'})
        return EDPSClient.handle_error(r, lambda response: response.content)

    def continue_execution(self, job_id: UUID) -> Either[UUID, RestError]:
        r = requests.post(self.endpoint + EDPSClient.CONTINUE_EXECUTION.format(job_id=job_id),
                          headers={'Content-type': 'application/json'})
        return EDPSClient.handle_error(r, lambda response: response.content)

    def inspect_breakpoint(self, job_id: UUID) -> Either[BreakpointInspectResultDTO, RestError]:
        r = requests.get(self.endpoint + EDPSClient.INSPECT_BREAKPOINT.format(job_id=job_id))
        return EDPSClient.handle_error(r, lambda resp: BreakpointInspectResultDTO.from_dict(json.loads(resp.content)))

    def list_breakpoints_state(self) -> Either[BreakpointsStateDTO, RestError]:
        r = requests.get(self.endpoint + EDPSClient.BREAKPOINTS_PATH)
        return EDPSClient.handle_error(r, lambda response: BreakpointsStateDTO.from_dict(json.loads(response.content)))

    def list_pipelines(self) -> Either[List[str], RestError]:
        r = requests.get(self.endpoint + EDPSClient.PIPELINES_PATH)
        return EDPSClient.handle_error(r, lambda response: json.loads(response.content))

    def stop_processing(self) -> Either[str, RestError]:
        r = requests.post(self.endpoint + EDPSClient.STOP_PROCESSING_PATH)
        return EDPSClient.handle_error(r, lambda response: json.loads(response.content))

    def shutdown_edps(self) -> Either[str, RestError]:
        r = requests.post(self.endpoint + EDPSClient.SHUTDOWN_PATH)
        return EDPSClient.handle_error(r, lambda response: json.loads(response.content))

    def get_edps_version(self) -> Either[str, RestError]:
        r = requests.get(self.endpoint + EDPSClient.VERSION_PATH)
        return EDPSClient.handle_error(r, lambda response: json.loads(response.content))

    def run_reports(self, request: RunReportsRequestDTO) -> Either[List[str], RestError]:
        r = requests.post(self.endpoint + EDPSClient.REPORT_EXECUTION_PATH,
                          data=as_json(request),
                          headers={'Content-type': 'application/json'})
        return EDPSClient.handle_error(r, lambda response: json.loads(response.content))

    def package_phase3(self, request: Phase3Configuration) -> Either[str, RestError]:
        r = requests.post(self.endpoint + EDPSClient.PHASE3_PACKAGE_PATH,
                          data=as_json(request),
                          headers={'Content-type': 'application/json'})
        return EDPSClient.handle_error(r, lambda response: response.text)

    def get_scheduled_jobs(self) -> Either[List[UUID], RestError]:
        r = requests.get(self.endpoint + EDPSClient.SCHEDULED_JOBS_PATH)
        return EDPSClient.handle_error(r, lambda response: [UUID(job_id) for job_id in json.loads(response.content)])

    @staticmethod
    def handle_error(r: requests.Response, f: Callable[[requests.Response], T]) -> Either[T, RestError]:
        if r.status_code // 100 == 2:
            return Either.right(f(r))
        else:
            return Either.left(RestError(r.status_code, r.text))
