import re
from typing import Dict, Optional, Any

from pydantic import BaseModel


class FitsFile(BaseModel):
    name: Optional[str] = None
    category: Optional[str] = None
    keywords: Dict[str, Optional[Any]] = {}

    def match(self, pattern):
        return re.match(pattern, self.name) or re.match(pattern, self.category)

    def __eq__(self, o: object) -> bool:
        return isinstance(o, type(self)) and o.name == self.name and o.category == self.category

    def __hash__(self):
        return hash((self.name, self.category))

    def __repr__(self):
        return self.model_dump_json()

    def __str__(self):
        return f'{self.name} {self.category}'

    class Config:
        json_schema_extra = {
            "example": {
                "name": "/opt/calibfiles/calib.fits",
                "category": "BIAS",
                "keywords": {"PRODCATG": "BIAS"}
            }
        }

    @staticmethod
    def from_dict(d):
        return FitsFile(name=d['name'], category=d['category'], keywords=d.get('keywords', {}))
