from __future__ import annotations

import os
from typing import List, Dict, Optional

from pydantic import BaseModel

from edps.client.AssociationResult import AssociationResultDTO
from edps.client.FitsFile import FitsFile
from edps.client.JobInfo import JobInfo


class DatasetDTO(BaseModel):
    task_name: str
    job_id: str
    main_input_files: List[FitsFile]
    main_input_jobs: List['InputJob']
    associated_input_files: List[FitsFile]
    associated_input_jobs: List['InputJob']
    missing_associated_inputs: List[AssociationResultDTO]
    complete: bool
    assoc_level: Optional[float] = None

    @staticmethod
    def from_job_info(job: JobInfo, jobs_by_id: Dict[str, JobInfo]) -> 'DatasetDTO':
        associated_job_ids = [job_id for job_id in job.associated_job_ids if job_id in jobs_by_id]
        return DatasetDTO(
            task_name=job.task_name,
            job_id=job.job_id,
            main_input_files=job.input_files,
            main_input_jobs=[InputJob.from_job_info(jobs_by_id[job_id], jobs_by_id)
                             for job_id in job.input_job_ids if job_id in jobs_by_id],
            associated_input_files=job.associated_files,
            associated_input_jobs=[InputJob.from_job_info(jobs_by_id[job_id], jobs_by_id)
                                   for job_id in associated_job_ids],
            missing_associated_inputs=[association for association in job.association_details if
                                  not association.complete and
                                  not set(associated_job_ids).intersection([associated_job.name for associated_job in association.jobs])],
            complete=job.complete,
            assoc_level=job.assoc_level
        )

    @staticmethod
    def from_dict(job_dict: Dict[str, object]) -> DatasetDTO:
        return DatasetDTO(
            task_name=job_dict['task_name'],
            job_id=job_dict['job_id'],
            complete=job_dict['complete'],
            assoc_level=job_dict.get('assoc_level', None),
            main_input_files=[FitsFile.from_dict(file) for file in job_dict.get('main_input_files', [])],
            main_input_jobs=[InputJob.from_dict(job) for job in job_dict.get('main_input_jobs', [])],
            associated_input_files=[FitsFile.from_dict(file) for file in job_dict.get('associated_input_files', [])],
            associated_input_jobs=[InputJob.from_dict(job) for job in job_dict.get('associated_input_jobs', [])],
            missing_associated_inputs=[AssociationResultDTO.from_dict(missing_association)
                                       for missing_association in job_dict.get('missing_associated_inputs', [])]
        )

    class Config:
        json_schema_extra = {
            "example": {
                "task_name": "flat",
                "job_id": "5c04c6ce-5dd0-4a32-a9b8-41b55f66319c",
                "complete": True,
                "assoc_level": 0,
                "main_input_files": [FitsFile.Config.json_schema_extra['example']],
                "main_input_jobs": [{
                    "task_name": "flat",
                    "dataset": {
                        "task_name": "flat",
                        "job_id": "5c04c6ce-5dd0-4a32-a9b8-41b55f66319c",
                        "main_input_files": [FitsFile.Config.json_schema_extra['example']],
                        "main_input_jobs": [],
                        "associated_input_files": [FitsFile.Config.json_schema_extra['example']],
                        "associated_input_jobs": [],
                    }
                }],
                "associated_input_files": [FitsFile.Config.json_schema_extra['example']],
                "associated_input_jobs": [{
                    "task_name": "flat",
                    "dataset": {
                        "task_name": "flat",
                        "job_id": "5c04c6ce-5dd0-4a32-a9b8-41b55f66319c",
                        "main_input_files": [FitsFile.Config.json_schema_extra['example']],
                        "main_input_jobs": [],
                        "associated_input_files": [FitsFile.Config.json_schema_extra['example']],
                        "associated_input_jobs": [],
                    }}],
                "missing_associated_inputs": [AssociationResultDTO.Config.json_schema_extra['example']]
            },
        }


class InputJob(BaseModel):
    task_name: Optional[str] = None
    dataset: Optional[DatasetDTO] = None

    @staticmethod
    def from_job_info(job: JobInfo, jobs_by_id: Dict[str, JobInfo]) -> 'InputJob':
        return InputJob(
            task_name=job.task_name,
            dataset=DatasetDTO.from_job_info(job, jobs_by_id)
        )

    @staticmethod
    def from_dict(job_dict: Dict[str, Dict]) -> InputJob:
        return InputJob(
            task_name=job_dict['task_name'],
            dataset=DatasetDTO.from_dict(job_dict['dataset']),
        )

    class Config:
        json_schema_extra = {
            "example": {
                "task_name": 'flat',
                "dataset": DatasetDTO.Config.json_schema_extra['example']
            }
        }


class LabelledDatasetDTO(BaseModel):
    dataset_name: Optional[str] = None
    dataset: Optional[DatasetDTO] = None

    @staticmethod
    def extract_dataset_name(job: JobInfo, jobs_by_id: Dict[str, JobInfo]) -> str:
        if len(job.input_files) > 0:
            return os.path.splitext(os.path.basename(job.input_files[0].name))[0]
        elif len(job.input_job_ids) > 0 and job.input_job_ids[0] in jobs_by_id:
            return LabelledDatasetDTO.extract_dataset_name(jobs_by_id[job.input_job_ids[0]], jobs_by_id)
        else:
            return job.task_name + "#MAIN_INPUTS_NOT_FOUND"

    @staticmethod
    def from_job_info(job: JobInfo, jobs_by_id: Dict[str, JobInfo]) -> 'LabelledDatasetDTO':
        return LabelledDatasetDTO(
            dataset_name=LabelledDatasetDTO.extract_dataset_name(job, jobs_by_id),
            dataset=DatasetDTO.from_job_info(job, jobs_by_id)
        )

    @classmethod
    def from_dict(cls, job_dict: Dict[str, Dict]) -> LabelledDatasetDTO:
        return LabelledDatasetDTO(
            dataset_name=job_dict['dataset_name'],
            dataset=DatasetDTO.from_dict(job_dict['dataset'])
        )

    class Config:
        json_schema_extra = {
            "example": {
                "dataset_name": FitsFile.Config.json_schema_extra['example'],
                "dataset": DatasetDTO.Config.json_schema_extra['example']
            }
        }


DatasetDTO.update_forward_refs()
LabelledDatasetDTO.update_forward_refs()


class DatasetsDTO(BaseModel):
    datasets: Optional[List[LabelledDatasetDTO]] = None

    class Config:
        json_schema_extra = {
            "example": {
                "datasets": [LabelledDatasetDTO.Config.json_schema_extra['example']],
            }
        }

    @classmethod
    def from_dict(cls, d: Dict[str, object]) -> DatasetsDTO:
        return DatasetsDTO(datasets=[LabelledDatasetDTO.from_dict(job_dict) for job_dict in d.get("datasets", [])])
