from typing import List, Dict, Optional

from pydantic import BaseModel

from edps.client.AssociationResult import AssociationResultDTO
from edps.client.FitsFile import FitsFile
from edps.client.JobParametersDTO import JobParametersDTO
from edps.client.JobSummary import JobSummary, Setup, Header


class ReportConfigDTO(BaseModel):
    name: str
    input: str
    driver: str

    @classmethod
    def from_dict(cls, d: Dict):
        return cls(
            name=d['name'],
            input=d['input'],
            driver=d['driver']
        )

    class Config:
        json_schema_extra = {
            "example": {
                "name": "report_name",
                "input": "RECIPE_INPUTS",
                "driver": "png"
            }
        }


class JobInfo(BaseModel):
    job_id: Optional[str] = None
    input_files: List[FitsFile] = []
    associated_files: List[FitsFile] = []
    input_job_ids: List[str] = []
    associated_job_ids: List[str] = []
    command_type: Optional[str] = None
    command: Optional[str] = None
    parameters: Optional[JobParametersDTO] = None
    instrument: Optional[str] = None
    mjdobs: Optional[float] = None
    task_name: Optional[str] = None
    assoc_level: Optional[float] = None
    task_id: Optional[str] = None
    association_details: List[AssociationResultDTO] = []
    workflow_names: List[str] = []
    complete: bool = True
    setup: Optional[Setup] = None
    header: Optional[Header] = None
    reports: List[ReportConfigDTO] = []
    input_filter: List[str] = []
    input_filter_mode: str = ""
    output_filter: List[str] = []
    output_filter_mode: str = ""
    input_map: Dict[str, str] = {}
    future: bool = False
    meta_targets: List[str] = []

    def __hash__(self):
        return hash(self.job_id)

    def is_future(self):
        return self.future or self.command_type == 'future'

    @property
    def parents_ids(self) -> List[str]:
        return self.input_job_ids + self.associated_job_ids

    def exemplar_as_str(self) -> str:
        if self.input_files:
            return f'{self.input_files[0]} {self.mjdobs}'
        else:
            return f'{self.input_job_ids[0]} {self.mjdobs}'

    def to_summary_dto(self) -> JobSummary:
        return JobSummary(
            job_id=self.job_id,
            input_files=self.input_files,
            input_job_ids=self.input_job_ids,
            instrument=self.instrument,
            mjdobs=self.mjdobs,
            task_name=self.task_name,
            assoc_level=self.assoc_level,
            task_id=self.task_id,
            association_details=self.association_details,
            setup=self.setup,
            header=self.header,
            workflow_names=self.workflow_names
        )

    @classmethod
    def from_dict(cls, d: Dict) -> 'JobInfo':
        return cls(
            job_id=d['job_id'],
            input_files=[FitsFile.from_dict(x) for x in d.get('input_files', [])],
            associated_files=[FitsFile.from_dict(x) for x in d.get('associated_files', [])],
            input_job_ids=d.get('input_job_ids', []),
            associated_job_ids=d.get('associated_job_ids', []),
            command_type=d['command_type'],
            command=d['command'],
            parameters=JobParametersDTO.from_dict(d.get('parameters', {})),
            instrument=d['instrument'],
            mjdobs=d['mjdobs'],
            task_name=d['task_name'],
            assoc_level=d['assoc_level'],
            task_id=d['task_id'],
            association_details=[AssociationResultDTO.from_dict(x) for x in d.get('association_details', [])],
            workflow_names=d.get('workflow_names', []),
            complete=d.get('complete', True),
            setup=d.get('setup', {}),
            header=d.get('header', {}),
            reports=[ReportConfigDTO.from_dict(x) for x in d.get('reports', [])],
            input_filter=d.get('input_filter', []),
            input_filter_mode=d.get('input_filter_mode', ""),
            output_filter=d.get('output_filter', []),
            output_filter_mode=d.get('output_filter_mode', ""),
            input_map=d.get('input_map', {}),
            future=d['command_type'] == 'future' or d.get('future', False),
            meta_targets=d.get('meta_targets', [])
        )

    class Config:
        json_schema_extra = {
            "example": {
                "job_id": "5c04c6ce-5dd0-4a32-a9b8-41b55f66319c",
                "input_files": [FitsFile.Config.json_schema_extra['example']],
                "associated_files": [FitsFile.Config.json_schema_extra['example']],
                "input_job_ids": ["6c04c6ce-5dd0-4a32-a9b8-41b55f66319c"],
                "associated_job_ids": ["7c04c6ce-5dd0-4a32-a9b8-41b55f66319c"],
                "command_type": "recipe",
                "command": "fors_bias",
                "parameters": JobParametersDTO.Config.json_schema_extra['example'],
                "instrument": "FORS1",
                "mjdobs": 508346.93563551357,
                "task_name": "bias",
                "assoc_level": 0,
                "task_id": "bias.fors_bias",
                "association_details": [AssociationResultDTO.Config.json_schema_extra['example']],
                "workflow_names": ["edps.workflow.fors_imaging_wkf", "edps.workflow.kmos"],
                "complete": True,
                "setup": {"det.binx": "1", "det.biny": "1"},
                "header": {"tpl.start": "xxx", "tpl.id": "yyy", "tpl.name": "zzz"},
                "reports": [ReportConfigDTO.Config.json_schema_extra['example']],
                "input_filter": ["CATEGORY1", "CATEGORY2"],
                "input_filter_mode": "SELECT",
                "output_filter": ["CATEGORY1", "CATEGORY2"],
                "output_filter_mode": "SELECT",
                "input_map": {"REAL_CATEGORY": "OVERRIDE_CATEGORY"},
                "future": False,
                "meta_targets": ["QC0"]
            }
        }
