from typing import List, Dict, Optional

from pydantic import BaseModel

from edps.client.AssociationResult import AssociationResultDTO
from edps.client.FitsFile import FitsFile

Setup = Dict[str, str]
Header = Dict[str, str]


class JobSummary(BaseModel):
    job_id: Optional[str] = None
    input_files: Optional[List[FitsFile]] = None
    input_job_ids: Optional[List[str]] = None
    instrument: Optional[str] = None
    mjdobs: Optional[float] = None
    task_name: Optional[str] = None
    assoc_level: Optional[float] = None
    task_id: Optional[str] = None
    association_details: Optional[List[AssociationResultDTO]] = None
    setup: Optional[Setup] = None
    header: Optional[Header] = None
    workflow_names: Optional[List[str]] = None

    @classmethod
    def from_dict(cls, d: Dict) -> 'JobSummary':
        return cls(
            job_id=d['job_id'],
            input_files=[FitsFile.from_dict(x) for x in d.get('input_files', [])],
            input_job_ids=d.get('input_job_ids', []),
            instrument=d['instrument'],
            mjdobs=d['mjdobs'],
            task_name=d['task_name'],
            assoc_level=d['assoc_level'],
            task_id=d['task_id'],
            association_details=[AssociationResultDTO.from_dict(x) for x in d.get('association_details', [])],
            setup=d.get('setup', {}),
            header=d.get('header', {}),
            workflow_names=d.get('workflow_names', []),
        )

    class Config:
        json_schema_extra = {
            "example": {
                "job_id": "5c04c6ce-5dd0-4a32-a9b8-41b55f66319c",
                "input_files": [FitsFile.Config.json_schema_extra['example']],
                "input_job_ids": ["6c04c6ce-5dd0-4a32-a9b8-41b55f66319c"],
                "instrument": "FORS1",
                "mjdobs": 508346.93563551357,
                "task_name": "bias",
                "assoc_level": 0,
                "task_id": "bias.fors_bias",
                "association_details": [AssociationResultDTO.Config.json_schema_extra['example']],
                "setup": {"det.binx": "1", "det.biny": "1"},
                "header": {"tpl.start": "xxx", "tpl.id": "yyy", "tpl.name": "zzz"},
                "workflow_names": ["edps.workflow.fors_imaging_wkf", "edps.workflow.kmos"],
            }
        }
