from datetime import datetime
from typing import List, Dict, Optional, Any

from pydantic import BaseModel

from edps.client.FitsFile import FitsFile
from edps.client.JobInfo import JobInfo
from edps.client.ProcessingJobStatus import ProcessingJobStatus


class LogEntry(BaseModel):
    file_name: Optional[str] = None

    class Config:
        json_schema_extra = {
            "example": {
                "file_name": "esorex.log",
            }
        }

    @staticmethod
    def from_dict(d):
        return LogEntry(file_name=d['file_name'])


class ReportEntryPanel(BaseModel):
    file_name: Optional[str] = None
    media_type: Optional[str] = None
    description: Optional[str] = None
    prodcatg: Optional[str] = None
    input_files: Optional[List[str]] = None

    class Config:
        json_schema_extra = {
            "example": {
                "file_name": "some_report.png",
                "media_type": "image/png",
                "description": "some report",
                "prodcatg": "ANCILLARY.PREVIEW",
                "input_files": ["/a/b/c/bias.fits", "/d/e/f/flat.fits"]
            }
        }

    @staticmethod
    def from_dict(d) -> 'ReportEntryPanel':
        return ReportEntryPanel(file_name=d.get('file_name', None), media_type=d.get('media_type', None),
                                description=d.get('description', None), prodcatg=d.get('prodcatg', None),
                                input_files=d.get('input_files', None))


class ReportEntry(BaseModel):
    report_name: Optional[str] = None
    return_code: int = 0
    panels: List[ReportEntryPanel] = []

    class Config:
        json_schema_extra = {
            "example": {
                "report_name": "some_report",
                "return_code": 0,
                "panels": [ReportEntryPanel.Config.json_schema_extra['example']]
            }
        }

    @staticmethod
    def from_dict(d) -> 'ReportEntry':
        return ReportEntry(report_name=d.get('report_name', None), return_code=d.get('return_code', None),
                           panels=[ReportEntryPanel.from_dict(entry) for entry in d.get('panels', [])])


class ProcessingJob(BaseModel):
    completion_date: Optional[str] = None
    submission_date: Optional[str] = None
    status: Optional[ProcessingJobStatus] = None
    input_files: List[FitsFile] = []
    output_files: List[FitsFile] = []
    logs: List[LogEntry] = []
    reports: List[ReportEntry] = []
    recipe_parameters: Dict[str, Any] = {}
    configuration: Optional[JobInfo] = None
    rejected: bool = False
    interrupted: bool = False

    @staticmethod
    def from_dict(d) -> 'ProcessingJob':
        return ProcessingJob(
            completion_date=d['completion_date'],
            submission_date=d['submission_date'],
            status=ProcessingJobStatus(d['status']),
            input_files=[FitsFile.from_dict(file) for file in d.get('input_files', [])],
            output_files=[FitsFile.from_dict(file) for file in d.get('output_files', [])],
            logs=[LogEntry.from_dict(log) for log in d.get('logs', [])],
            reports=[ReportEntry.from_dict(report) for report in d.get('reports', [])],
            recipe_parameters=d.get('recipe_parameters', {}),
            configuration=JobInfo.from_dict(d['configuration']),
            rejected=d.get('rejected', False),
            interrupted=d.get('interrupted', False),
        )

    def __repr__(self):
        return self.model_dump_json()

    class Config:
        json_schema_extra = {
            "example": {
                "completion_date": datetime.now().isoformat(),
                "submission_date": datetime.now().isoformat(),
                "status": ProcessingJobStatus.COMPLETED,
                "input_files": [FitsFile.Config.json_schema_extra['example']],
                "output_files": [FitsFile.Config.json_schema_extra['example']],
                "reports": [ReportEntry.Config.json_schema_extra['example']],
                "logs": [LogEntry.Config.json_schema_extra['example']],
                "recipe_parameters": {"a": "b"},
                "configuration": JobInfo.Config.json_schema_extra['example'],
                "rejected": False,
                "interrupted": False
            }
        }
