import uuid
from typing import List

from pydantic import BaseModel

from edps.interfaces.JobsRepository import RejectionResult


class Rejection(BaseModel):
    rejected_job_ids: List[str] = []
    incomplete_job_ids: List[str] = []
    files_to_resubmit: List[str] = []

    @staticmethod
    def from_rejection_result(r: RejectionResult) -> 'Rejection':
        return Rejection(
            rejected_job_ids=[str(job_id) for job_id in r.rejected_jobs],
            incomplete_job_ids=[str(job_id) for job_id in r.incomplete_jobs],
            files_to_resubmit=r.files_to_resubmit
        )

    @staticmethod
    def from_dict(d) -> 'Rejection':
        return Rejection(
            rejected_job_ids=d.get('rejected_job_ids', []),
            incomplete_job_ids=d.get('incomplete_job_ids', []),
            files_to_resubmit=d.get('files_to_resubmit', []),
        )

    def __repr__(self):
        return self.model_dump_json()

    class Config:
        json_schema_extra = {
            "example": {
                "rejected_job_ids": [str(uuid.uuid4())],
                "incomplete_job_ids": [str(uuid.uuid4())],
                "files_to_resubmit": ["/some/path/FILE.fits", "/some/path2/FILE2.fits"]
            }
        }
