from enum import Enum
from typing import List, Optional, Dict, Union

from pydantic import BaseModel

from edps.client.FitsFile import FitsFile

ClassificationDict = Dict[str, Union[List[object], object]]


class RelativeTimeRangeDTO(BaseModel):
    left: Optional[float]
    right: Optional[float]

    @classmethod
    def from_dict(cls, d):
        return cls(left=d['left'], right=d['right'])

    class Config:
        json_schema_extra = {
            "example": {
                "left": -0.5,
                "right": 0.5,
            }
        }


class AssociationConfigurationDTO(BaseModel):
    level: float
    time_range: RelativeTimeRangeDTO
    match_function: Optional[str]
    match_keywords: List[str]

    @classmethod
    def from_dict(cls, d):
        return cls(level=d['level'],
                   time_range=RelativeTimeRangeDTO.from_dict(d['time_range']),
                   match_function=d['match_function'],
                   match_keywords=d['match_keywords'])

    class Config:
        json_schema_extra = {
            "example": {
                "level": 0,
                "time_range": RelativeTimeRangeDTO.Config.json_schema_extra['example'],
                "match_function": "match(['det.binx', 'det.biny', 'ins.mode'])",
                "match_keywords": ["det.binx", "det.biny", "ins.mode"]
            }
        }


class ClassificationRuleDTO(BaseModel):
    category: str
    function: Optional[str]
    keyword_values: Optional[ClassificationDict]

    @classmethod
    def from_dict(cls, d):
        return cls(category=d['category'], function=d['function'], keyword_values=d['keyword_values'])

    class Config:
        json_schema_extra = {
            "example": {
                "category": "DARK",
                "function": None,
                "keyword_values": {"dpr.catg": "CALIB", "dpr.tech": "IMAGE", "dpr.type": "DARK"}
            }
        }


class TaskType(str, Enum):
    BASE = "base"
    TASK = "task"
    DATA_SOURCE = "datasource"


class AssociatedInputDTO(BaseModel):
    input_id: str
    input_name: str
    input_type: TaskType
    input_categories: List[str]
    assoc_levels: List[AssociationConfigurationDTO]
    prod_categories: List[str]
    min_ret: int
    max_ret: int
    order_by: List[str]
    classification_rules: List[ClassificationRuleDTO]

    @classmethod
    def from_dict(cls, d):
        return cls(input_id=d['input_id'], input_name=d['input_name'], input_type=d['input_type'],
                   input_categories=d['input_categories'],
                   assoc_levels=[AssociationConfigurationDTO.from_dict(x) for x in d['assoc_levels']],
                   prod_categories=d['prod_categories'], min_ret=d['min_ret'], max_ret=d['max_ret'],
                   order_by=d['order_by'],
                   classification_rules=[ClassificationRuleDTO.from_dict(x) for x in d.get('classification_rules',[])])

    class Config:
        json_schema_extra = {
            "example": {
                "input_id": "73f57e2d-0415-4439-bcb3-40c675551aaa",
                "input_name": "bias",
                "input_type": TaskType.TASK,
                "input_categories": ["MASTER_BIAS_RES"],
                "assoc_levels": [AssociationConfigurationDTO.Config.json_schema_extra['example']],
                "prod_categories": ["MASTER_BIAS_RES"],
                "min_ret": 1,
                "max_ret": 1,
                "order_by": ["mjd-obs"],
                "classification_rules": [ClassificationRuleDTO.Config.json_schema_extra['example']]
            }
        }


class AssociatedInputGroupDTO(BaseModel):
    associated_inputs: List[AssociatedInputDTO]
    order_by: List[str]

    @classmethod
    def from_dict(cls, d):
        return cls(associated_inputs=[AssociatedInputDTO.from_dict(x) for x in d['associated_inputs']],
                   order_by=d['order_by'])

    class Config:
        json_schema_extra = {
            "example": {
                "associated_inputs": [AssociatedInputDTO.Config.json_schema_extra['example']],
                "order_by": ["mjd-obs"]
            }
        }


class TaskBaseDTO(BaseModel):
    id: str
    name: str
    categories: List[str]
    assoc_levels: List[AssociationConfigurationDTO]
    prod_categories: List[str]
    group_by: List[str]
    min_group_size: int
    setup: List[str]

    @classmethod
    def from_dict(cls, d):
        return cls(id=d['id'], name=d['name'], categories=d['categories'], setup=d['setup'],
                   assoc_levels=[AssociationConfigurationDTO.from_dict(x) for x in d['assoc_levels']],
                   prod_categories=d['prod_categories'], group_by=d['group_by'], min_group_size=d['min_group_size'])


class DataSourceDTO(TaskBaseDTO):

    @classmethod
    def from_dict(cls, d):
        base = TaskBaseDTO.from_dict(d)
        return cls(id=base.id, name=base.name, categories=base.categories, setup=base.setup,
                   assoc_levels=base.assoc_levels, prod_categories=base.prod_categories,
                   group_by=base.group_by, min_group_size=base.min_group_size)

    class Config:
        json_schema_extra = {
            "example": {
                "id": "7ed5e574-560e-4aad-a7eb-caad499bc970",
                "name": "FLAT",
                "categories": ["FLAT_BLUE", "FLAT_RED"],
                "assoc_levels": [AssociationConfigurationDTO.Config.json_schema_extra['example']],
                "prod_categories": [],
                "group_by": ["tpl.start"],
                "min_group_size": 1,
                "setup": ["det.binx", "det.biny"]
            }
        }


class TaskDTO(TaskBaseDTO):
    main_input_id: str
    main_input_name: str
    associated_inputs: List[AssociatedInputDTO]
    associated_input_groups: List[AssociatedInputGroupDTO]
    task_id: str
    command: str
    command_type: str
    meta_targets: List[str]
    workflows: List[str]
    dynamic_parameters: List[str]

    @classmethod
    def from_dict(cls, d):
        base = TaskBaseDTO.from_dict(d)
        return cls(id=base.id, name=base.name, categories=base.categories, setup=base.setup,
                   assoc_levels=base.assoc_levels, prod_categories=base.prod_categories,
                   group_by=base.group_by, min_group_size=base.min_group_size,
                   main_input_id=d['main_input_id'], main_input_name=d['main_input_name'],
                   associated_inputs=[AssociatedInputDTO.from_dict(x) for x in d['associated_inputs']],
                   associated_input_groups=[AssociatedInputGroupDTO.from_dict(x) for x in d['associated_input_groups']],
                   task_id=d['task_id'], command=d['command'], command_type=d['command_type'],
                   meta_targets=d['meta_targets'], workflows=d['workflows'], dynamic_parameters=d['dynamic_parameters'])

    class Config:
        json_schema_extra = {
            "example": {
                "id": "76b23cc7-e1eb-494f-af60-b638731bde45",
                "name": "flat",
                "categories": [],
                "assoc_levels": [AssociationConfigurationDTO.Config.json_schema_extra['example']],
                "prod_categories": ["MASTER_FLAT_REDU", "MASTER_FLAT_REDL"],
                "group_by": [],
                "setup": ["det.binx", "det.biny"],
                "main_input_id": "7ed5e574-560e-4aad-a7eb-caad499bc970",
                "main_input_name": "FLAT",
                "associated_inputs": [AssociatedInputDTO.Config.json_schema_extra['example']],
                "task_id": "flat.uves_cal_mflat",
                "command": "uves_cal_mflat",
                "command_type": "recipe",
                "meta_targets": ["qc1calib", "all"],
                "workflows": ["edps.workflow.uves_wkf"],
                "dynamic_parameters": ["obs_type"]
            }
        }


class WorkflowDTO(BaseModel):
    keywords: List[str]
    classification_rules: List[ClassificationRuleDTO]
    data_sources: List[DataSourceDTO]
    tasks: List[TaskDTO]

    @classmethod
    def from_dict(cls, d):
        return cls(keywords=d['keywords'],
                   classification_rules=[ClassificationRuleDTO.from_dict(x) for x in d['classification_rules']],
                   data_sources=[DataSourceDTO.from_dict(x) for x in d['data_sources']],
                   tasks=[TaskDTO.from_dict(x) for x in d['tasks']])

    class Config:
        json_schema_extra = {
            "example": {
                "keywords": ['instrume', 'arcfile', 'mjd-obs', 'dpr.catg', 'pro.catg'],
                "classification_rules": [ClassificationRuleDTO.Config.json_schema_extra['example']],
                "data_sources": [DataSourceDTO.Config.json_schema_extra['example']],
                "tasks": [TaskDTO.Config.json_schema_extra['example']]
            }
        }


class CalselectorJob(BaseModel):
    task_name: Optional[str] = None
    input_files: List[FitsFile]
    exemplar: FitsFile
    associated_input_groups: List[AssociatedInputGroupDTO]

    @classmethod
    def from_dict(cls, d):
        return cls(task_name=d['task_name'], input_files=[FitsFile.from_dict(x) for x in d['input_files']],
                   exemplar=FitsFile.from_dict(d['exemplar']),
                   associated_input_groups=[AssociatedInputGroupDTO.from_dict(x) for x in d['associated_input_groups']])

    class Config:
        json_schema_extra = {
            "example": {
                "task_name": "object",
                "input_files": [FitsFile.Config.json_schema_extra['example']],
                "exemplar": FitsFile.Config.json_schema_extra['example'],
                "associated_input_groups": [AssociatedInputGroupDTO.Config.json_schema_extra['example']]
            }
        }
