import datetime
import itertools
import logging
import os
from time import sleep
from typing import List, Dict
from uuid import UUID

from .command import Command, File, RecipeInputs
from .constants import ESOREX_TEMPLOG, README_RE_RUN
from .filtering import ProductFilter
from ..client.FitsFile import FitsFile
from ..client.ProcessingJob import LogEntry
from ..client.ProcessingJobStatus import ProcessingJobStatus
from ..interfaces.JobsRepository import JobsRepository, ExecutionResult


class Action(object):
    def __init__(self, name: UUID, input_files: List[File], associated_files: List[File], base_dir: str):
        self.name = name
        self.input_files = input_files
        self.associated_files = associated_files
        self.base_dir = base_dir
        self.logger = logging.getLogger('Action')

    def get_label(self) -> str:
        raise NotImplementedError

    def get_job_dir(self) -> str:
        return os.path.abspath(os.path.join(self.base_dir, str(self.name)))

    def execute(self, main_upstream_outputs: List[ExecutionResult], associated_upstream_outputs: List[ExecutionResult],
                recipe_parameters: Dict) -> ExecutionResult:
        raise NotImplementedError

    def get_log_prefix(self) -> str:
        return f"[{self.get_label()} {self.name}]"

    def is_accepted(self, f: FitsFile) -> bool:
        raise NotImplementedError

    def combine_inputs(self, main_upstream_results: List[ExecutionResult],
                       associated_upstream_results: List[ExecutionResult]) -> RecipeInputs:
        main_upstream_outputs = [[f for f in res.output_files if self.is_accepted(f)] for res in main_upstream_results]
        associated_upstream_outputs = [[f for f in res.output_files if self.is_accepted(f)] for res in associated_upstream_results]
        self.logger.debug("%s Filtered upstream outputs: main %s associated %s", self.get_log_prefix(),
                          main_upstream_outputs, associated_upstream_outputs)
        main_upstream_files = [File(f.name, f.category, "main_upstream_file") for f in itertools.chain.from_iterable(main_upstream_outputs)]
        associated_upstream_files = [File(f.name, f.category, "associated_upstream_file") for f in itertools.chain.from_iterable(associated_upstream_outputs)]
        self.logger.debug("%s Recipe inputs %s %s and upstream inputs %s %s", self.get_log_prefix(),
                          self.input_files, self.associated_files, main_upstream_files, associated_upstream_files)
        return RecipeInputs(main_inputs=self.input_files, associated_inputs=self.associated_files,
                            main_upstream_inputs=main_upstream_files,
                            associated_upstream_inputs=associated_upstream_files)


class FileProcessingAction(Action):
    def __init__(self, input_files: List[File], associated_files: List[File], repository: JobsRepository,
                 input_filter: ProductFilter, command: Command, reexecution_window: datetime.timedelta = datetime.timedelta(hours=24)):
        super().__init__(command.name, input_files, associated_files, command.base_dir)
        self.input_filter = input_filter
        self.repository = repository
        self.command = command
        self.reexecution_window = reexecution_window

    def get_label(self) -> str:
        return self.command.get_label()

    def run_command(self, recipe_inputs: RecipeInputs, parameters: Dict) -> ExecutionResult:
        return self.command.run_command(recipe_inputs, parameters)

    def is_accepted(self, f: FitsFile) -> bool:
        return self.input_filter.is_accepted(f)

    def execute(self, main_upstream_outputs: List[ExecutionResult], associated_upstream_outputs: List[ExecutionResult],
                recipe_parameters: Dict) -> ExecutionResult:
        recipe_inputs = self.combine_inputs(main_upstream_outputs, associated_upstream_outputs)
        equivalent_job_id = self.repository.find_complete_job(self.get_label(), recipe_inputs.sorted, recipe_parameters, self.input_filter, self.command.output_filter)
        if equivalent_job_id:
            self.logger.debug("%s Found equivalent job %s", self.get_log_prefix(), equivalent_job_id)
            result = self.handle_existing_job(equivalent_job_id, recipe_inputs, recipe_parameters)
        else:
            self.logger.debug("%s No equivalent job found, proceeding with regular execution", self.get_log_prefix())
            result = self.execute_processing(recipe_inputs, recipe_parameters)
        return result

    def handle_existing_job(self, job_id: UUID, recipe_inputs: RecipeInputs, recipe_parameters: Dict) -> ExecutionResult:
        while True:
            details = self.repository.get_job_details(job_id)
            if details.needs_re_execution(self.reexecution_window):
                self.logger.debug("%s Future or equivalent job status is %s, re-triggering",
                                  self.get_log_prefix(), details.result.status)
                return self.execute_processing(recipe_inputs, recipe_parameters)
            elif details.is_complete() or details.should_be_skipped(self.reexecution_window):
                self.logger.debug("%s Future or equivalent job execution doesn't need re-execution, re-using outputs",
                                  self.get_log_prefix())
                if self.name != job_id:
                    dirname = self.command.get_base_dir()
                    os.makedirs(dirname, exist_ok=True)
                    with open(os.path.join(dirname, README_RE_RUN), 'wb') as log_file:
                        msg = f"This directory is artifact of smart re-run, using equivalent job '{job_id}'".encode()
                        log_file.write(msg)
                    return ExecutionResult(
                        status=details.result.status,
                        input_files=details.result.input_files,
                        output_files=details.result.output_files,
                        logs=details.result.logs,
                        reports=details.result.reports,
                        completion_date=datetime.datetime.now().isoformat()
                    )
                else:
                    return details.result
            else:
                self.logger.debug("%s Waiting for future job execution to complete", self.get_log_prefix())
                sleep(5)

    def execute_processing(self, recipe_inputs: RecipeInputs, recipe_parameters: Dict) -> ExecutionResult:
        try:
            dirname = self.command.get_base_dir()
            self.repository.set_job_status(self.name, ExecutionResult(status=ProcessingJobStatus.RUNNING,
                                                                      input_files=recipe_inputs.combined,
                                                                      recipe_parameters=recipe_parameters,
                                                                      logs=[LogEntry(file_name=ESOREX_TEMPLOG)]))
            os.makedirs(dirname, exist_ok=True)
            result = self.run_command(recipe_inputs, recipe_parameters)
            return result
        except KeyboardInterrupt:
            self.logger.info("%s Command processing was interrupted by shutdown signal", self.get_log_prefix())
            return ExecutionResult(status=ProcessingJobStatus.FAILED, completion_date=datetime.datetime.now().isoformat(),
                                   input_files=recipe_inputs.sorted, recipe_parameters=recipe_parameters, interrupted=True)
        except Exception as e:
            self.logger.exception("%s Command processing crashed %s", self.get_log_prefix(), e)
            return ExecutionResult(status=ProcessingJobStatus.FAILED, completion_date=datetime.datetime.now().isoformat(),
                                   input_files=recipe_inputs.sorted, recipe_parameters=recipe_parameters)


class FutureAction(Action):
    def __init__(self, concrete_action: FileProcessingAction, input_files: List[File], associated_files: List[File]):
        super().__init__(concrete_action.name, input_files, associated_files, concrete_action.base_dir)
        self.concrete_action = concrete_action

    def get_label(self) -> str:
        return self.concrete_action.get_label()

    def get_log_prefix(self) -> str:
        return self.concrete_action.get_log_prefix()

    def is_accepted(self, f: FitsFile) -> bool:
        return self.concrete_action.is_accepted(f)

    def execute(self, main_upstream_outputs: List[ExecutionResult], associated_upstream_outputs: List[ExecutionResult],
                recipe_parameters: Dict) -> ExecutionResult:
        self.logger.debug("%s Running future job", self.get_log_prefix())
        combined_inputs = self.concrete_action.combine_inputs(main_upstream_outputs, associated_upstream_outputs)
        return self.concrete_action.handle_existing_job(self.name, combined_inputs, recipe_parameters)
