import logging
import threading
from threading import Event
from typing import Dict, Optional, List
from uuid import UUID

from edps.client.BreakpointsStateDTO import BreakpointsStateDTO, BreakpointStateDTO, BreakpointedJobState, BreakpointInspectResultDTO
from edps.client.FitsFile import FitsFile
from edps.executor.command import Command
from edps.executor.recipe import RecipeData, File
from edps.interfaces.JobsRepository import JobNotFoundError

logger = logging.getLogger("BreakpointManager")


class JobHolder:
    def __init__(self):
        self.breakpoint: Event = threading.Event()
        self.command: Optional[Command] = None
        self.new_data: Optional[RecipeData] = None
        self.waiting: bool = False
        self.last_iteration: bool = False

    def is_looping(self) -> bool:
        return self.new_data is not None

    def is_last_iteration(self) -> bool:
        return self.last_iteration

    def pop_data(self) -> RecipeData:
        self.waiting = False
        self.breakpoint.clear()
        result = self.new_data or RecipeData()
        self.new_data = None
        self.command = None
        return result

    def get_state(self) -> BreakpointedJobState:
        if self.waiting:
            return BreakpointedJobState.STOPPED_ON_BREAKPOINT
        else:
            return BreakpointedJobState.RUNNING

    def loop(self, data: RecipeData):
        self.new_data = data
        self.breakpoint.set()

    def loop_once(self, parameters: Dict):
        self.new_data = RecipeData(parameters)
        self.last_iteration = True
        self.breakpoint.set()

    def wait(self, command: Command) -> bool:
        self.command = command
        self.waiting = True
        self.breakpoint.wait()
        return self.is_looping()


class BreakpointManager:
    def __init__(self):
        self.breakpoints: Dict[UUID, JobHolder] = {}
        self.stopped = False

    def get_holder_or_empty(self, job_id: UUID) -> Optional[JobHolder]:
        return self.breakpoints.get(job_id, None)

    def get_holder(self, job_id: UUID) -> JobHolder:
        job_holder = self.get_holder_or_empty(job_id)
        if job_holder:
            return job_holder
        else:
            raise JobNotFoundError("Job {} is not breakpointed".format(job_id))

    def create_breakpoint(self, job_id: UUID):
        logger.debug("Creating breakpoint for %s", job_id)
        self.breakpoints[job_id] = JobHolder()

    def create_breakpoints(self, job_ids: List[UUID]):
        for job_id in job_ids:
            self.create_breakpoint(job_id)

    def remove_breakpoint(self, job_id: UUID):
        logger.debug("Removing breakpoint for %s", job_id)
        self.breakpoints.pop(job_id, None)

    def continue_execution(self, job_id: UUID) -> UUID:
        logger.debug("Continue execution %s", job_id)
        self.get_holder(job_id).breakpoint.set()
        self.remove_breakpoint(job_id)
        return job_id

    def loop_execution(self, job_id: UUID, sof: Optional[List[FitsFile]], parameters: Optional[Dict]) -> UUID:
        logger.debug("Loop execution %s with %s and %s", job_id, sof, parameters)
        job_holder = self.get_holder(job_id)
        job_holder.loop(RecipeData(parameters, [File(f.name, f.category, "") for f in sof] if sof else None))
        return job_id

    def loop_execution_once(self, job_ids: List[UUID], parameters: Dict):
        logger.debug("Loop execution one last time for %s with %s", job_ids, parameters)
        for job_id in job_ids:
            self.get_holder(job_id).loop_once(parameters)

    def wait_if_needed(self, command: Command) -> bool:
        logger.debug("Wait for breakpoint %s if needed", command.name)
        job_holder = self.breakpoints.get(command.name, None)
        if job_holder:
            return job_holder.wait(command)
        return False

    def get_results(self, job_id: UUID) -> Optional[BreakpointInspectResultDTO]:
        command = self.get_holder(job_id).command
        if command:
            return BreakpointInspectResultDTO.from_execution_result(command.result)
        raise JobNotFoundError("Job {} is not yet ready to provide results".format(job_id))

    def get_data(self, job_id: UUID) -> Optional[RecipeData]:
        if self.stopped:
            raise KeyboardInterrupt()
        holder = self.get_holder_or_empty(job_id)
        if holder:
            data = holder.pop_data()
            if holder.is_last_iteration():
                self.remove_breakpoint(job_id)
            return data
        else:
            return RecipeData()

    def get_breakpoints_state(self) -> BreakpointsStateDTO:
        return BreakpointsStateDTO(
            breakpoints=[BreakpointStateDTO(
                job_id=str(job_id),
                state=brk.get_state(),
                next_parameters=brk.new_data.parameters if brk.new_data else None,
                next_inputs=[file.as_fits_file() for file in brk.new_data.inputs] if (brk.new_data and brk.new_data.inputs) else None)
                for job_id, brk in self.breakpoints.items()
            ]
        )

    def shutdown(self):
        self.stopped = True
        b = self.breakpoints
        self.breakpoints = {}
        for holder in b.values():
            holder.breakpoint.set()
