import datetime
import logging
from typing import List, Dict
from uuid import UUID

from .breakpoint import BreakpointManager
from .cloner import JobCloner
from .command import Command
from .filtering import ProductFilter
from .recipe import CommandInvoker, RecipeInputs, RecipeInvocationArguments
from .renamer import ProductRenamer
from .reporting import ReportingScript
from ..client.ProcessingJobStatus import ProcessingJobStatus
from ..client.ReportsConfiguration import ReportsConfiguration
from ..interfaces.JobsRepository import ExecutionResult

logger = logging.getLogger("CascadeExecutionCommand")


class CLICommand(Command):
    def __init__(self, job_id: UUID, command: str, base_dir: str, input_map: Dict[str, str],
                 report_scripts: List[ReportingScript], product_renamer: ProductRenamer, invoker: CommandInvoker,
                 breakpoint_manager: BreakpointManager, job_cloner: JobCloner, output_filter: ProductFilter,
                 reports_configuration: ReportsConfiguration, task_name: str):
        super().__init__(job_id, base_dir, report_scripts, reports_configuration, task_name, output_filter)
        self.invoker = invoker
        self.command = command
        self.input_map = input_map
        self.renamer = product_renamer
        self.breakpoint_manager = breakpoint_manager
        self.job_cloner = job_cloner

    def get_label(self) -> str:
        return self.command

    def run_command(self, recipe_inputs: RecipeInputs, parameters: Dict) -> ExecutionResult:
        should_loop = True
        while should_loop:
            data = self.breakpoint_manager.get_data(self.name)
            parameters = data.parameters or parameters
            recipe_inputs = RecipeInputs(data.inputs) if data.inputs else recipe_inputs
            job_dir = self.get_base_dir()
            input_reports = self.report_generator.generate_input_reports(recipe_inputs.combined)
            recipe_arguments = RecipeInvocationArguments(inputs=recipe_inputs, parameters=parameters, job_dir=job_dir, input_map=self.input_map,
                                                         logging_prefix=self.get_log_prefix())
            result = self.invoker.invoke(command=self.command, parameters=recipe_arguments, create_subdir=False, renamer=self.renamer)
            filtered_outputs = [f for f in result.output_files if self.output_filter.is_accepted(f)]
            if result.return_code == 0:
                output_reports = self.report_generator.generate_output_reports(result.output_files)
                input_output_reports = self.report_generator.generate_input_output_reports(recipe_inputs.combined + result.output_files)
                self.result = ExecutionResult(status=ProcessingJobStatus.COMPLETED,
                                              input_files=recipe_inputs.sorted,
                                              output_files=filtered_outputs,
                                              logs=self.get_logfiles(),
                                              reports=input_reports + output_reports + input_output_reports,
                                              recipe_parameters=parameters,
                                              completion_date=datetime.datetime.now().isoformat())
            else:
                self.result = ExecutionResult(status=ProcessingJobStatus.FAILED,
                                              input_files=recipe_inputs.sorted,
                                              output_files=filtered_outputs,
                                              logs=self.get_logfiles(),
                                              reports=input_reports,
                                              recipe_parameters=parameters,
                                              completion_date=datetime.datetime.now().isoformat())
            should_loop = self.breakpoint_manager.wait_if_needed(self)
            if should_loop:
                self.job_cloner.clone_job(self.name, self.result)
        return self.result
