import datetime
import logging
import os
from typing import List, Dict, Optional
from uuid import UUID

from . import utils
from .filtering import ProductFilter
from .recipe import RecipeInputs, File
from .reporting import ReportingScript, ReportGenerator
from ..client.FitsFile import FitsFile
from ..client.ProcessingJob import LogEntry
from ..client.ProcessingJobStatus import ProcessingJobStatus
from ..client.ReportsConfiguration import ReportsConfiguration, ALL_REPORTS
from ..interfaces.JobScheduler import JobScheduler
from ..interfaces.JobsRepository import ExecutionResult
from ..test.dsl.configuration import TestHelper, TestFitsFile

logger = logging.getLogger("CascadeExecutionCommand")


class Command:
    def __init__(self, job_id: UUID,
                 base_dir: str,
                 report_scripts: List[ReportingScript],
                 reports_configuration: ReportsConfiguration,
                 task_name: str,
                 output_filter:ProductFilter):
        self.name = job_id
        self.base_dir = base_dir
        self.report_scripts = report_scripts
        self.result: Optional[ExecutionResult] = None
        self.task_name = task_name
        self.report_generator = ReportGenerator(self.get_base_dir(),
                                                self.report_scripts,
                                                self.get_log_prefix,
                                                reports_configuration,
                                                task_name)
        self.output_filter = output_filter

    def get_base_dir(self) -> str:
        return utils.get_path(str(self.name), self.base_dir)

    def get_path(self, item: str):
        return utils.get_path(item, self.get_base_dir())

    def get_logfiles(self) -> List[LogEntry]:
        return [LogEntry(file_name=name) for name in JobScheduler.get_logfiles() if os.path.exists(self.get_path(name))
                and os.path.getsize(self.get_path(name)) > 0]

    def get_log_prefix(self) -> str:
        return f"[{self.get_label()} {self.name}]"

    def get_label(self) -> str:
        raise NotImplementedError

    def run_command(self, recipe_inputs: RecipeInputs, parameters: Dict) -> ExecutionResult:
        raise NotImplementedError


class DummyCommand(Command):
    def __init__(self, name: UUID, recipe: str, base_dir: str, report_scripts: List[ReportingScript], task_name: str, output_filter:ProductFilter):
        super().__init__(name, base_dir, report_scripts, ALL_REPORTS, task_name, output_filter)
        self.recipe = recipe
        self.task_name = task_name

    def get_label(self) -> str:
        return self.recipe

    def run_command(self, recipe_inputs: RecipeInputs, parameters: Dict) -> ExecutionResult:
        logger.debug("Running DummyTask with inputs %s and parameters %s", recipe_inputs.combined, parameters)
        input_reports = self.report_generator.generate_input_reports(recipe_inputs.combined)
        output_files = [File(os.path.join(self.get_base_dir(), "{}_{}_result.fits".format(self.name, self.recipe)),
                             "DUMMY", str(self.name)),
                        File(os.path.join(self.get_base_dir(), "{}_{}_science_result.fits".format(self.name, self.recipe)),
                             "SCIENCE.DUMMY", str(self.name))
                        ]
        for file in output_files:
            TestHelper.create_file(TestFitsFile.builder().with_name(os.path.basename(file.file_path)).with_keyword("DUMMYKWD", "DUMMY").build(),
                                   self.get_base_dir())
        for name in JobScheduler.get_logfiles():
            with open(os.path.join(self.get_base_dir(), name), 'wb') as log_file:
                log_file.write(b"Dummy esorex log")
        self.result = ExecutionResult(status=ProcessingJobStatus.COMPLETED,
                                      input_files=recipe_inputs.sorted,
                                      output_files=[FitsFile(name=f.file_path, category=f.category) for f in output_files],
                                      logs=self.get_logfiles(),
                                      reports=input_reports,
                                      completion_date=datetime.datetime.now().isoformat(),
                                      recipe_parameters=parameters
                                      )
        logger.debug("Dummy task result is: %s", self.result)
        return self.result
