import importlib
import json
import logging
import os
import uuid
from dataclasses import dataclass
from typing import List, Dict, Optional, Callable

from edps.client.FitsFile import FitsFile
from edps.executor import utils
from edps.executor.constants import ESOREX_INPUT, ESOREX_OUTPUT, OMP_NUM_THREADS, README_FUNCTION
from edps.executor.renamer import ProductRenamer
from edps.executor.sof import SofWriter
from edps.generator.fits import FitsFileFactory
from edps.interfaces.JobScheduler import JobScheduler
from edps.scheduler.LocalJobScheduler import LocalJobScheduler
from edps.utils import log_time

logger = logging.getLogger("CommandInvoker")


class File:
    def __init__(self, file_path: str, category: str, target_name: str):
        self.target_name = target_name
        self.file_path = file_path
        self.category = category

    def __repr__(self) -> str:
        return self.file_path + " " + self.category + " generated by " + self.target_name

    def as_fits_file(self) -> FitsFile:
        return FitsFile(name=self.file_path, category=self.category)


@dataclass
class RecipeData:
    parameters: Optional[Dict[str, str]] = None
    inputs: Optional[List[File]] = None


class RecipeInputs:
    def __init__(self, main_inputs: List[File] = None, associated_inputs: List[File] = None, main_upstream_inputs: List[File] = None,
                 associated_upstream_inputs: List[File] = None):
        self.main_inputs: List[File] = main_inputs or []
        self.main_upstream_inputs: List[File] = main_upstream_inputs or []
        self.associated_inputs: List[File] = associated_inputs or []
        self.associated_upstream_inputs: List[File] = associated_upstream_inputs or []

        self.associated: List[FitsFile] = [f.as_fits_file() for f in self.associated_upstream_inputs + self.associated_inputs]
        self.main: List[FitsFile] = [f.as_fits_file() for f in self.main_upstream_inputs + self.main_inputs]
        self.combined: List[FitsFile] = self.main + self.associated
        self.sorted: List[FitsFile] = sorted(self.combined, key=lambda x: x.name)


@dataclass
class RecipeInvocationArguments:
    inputs: RecipeInputs
    parameters: Dict[str, str]
    job_dir: str
    input_map: Dict[str, str]
    logging_prefix: str


@dataclass
class RecipeInvocationResult:
    return_code: int
    output_files: List[FitsFile]


RecipeFunction = Callable[[RecipeInvocationArguments, 'InvokerProvider', ProductRenamer], RecipeInvocationResult]


class CommandInvoker:
    def __init__(self, scheduler: Optional[JobScheduler] = None, default_omp_threads: int = 1):
        self.scheduler = scheduler or LocalJobScheduler(1)
        self.default_omp_threads = default_omp_threads

    @log_time()
    def invoke(self, command: str, parameters: RecipeInvocationArguments, renamer: ProductRenamer, create_subdir=False) -> RecipeInvocationResult:
        results = self._generate_results(command, parameters, renamer, create_subdir)
        return RecipeInvocationResult(return_code=results.return_code, output_files=[self._load_keywords(file) for file in results.output_files])

    def _generate_results(self, command: str, parameters: RecipeInvocationArguments, renamer: ProductRenamer, create_subdir=False) -> RecipeInvocationResult:
        job_dir = parameters.job_dir
        if create_subdir:
            job_dir = os.path.join(job_dir, str(uuid.uuid4()))
        os.makedirs(job_dir, exist_ok=True)
        sof_path = utils.get_path(ESOREX_INPUT, job_dir)
        SofWriter.generate_sof(parameters.inputs.combined, sof_path, parameters.logging_prefix, parameters.input_map)
        cmd = self.get_command(command, job_dir, parameters.parameters)
        logger.debug("Invoking %s", cmd)
        ret_code = self.scheduler.execute(cmd, job_dir, int(parameters.parameters.get(OMP_NUM_THREADS, self.default_omp_threads)))
        logger.debug("%s Execution completed with %s", parameters.logging_prefix, ret_code)
        output_files = self._load_output_files(job_dir, parameters.logging_prefix)
        output_files = [FitsFile(name=renamer.rename(utils.get_path(r.name, job_dir)), category=r.category, keywords=r.keywords) for r in output_files]
        return RecipeInvocationResult(return_code=ret_code, output_files=output_files)

    def get_command(self, command: str, base_dir: str, parameters: Dict) -> List[str]:
        return [command, "--products-sof=" + utils.get_path(ESOREX_OUTPUT, base_dir),
                '--recipe-config=' + self._prepare_config_file(parameters, base_dir), utils.get_path(ESOREX_INPUT, base_dir)]

    def _prepare_config_file(self, parameters: Dict, base_dir: str) -> str:
        name = utils.get_path("parameters.rc", base_dir)
        with open(name, 'wb') as config_file:
            for key, value in parameters.items():
                if key != OMP_NUM_THREADS:
                    config_file.write(f"{key}={value}\n".encode("utf-8"))
        return name

    def _load_output_files(self, base_dir: str, log_prefix: str) -> List[FitsFile]:
        if os.path.exists(utils.get_path(ESOREX_OUTPUT, base_dir)):
            with open(utils.get_path(ESOREX_OUTPUT, base_dir), 'rb') as result_file:
                results = json.loads(result_file.read())
                logger.debug("%s Products %s", log_prefix, results)
                return [FitsFile(name=r['name'], category=r['category']) for r in results]
        else:
            return []

    def _load_keywords(self, file: FitsFile) -> FitsFile:
        keywords = dict()
        try:
            keywords = FitsFileFactory.extract_provided_keywords(file.name, {"PRODCATG"})
        except Exception as e:
            logger.warning(f"Failed reading keywords from product {file.name} due to {e}")
        return FitsFile(name=file.name, category=file.category, keywords=keywords)


class RecipeInvoker(CommandInvoker):
    def __init__(self, esorex_path: str, scheduler: Optional[JobScheduler] = None, default_omp_threads: int = 1):
        super(RecipeInvoker, self).__init__(scheduler, default_omp_threads)
        self.esorex_path = esorex_path

    def get_command(self, recipe: str, base_dir: str, parameters: Dict) -> List[str]:
        return [self.esorex_path, "--log-dir=.", "--products-sof=" + utils.get_path(ESOREX_OUTPUT, base_dir),
                '--recipe-config=' + self._prepare_config_file(parameters, base_dir),
                recipe, utils.get_path(ESOREX_INPUT, base_dir)]


class FunctionInvoker(CommandInvoker):

    def __init__(self, provider: 'InvokerProvider', scheduler: Optional[JobScheduler] = None):
        super().__init__(scheduler)
        self.provider = provider

    def _generate_results(self, command: str, parameters: RecipeInvocationArguments, renamer: ProductRenamer, create_subdir=False) -> RecipeInvocationResult:
        os.makedirs(parameters.job_dir, exist_ok=True)
        dot = command.rindex(".")
        module_name = command[:dot]
        function_name = command[dot + 1:]
        function: RecipeFunction = getattr(importlib.import_module(module_name), function_name)
        logger.info("Invoking %s with inputs %s and parameters %s", command, parameters.inputs.combined, parameters)
        result = RecipeInvocationResult(return_code=1, output_files=[])
        try:
            result = function(parameters, self.provider, renamer)
            with open(os.path.join(parameters.job_dir, README_FUNCTION), 'wb') as log_file:
                log_file.write(f"This directory is artifact of calling FUNCTION '{command}' with outputs: {result.output_files}".encode())
        except Exception as e:
            logger.exception("%s Function %s processing crashed %s", parameters.logging_prefix, command, e)
        logger.debug("%s Execution completed with %s", parameters.logging_prefix, result.return_code)
        return result


class InvokerProvider:
    def __init__(self, esorex_path: str, job_scheduler: JobScheduler, default_omp_threads: int):
        self.function_invoker = FunctionInvoker(provider=self, scheduler=job_scheduler)
        self.cli_invoker = CommandInvoker(scheduler=job_scheduler, default_omp_threads=default_omp_threads)
        self.recipe_invoker = RecipeInvoker(esorex_path=esorex_path, scheduler=job_scheduler, default_omp_threads=default_omp_threads)
