import datetime
import itertools
import logging
import os
import re
import threading
import time

from edps.client.FitsFile import FitsFile
from edps.generator.constants import MJD_UNIX_EPOCH, NIGHT
from edps.generator.fits import FitsFileFactory
from edps.utils import mjd_to_datetime_string


class ProductRenamer:
    def rename(self, path: str) -> str:
        raise NotImplementedError

    def rename_file(self, file: FitsFile) -> FitsFile:
        return FitsFile(name=self.rename(file.name), category=file.category)


class NopProductRenamer(ProductRenamer):
    def rename(self, path: str) -> str:
        return path


class PrefixProductRenamer(ProductRenamer):
    def __init__(self, prefix: str):
        self.prefix = prefix
        self.lock = threading.RLock()
        self.count = itertools.count()
        self.logger = logging.getLogger('PrefixProductRenamer')

    def rename(self, path: str) -> str:
        with self.lock:
            _, file_ext = os.path.splitext(path)
            dirname = os.path.dirname(path)
            while True:
                timestamp = datetime.datetime.now().strftime("%Y-%m-%dT%H:%M:%S") + f".{next(self.count) % 1000:03d}"
                new_path = os.path.join(dirname, f'{self.prefix}.{timestamp}{file_ext}')
                if os.path.exists(new_path):
                    self.logger.warning(f"Target renaming path {new_path} already exists, waiting another second")
                    time.sleep(1)
                    continue
                else:
                    self.logger.debug(f"Renaming {path} into {new_path}")
                    os.rename(path, new_path)
                    return new_path


class PatternProductRenamer(ProductRenamer):
    def __init__(self, pattern: str, dataset: str = "", submission_date: str = "", task_name: str = ""):
        self.pattern = pattern
        self.dataset = dataset
        self.submission_date = submission_date
        self.task_name = task_name
        self.illegal_characters = ["\\", "/", "!", "?", "*", "~", "&", " ", "\n", "\r"]

    def rename(self, path: str) -> str:
        final_name = self.pattern
        final_name = final_name.replace("$FILENAME", os.path.basename(path))
        _, ext = os.path.splitext(path)
        final_name = final_name.replace("$EXT", ext.replace(".", ""))
        final_name = final_name.replace("$TASK", self.task_name)
        final_name = final_name.replace("$TIMESTAMP", self.submission_date)
        final_name = final_name.replace("$DATASET", self.dataset)
        keywords = set(re.findall("\\$(.*?)\\$", final_name))
        file = FitsFileFactory.create_fits_file(path, keywords)
        with file:
            if "$NIGHT" in self.pattern:
                night = file.get_keyword_value(NIGHT, MJD_UNIX_EPOCH)
                final_name = final_name.replace("$NIGHT", mjd_to_datetime_string(night, "%Y-%m-%d"))
            for keyword in keywords:
                keyword_value = self.sanitize(str(file.get_keyword_value(keyword, "")))
                final_name = final_name.replace("${}$".format(keyword), keyword_value)
        return final_name

    def sanitize(self, value: str) -> str:
        for illegal in self.illegal_characters:
            value = value.replace(illegal, "_")
        return value
