import json
import logging
import os.path
import shutil
from dataclasses import dataclass
from typing import List, Callable

from edps.client.FitsFile import FitsFile
from edps.client.ProcessingJob import ReportEntry, ReportEntryPanel
from edps.client.ReportsConfiguration import ReportsConfiguration
from edps.executor import utils
from edps.executor.constants import ESOREX_INPUT, GENREPORT_RECIPE_OUTPUT, GENREPORT_RECIPE_INPUT_OUTPUT
from edps.executor.sof import SofWriter
from edps.generator.task_details import ReportInput
from edps.utils import log_time, run_command

logger = logging.getLogger("Reporting")


@dataclass
class ReportingScriptPanel:
    output_name: str
    output_type: str
    description: str
    prodcatg: str
    input_files: List[str]

    def to_dto(self) -> ReportEntryPanel:
        return ReportEntryPanel(file_name=self.output_name, media_type=self.output_type, description=self.description,
                                prodcatg=self.prodcatg, input_files=self.input_files)


@dataclass
class ReportingScriptResult:
    report_name: str
    exit_code: int
    panels: List[ReportingScriptPanel]

    def to_dto(self) -> ReportEntry:
        return ReportEntry(report_name=self.report_name, return_code=self.exit_code,
                           panels=[p.to_dto() for p in self.panels])


class ReportingScript:
    def __init__(self, genreport_path: str, command: str, input_type: str, driver: str, save_inputs: bool = False):
        self.genreport_path = genreport_path
        self.command = command
        self.input_type = ReportInput(input_type)
        self.driver = driver
        self.save_inputs = save_inputs

    @log_time()
    def execute(self, directory: str, sof: str) -> ReportingScriptResult:
        try:
            reporting_directory = os.path.join(directory, self.command)
            if not os.path.exists(reporting_directory):
                os.mkdir(reporting_directory)
            shutil.copy(os.path.join(directory, sof), reporting_directory)
            logger.info("Running reporting script %s in %s for %s", self.command, reporting_directory, sof)
            output_json = "output.json"
            command = [self.genreport_path, '--inputs=' + sof, '--driver=' + self.driver, '--outputs=' + output_json,
                       self.command]
            with open(os.path.join(reporting_directory, "genreport.stdout"), 'wb') as stdout_file, \
                    open(os.path.join(reporting_directory, "genreport.stderr"), 'wb') as stderr_file:
                ret_code = run_command(command, reporting_directory, stdout_file, stderr_file)
                if ret_code != 0:
                    logger.warning("Report execution finished with ret_code %d", ret_code)
                panels = self.parse_outputs(os.path.join(reporting_directory, output_json))
                return ReportingScriptResult(self.command, ret_code, panels)
        except Exception as e:
            logger.error("Report execution crashed", exc_info=e)
            return ReportingScriptResult(self.command, -1, [])

    def parse_outputs(self, output_json: str) -> List[ReportingScriptPanel]:
        if os.path.exists(output_json):
            with open(output_json, 'rb') as result_file:
                results = json.loads(result_file.read())
                logger.debug("Report results %s", results)
                return [ReportingScriptPanel(output_name=os.path.basename(r['filename']),
                                             output_type=r['mime_type'],
                                             description=r['description'],
                                             prodcatg=r.get('prodcatg'),
                                             input_files=r.get('input_files') if self.save_inputs else [])
                        for r in results]
        else:
            logger.warning("Output report file %s not found", output_json)
            return []

    def __repr__(self) -> str:
        return f'ReportingScript(report={self.command}, inputs={self.input_type}, driver={self.driver})'


class ReportGenerator:
    def __init__(self, job_dir: str,
                 report_scripts: List[ReportingScript],
                 get_log_prefix: Callable[[], str], reports_configuration: ReportsConfiguration,
                 task_name: str):
        self.job_dir = job_dir
        self.report_scripts = report_scripts
        self.get_log_prefix = get_log_prefix
        self.reports_configuration = reports_configuration
        self.task_name = task_name

    def generate_input_reports(self, input_files: List[FitsFile]) -> List[ReportEntry]:
        SofWriter.generate_sof(input_files, utils.get_path(ESOREX_INPUT, self.job_dir), self.get_log_prefix())
        return self.generate_reports(ReportInput.RECIPE_INPUTS)

    def generate_output_reports(self, input_files: List[FitsFile]) -> List[ReportEntry]:
        SofWriter.generate_sof(input_files, utils.get_path(GENREPORT_RECIPE_OUTPUT, self.job_dir),
                               self.get_log_prefix())
        return self.generate_reports(ReportInput.RECIPE_OUTPUTS)

    def generate_input_output_reports(self, input_files: List[FitsFile]) -> List[ReportEntry]:
        SofWriter.generate_sof(input_files, utils.get_path(GENREPORT_RECIPE_INPUT_OUTPUT, self.job_dir),
                               self.get_log_prefix())
        return self.generate_reports(ReportInput.RECIPE_INPUTS_OUTPUTS)

    @staticmethod
    def get_sof(input_type: ReportInput) -> str:
        if input_type == ReportInput.RECIPE_INPUTS:
            return ESOREX_INPUT
        elif input_type == ReportInput.RECIPE_INPUTS_OUTPUTS:
            return GENREPORT_RECIPE_INPUT_OUTPUT
        else:
            return GENREPORT_RECIPE_OUTPUT

    def generate_reports(self, input_type: ReportInput) -> List[ReportEntry]:
        results = []
        if self.reports_configuration.should_run(input_type, self.task_name):
            scripts = [script for script in self.report_scripts if script.input_type == input_type]
            for script in scripts:
                sof = ReportGenerator.get_sof(input_type)
                script_result = script.execute(self.job_dir, sof)
                results.append(script_result.to_dto())
        return results
