import datetime
import logging
import os
import threading
from typing import List, Dict
from uuid import UUID

import requests

from .action import Action
from .constants import ERROR_LOG
from .dataset_packager import DatasetPackager
from ..client.ProcessingJob import LogEntry
from ..client.ProcessingJobStatus import ProcessingJobStatus
from ..interfaces.JobsRepository import JobsRepository, ExecutionResult

logger = logging.getLogger("CascadeExecutionTask")


class Task(object):
    def __init__(self, job_id: UUID, action: Action, main_parents: List['Task'],
                 associated_parents: List['Task'], repository: JobsRepository,
                 packager: DatasetPackager,
                 recipe_parameters: Dict, continue_on_error=False):
        self.job_id = job_id
        self.repository = repository
        self.packager = packager
        self.ready = threading.Event()
        self.result = ExecutionResult(ProcessingJobStatus.CREATED, recipe_parameters=recipe_parameters)
        self.action = action
        self.main_parents = main_parents
        self.associated_parents = associated_parents
        self.recipe_parameters = recipe_parameters
        self.continue_on_error = continue_on_error

    @property
    def parents(self):
        return self.main_parents + self.associated_parents

    def execute(self, callback_url: str) -> ExecutionResult:
        logger.debug("%s Initialize execution", self)
        try:
            main_inputs = self.collect_parents(self.main_parents)
            associated_inputs = self.collect_parents(self.associated_parents)
            logger.debug("%s Ready for execution", self)
            self.result = self.action.execute(main_inputs, associated_inputs, self.recipe_parameters)
            self.packager.package(self.result)
        except BaseException as e:
            logger.exception("%s Job execution crashed prematurely because: %s", self, e)
            os.makedirs(self.action.get_job_dir(), exist_ok=True)
            error_log_path = os.path.join(self.action.get_job_dir(), ERROR_LOG)
            with open(error_log_path, 'w') as log_file:
                log_file.write(f"{self} job execution crashed prematurely because: {e}")
            self.result = ExecutionResult(status=ProcessingJobStatus.FAILED,
                                          logs=[LogEntry(file_name=ERROR_LOG)],
                                          completion_date=datetime.datetime.now().isoformat(),
                                          recipe_parameters=self.recipe_parameters,
                                          interrupted=isinstance(e, KeyboardInterrupt))
        logger.debug("%s Marking as completed", self)
        self.ready.set()
        try:
            self.repository.set_job_status(self.action.name, self.result)
        except Exception as e:
            logger.error("Failed to update status of job %s in DB due to %s", self, e, exc_info=e)
        if callback_url:
            requests.get(callback_url)
        return self.result

    def collect_parents(self, parents: List['Task']) -> List[ExecutionResult]:
        inputs = []
        for parent in parents:
            logger.debug("%s Waiting for parent %s to complete", self, parent)
            parent.wait()
            logger.debug("%s Parent %s completed, fetching results", self, parent)
            result = parent.get_result()
            if result.is_complete() or self.continue_on_error:
                inputs.append(result)
            else:
                if result.interrupted:
                    logger.debug("%s Parent %s interrupted, breaking execution", self, parent)
                    raise KeyboardInterrupt(f"Parent {parent} was interrupted")
                else:
                    logger.debug("%s Parent %s failed, breaking execution", self, parent)
                    raise RuntimeError(f"Parent {parent} execution failed")

        return inputs

    def wait(self):
        self.ready.wait()

    def get_result(self) -> ExecutionResult:
        return self.result

    def __repr__(self) -> str:
        return "{} {} {} ({}) ({})".format(self.job_id, self.action.get_label(), self.action.name, [x.action.name for x in self.main_parents],
                                           [x.action.name for x in self.associated_parents])
