from __future__ import annotations

import sys
from dataclasses import dataclass
from datetime import datetime, timedelta
from enum import Enum
from typing import List

from .association_breakpoints import AssociationBreakpoints
from .association_preference import AssociationPreference
from .fits import ClassifiedFitsFile
from .job import Job, AssociationResult
from .task import AssociationConfiguration
from .. import JobParameters


def file_diff(ref: ClassifiedFitsFile, f: ClassifiedFitsFile, keywords: List[str]) -> List[float]:
    with ref:
        with f:
            res = []
            for key in keywords:
                try:
                    res.append(abs(ref[key] - f[key]))
                except TypeError:
                    res.append(sys.float_info.max)
            return res


def order_assoc_results_by(inputs: List[AssociationResult], ref_file: ClassifiedFitsFile, sort_keys: List[str],
                           parameters: JobParameters) -> List[AssociationResult]:
    return sorted(inputs, key=lambda res: file_diff(ref_file, res.exemplar, parameters.resolve_keywords(sort_keys)))


def order_files_by(inputs: List[ClassifiedFitsFile], ref_file: ClassifiedFitsFile, sort_keys: List[str],
                   parameters: JobParameters) -> List[ClassifiedFitsFile]:
    return sorted(inputs, key=lambda file: file_diff(ref_file, file, parameters.resolve_keywords(sort_keys)))


def order_jobs_by(inputs: List[Job], ref_file: ClassifiedFitsFile, sort_keys: List[str],
                  parameters: JobParameters) -> List[Job]:
    # job.assoc_level: DFS-20574
    # job.submission_date: DFS-21627
    # assoc level and submission date are only used if we have identical associations (eg. same files reduced multiple times)
    return sorted(inputs, key=lambda job: file_diff(ref_file, job.exemplar, parameters.resolve_keywords(sort_keys)) +
                                          [job.assoc_level, datetime_inverse_order(job.submission_date)])


def datetime_inverse_order(submission_date: str) -> timedelta:
    return datetime.max - datetime.fromisoformat(submission_date)


def limit(lst: List, max_ret: int):
    return lst[0:max_ret]


class ReferenceFile:
    def __init__(self, config: AssociationConfiguration, ref_file: ClassifiedFitsFile):
        self.config = config
        self.ref_file = ref_file

    def is_associated(self, file: ClassifiedFitsFile, parameters: JobParameters) -> bool:
        with self.ref_file:
            with file:
                return self.config.are_associated(parameters.wrap_file(self.ref_file), parameters.wrap_file(file))


class RequestType(Enum):
    REDUCTION = 1
    ORGANIZATION = 2
    CALSELECTOR = 3


@dataclass
class AssociationArgs:
    breakpoints: AssociationBreakpoints
    preference: AssociationPreference
    request_type: RequestType = RequestType.REDUCTION
