import inspect
import uuid
from typing import Callable, Dict, Optional, Union

from edps.client.WorkflowDTO import ClassificationRuleDTO, ClassificationDict
from edps.generator.constants import PRO_CATG
from .fits import FitsFile

ClassificationFunction = Callable[[FitsFile], bool]


class BaseClassificationRule:
    def __init__(self, classification: str):
        self.id = str(uuid.uuid4())
        self.classification = classification
        self.function: Optional[str] = None
        self.keyword_values: Optional[ClassificationDict] = None

    def as_dto(self) -> ClassificationRuleDTO:
        return ClassificationRuleDTO(category=self.classification, function=self.function,
                                     keyword_values=self.keyword_values)

    def is_classified(self, f: FitsFile) -> bool:
        raise NotImplementedError

    def is_product(self):
        return self.keyword_values and PRO_CATG in self.keyword_values

    def as_str(self) -> str:
        raise NotImplementedError


class FunctionClassificationRule(BaseClassificationRule):
    def __init__(self, classification: str, condition: ClassificationFunction):
        super().__init__(classification)
        self._is_classified = condition
        self.function = condition.__name__

    def is_classified(self, f: FitsFile) -> bool:
        with f:
            return self._is_classified(f)

    def as_str(self) -> str:
        return inspect.getsource(self._is_classified)


class ClassificationRule(FunctionClassificationRule):
    pass


class DictionaryClassificationRule(BaseClassificationRule):
    def __init__(self, classification: str, keyword_values: ClassificationDict):
        super().__init__(classification)
        self.keyword_values = keyword_values

    def is_classified(self, f: FitsFile) -> bool:
        with f:
            for key, value in self.keyword_values.items():
                if isinstance(value, list):
                    if f[key] not in value:
                        return False
                else:
                    if f[key] != value:
                        return False
        return True

    def as_str(self) -> str:
        return str(self.keyword_values)


class ProductClassificationRule(DictionaryClassificationRule):
    def __init__(self, classification: str):
        super().__init__(classification, {PRO_CATG: classification})


def classification_rule(classification: str,
                        condition: Union[ClassificationFunction, Dict, None] = None) -> BaseClassificationRule:
    if condition is None:
        return ProductClassificationRule(classification)
    elif callable(condition):
        return FunctionClassificationRule(classification, condition)
    elif isinstance(condition, dict):
        return DictionaryClassificationRule(classification, condition)
    else:
        raise TypeError(f"Invalid condition for classification {classification}")
