import logging
import time
from pathlib import Path
from typing import List, Set

from .classif_rule import BaseClassificationRule
from .fits import ClassifiedFitsFile, FitsFile, FitsFileFactory


class Classifier:
    logger = logging.getLogger('Classifier')

    @staticmethod
    def parse_and_classify(fits_factory: FitsFileFactory, paths: List[Path], keywords: Set[str],
                           classification_rules: List[BaseClassificationRule]) -> List[ClassifiedFitsFile]:
        t0 = time.time()
        files = fits_factory.create_fits_files(paths, keywords)
        t1 = time.time()
        classified_files = Classifier.classify_files(files, classification_rules)
        t2 = time.time()
        Classifier.logger.debug(
            'parsed and classified %d files in %.3f seconds. parsing took %.3f s, classification took %.3f s',
            len(files), t2 - t0, t1 - t0, t2 - t1)
        return classified_files

    @staticmethod
    def classify_files(files: List[FitsFile], classif_rules: List[BaseClassificationRule]) -> List[ClassifiedFitsFile]:
        classified_files = []
        for f in files:
            with f:
                matching_rules = Classifier.get_classification(f, classif_rules)
                if not matching_rules:
                    classified_files.append(ClassifiedFitsFile.from_fitsfile(f, None, None))
                else:
                    classified_files.extend([ClassifiedFitsFile.from_fitsfile(f, r.classification, r.id) for r in matching_rules])
        return classified_files

    @staticmethod
    def exception_safe_is_classified(f: FitsFile, rule: BaseClassificationRule) -> bool:
        if f.is_virtual():
            return rule.is_classified(f)
        try:
            return rule.is_classified(f)
        except Exception as e:
            Classifier.logger.warning("exception caught when trying to classify file %s as %s: %s",
                                      f.file_path, rule.classification, e, exc_info=e)
            return False

    @staticmethod
    def get_classification(fitsfile: FitsFile, classif_rules: List[BaseClassificationRule]) -> List[BaseClassificationRule]:
        result = [rule for rule in classif_rules if Classifier.exception_safe_is_classified(fitsfile, rule)]
        if len(result) == 0:
            Classifier.logger.info("file %s has no classification", fitsfile.file_path)
        elif len(result) > 1:
            categories = [res.classification for res in result]
            Classifier.logger.info("file %s has multiple classifications %s", fitsfile.file_path, categories)
        return result
