import itertools
import logging
import uuid
from dataclasses import dataclass
from typing import List, Dict, Set, Optional

from edps.client.AssociationResult import AssociationResultDTO, AssociationEntry
from edps.client.FitsFile import FitsFile as FitsFileDTO
from edps.client.JobInfo import JobInfo
from edps.client.JobSummary import Setup, Header
from edps.client.WorkflowDTO import CalselectorJob
from .constants import MJD_OBS
from .fits import FitsFile, ClassifiedFitsFile
from .parameters import JobParameters
from .task_details import TaskDetails, ActiveCondition, ReportConfig, FilterMode


@dataclass
class AssociatedFitsFile:
    file: ClassifiedFitsFile
    mjdobs: float

    def as_dto(self) -> AssociationEntry:
        return AssociationEntry(name=self.file.file_path,
                                category=self.file.classification,
                                mjdobs=self.mjdobs)

    @classmethod
    def from_name_cat_mjd(cls, name: str, cat: str, mjdobs: float) -> 'AssociatedFitsFile':
        return cls(ClassifiedFitsFile(FitsFile(name, {}), cat, None), mjdobs)


class EmptyAssociationResultError(Exception):
    def __init__(self, message: str):
        self.message = message


@dataclass
class AssociationResult:
    associated_files: List[AssociatedFitsFile]
    associated_jobs: List['Job']
    is_complete: bool = None
    is_optional: bool = None
    assoc_level: float = None
    task_name: str = None

    @property
    def exemplar(self) -> ClassifiedFitsFile:
        """
        :return:
            The exemplar file of this association result
        :raise:
             EmptyAssociationResultError: if this method is invoked on an empty result
        """
        if len(self.associated_files) > 0:
            return self.associated_files[0].file
        elif len(self.associated_jobs) > 0:
            return self.associated_jobs[0].exemplar
        else:
            raise EmptyAssociationResultError("This method must not be invoked on an empty result")

    def as_dto(self) -> AssociationResultDTO:
        files = [f.as_dto() for f in self.associated_files]
        jobs = [AssociationEntry(name=str(job.id),
                                 category=job.exemplar.classification,
                                 mjdobs=job.exemplar[MJD_OBS]) for job in self.associated_jobs]
        return AssociationResultDTO(
            files=files,
            jobs=jobs,
            complete=self.is_complete,
            optional=self.is_optional,
            level=self.assoc_level,
            task_name=self.task_name
        )

    def is_empty(self):
        return not (self.associated_jobs or self.associated_files)

    def compute_is_complete(self) -> bool:
        return bool(self.associated_files or (self.associated_jobs and all(job.is_complete for job in self.associated_jobs)))


class JobKey:
    def __init__(self, job: 'Job'):
        self.initialized = False
        self.job = job
        self.task_name = job.task_name
        self.str_input_files = ""
        self.str_input_jobs = ""
        self.str_associated_files = ""
        self.str_associated_jobs = ""
        self.str_parameters = ""

    def __initialize(self):
        if not self.initialized:
            self.str_associated_files = str(sorted([file.get_path() for file in self.job.associated_files]))
            self.str_associated_jobs = str(sorted([str(job.id) for job in self.job.associated_jobs]))
            self.str_input_files = str(sorted([file.get_path() for file in self.job.input_files]))
            self.str_input_jobs = str(sorted([str(job.id) for job in self.job.input_jobs]))
            parameters = self.job.parameters.as_dto() # DTO to make sure we're consistent with what is stored in the DB and reloaded at startup
            sorted_params = [(param, parameters.recipe_parameters[param]) for param in sorted(parameters.recipe_parameters)]
            self.str_parameters = str(sorted_params)
            self.initialized = True

    def key(self):
        self.__initialize()
        return (self.task_name,
                self.str_input_files,
                self.str_associated_files,
                self.str_input_jobs,
                self.str_associated_jobs,
                self.str_parameters)


class Job:
    def __init__(self, *, task_details: TaskDetails,
                 job_id: uuid.UUID,
                 exemplar: Optional[ClassifiedFitsFile],
                 product: Optional[ClassifiedFitsFile],
                 parameters: JobParameters,
                 input_files: List[ClassifiedFitsFile],
                 input_jobs: List['Job'],
                 setup: Setup,
                 header: Header,
                 meta_targets: Optional[List[str]],
                 submission_date: str):
        self.logger = logging.getLogger('Job')
        self.id = str(job_id)
        self.command: str = task_details.command
        self.command_type: str = task_details.command_type
        self.task_name: str = task_details.task_name
        self.task_id: str = task_details.task_id
        self.input_filter: List[str] = task_details.input_filter
        self.input_filter_mode: FilterMode = task_details.input_filter_mode
        self.output_filter: List[str] = task_details.output_filter
        self.output_filter_mode: FilterMode = task_details.output_filter_mode
        self.input_map: Dict[str, str] = task_details.input_map
        self.workflow_names = task_details.workflow_names
        self.active_condition: ActiveCondition = task_details.active_condition
        self.reports: List[ReportConfig] = task_details.reports
        self.input_jobs = input_jobs or []
        self.input_files = input_files or []
        self.exemplar = exemplar
        self.product = product
        self.parameters: JobParameters = parameters
        self.associated_files: List[ClassifiedFitsFile] = []
        self.associated_jobs: List[Job] = []
        self.association_results: List[AssociationResult] = []
        self.associated_input_groups = []
        self.is_complete: bool = True
        self.assoc_level: float = 0
        self.setup = setup
        self.header = header
        self.future = False
        self.meta_targets = meta_targets or []
        self.submission_date = submission_date
        self.job_key = JobKey(self)

    def __key(self):
        return self.job_key.key()

    def __eq__(self, other: 'Job'):
        return self.__key() == other.__key()

    def __hash__(self):
        return hash(self.__key())

    def set_assoc_level(self, level: float):
        if self.assoc_level < level:
            self.assoc_level = level

    def add_association_result(self, result: AssociationResult):
        self.association_results.append(result)
        self.set_assoc_level(result.assoc_level)
        self.associated_files.extend([f.file for f in result.associated_files])
        self.associated_jobs.extend(result.associated_jobs)

    def add_associated_input_group(self, group):
        self.associated_input_groups.append(group)

    def set_future(self):
        self.future = True

    def get_all_parents(self) -> List['Job']:
        parents = [job.get_all_parents() for job in self.parent_jobs]
        return [self] + list(itertools.chain.from_iterable(parents))

    def get_all_edges(self) -> Set:
        parents = [job.get_all_edges() for job in self.parent_jobs]
        current_level = [(parent, self) for parent in self.parent_jobs]
        return set(list(itertools.chain.from_iterable(parents)) + current_level)

    def to_dot(self) -> str:
        dot = ['digraph G {']
        for edge in self.get_all_edges():
            dot.append('"{}"->"{}"'.format(repr(edge[0]), repr(edge[1])))
        dot.append('}')
        return '\n'.join(dot)

    def dump(self, indent=0):
        print('+' * indent + repr(self))
        indent += 1
        for parent in self.parent_jobs:
            parent.dump(indent)

    @property
    def parent_jobs(self) -> Set['Job']:
        return set(self.input_jobs + self.associated_jobs)

    def as_dict(self) -> Dict:
        return self.as_job_info().model_dump()

    def as_job_info(self) -> JobInfo:
        return JobInfo(
            job_id=self.id,
            command=self.command,
            command_type=self.command_type,
            input_files=[FitsFileDTO(name=f.file_path, category=f.classification) for f in self.input_files],
            associated_files=[FitsFileDTO(name=f.file_path, category=f.classification) for f in self.associated_files],
            input_job_ids=[str(job.id) for job in self.input_jobs],
            associated_job_ids=[str(job.id) for job in self.associated_jobs],
            parameters=self.parameters.as_dto(),
            instrument=self.get_instrument(),
            mjdobs=self.get_mjdobs(),
            task_name=self.task_name,
            assoc_level=self.assoc_level,
            task_id=self.task_id,
            association_details=[details.as_dto() for details in self.association_results],
            workflow_names=self.workflow_names,
            complete=self.is_complete,
            setup=self.setup,
            header=self.header,
            reports=[report.as_dto() for report in self.reports],
            input_filter=self.input_filter,
            input_filter_mode=self.input_filter_mode.name,
            output_filter=self.output_filter,
            output_filter_mode=self.output_filter_mode.name,
            input_map=self.input_map,
            future=self.future,
            meta_targets=self.meta_targets
        )

    def as_calselector_job(self) -> CalselectorJob:
        return CalselectorJob(
            task_name=self.task_name,
            input_files=[file.as_fits_file_dto() for file in self.input_files],
            exemplar=self.exemplar.as_fits_file_dto(),
            associated_input_groups=[inp.as_dto() for inp in self.associated_input_groups]
        )

    def get_instrument(self) -> str:
        return self.exemplar.get_keyword_value('instrume', "UNKNOWN_INSTRUMENT")

    def get_mjdobs(self) -> float:
        return self.exemplar.get_mjdobs()

    def __repr__(self):
        return f'{self.task_id} {self.id}'

    def is_active(self):
        return self.active_condition(self.parameters)
