import logging
import uuid
from copy import deepcopy
from typing import List, Optional
from datetime import datetime

from edps import utils
from edps.client.JobSummary import Setup, Header
from edps.generator.constants import HEADER_KEYWORDS
from edps.generator.constants import MJD_OBS
from edps.generator.fits import ClassifiedFitsFile, FitsFile
from edps.generator.job import Job
from edps.generator.parameters import Parameters, JobParameters
from edps.generator.task_details import TaskDetails, DynamicParameterProvider
from edps.interfaces.JobsRepository import JobDetails


def create_job(task_details: TaskDetails, parameters: Parameters, meta_targets: Optional[List[str]],
               input_files: Optional[List[ClassifiedFitsFile]] = None,
               input_jobs: Optional[List['Job']] = None) -> Job:
    input_files = input_files or []
    input_jobs = input_jobs or []
    if len(input_files) == 0 and len(input_jobs) == 0:
        raise RuntimeError("attempt to create Job with no inputs")
    if input_files:
        exemplar = create_exemplar_for_group(input_files)
        product = create_product(input_files[0])
    else:
        exemplar = input_jobs[0].product
        product = input_jobs[0].product
    parameters = compute_job_parameters(task_details, parameters, input_files, exemplar)
    try:
        setup = compute_setup(task_details.setup_keywords, exemplar)
    except KeyError:
        setup = {}
    try:
        header = compute_header(exemplar)
    except KeyError:
        header = {}
    return Job(task_details=task_details, job_id=uuid.uuid4(), exemplar=exemplar, product=product,
               parameters=parameters, input_files=input_files, input_jobs=input_jobs, setup=setup,
               header=header, meta_targets=meta_targets, submission_date=datetime.now().isoformat())


def recreate_job(task_details: TaskDetails, job: JobDetails, input_files: List[ClassifiedFitsFile]) -> Job:
    job_config = job.configuration
    submission_date = job.submission_date
    job_id = uuid.UUID(job_config.job_id)
    parameters = JobParameters(job_config.parameters.workflow_parameters,
                               job_config.parameters.recipe_parameters,
                               job_config.parameters.workflow_parameter_set,
                               job_config.parameters.recipe_parameter_set)
    return Job(task_details=task_details, job_id=job_id, exemplar=None, product=None,
               parameters=parameters, input_files=input_files, input_jobs=[], setup=job_config.setup,
               header=job_config.header, meta_targets=job_config.meta_targets, submission_date=submission_date)


def compute_job_parameters(task_details: TaskDetails, parameters: Parameters, input_files: List[ClassifiedFitsFile],
                           exemplar: ClassifiedFitsFile) -> JobParameters:
    recipe_parameters = parameters.get_recipe_parameters(task_details.task_name)
    input_files = input_files or [exemplar]
    dynamic_workflow_parameters = {
        parameter: compute_dynamic_parameter(dynamic_parameter_provider, input_files, parameter) for
        parameter, dynamic_parameter_provider in task_details.dynamic_parameters.items()
    }
    logging.debug("dynamic parameters: %s", dynamic_workflow_parameters)
    workflow_parameters = utils.merge_dicts(parameters.get_workflow_parameters(), dynamic_workflow_parameters)
    return JobParameters(workflow_parameters=workflow_parameters,
                         workflow_parameter_set=parameters.workflow_parameter_set,
                         recipe_parameters=recipe_parameters,
                         recipe_parameter_set=parameters.recipe_parameter_set)


def compute_dynamic_parameter(dynamic_parameter_provider: DynamicParameterProvider,
                              input_files: List[ClassifiedFitsFile], parameter: str) -> Optional[object]:
    try:
        return dynamic_parameter_provider(input_files)
    except Exception as e:
        logging.warning("Failed to compute dynamic parameter for key '%s'", parameter, exc_info=e)
        return None


def create_exemplar(input_file: ClassifiedFitsFile, mjd_obs: float) -> ClassifiedFitsFile:
    keywords = deepcopy(input_file.get_keyword_values())
    keywords[MJD_OBS] = mjd_obs
    fitsfile = FitsFile(input_file.fitsfile.file_path, keywords)
    return ClassifiedFitsFile(fitsfile, input_file.classification, input_file.classification_rule_id)


def create_exemplar_for_group(input_files: List[ClassifiedFitsFile]) -> ClassifiedFitsFile:
    input_mjd_obs = [file.get_keyword_value(MJD_OBS, 0) for file in input_files if
                     file.get_keyword_value(MJD_OBS, -1) != -1]
    exemplar_mjd_obs = sum(input_mjd_obs) / len(input_mjd_obs) if input_mjd_obs else 0
    return create_exemplar(input_files[0], exemplar_mjd_obs)


def create_product(exemplar: ClassifiedFitsFile) -> ClassifiedFitsFile:
    fitsfile = FitsFile(exemplar.fitsfile.file_path, deepcopy(exemplar.get_keyword_values()))
    return ClassifiedFitsFile(fitsfile, exemplar.classification, exemplar.classification_rule_id)


def compute_setup(setup_keywords: List[str], exemplar: ClassifiedFitsFile) -> Setup:
    setup = {key: exemplar.get_keyword_value(key, None) for key in setup_keywords}
    return {key: str(val) for key, val in setup.items() if val is not None}


def compute_header(exemplar: ClassifiedFitsFile) -> Header:
    hdr = {key: exemplar.get_keyword_value(key, None) for key in HEADER_KEYWORDS}
    return {key: str(val) for key, val in hdr.items() if val is not None}
