import logging
import os.path
from dataclasses import dataclass
from typing import Dict, List, Optional, Set
from uuid import UUID

import frozendict
import yaml

from edps import utils
from edps.client.JobParametersDTO import JobParametersDTO
from edps.client.ParameterSetDTO import ParameterSetDTO
from edps.client.RequestParameters import RequestParameters
from edps.generator.fits import ParameterResolvingClassifiedFitsFile, ClassifiedFitsFile

WorkflowName = str
ParameterSetName = str
TaskName = str
TaskId = str


@dataclass
class ParameterSet:
    is_default: bool
    tags: List[str]
    recipe_parameters: Dict[TaskName, Dict[str, str]]
    workflow_parameters: Dict[str, str]
    parameter_metadata: Dict[str, str]

    @staticmethod
    def from_dict(data: Dict) -> 'ParameterSet':
        return ParameterSet(is_default=data.get('is_default', False),
                            tags=data.get('tags', []),
                            recipe_parameters=frozendict.frozendict(
                                {task_name: ParameterSet.__stringify_and_freeze(recipe_params) for
                                 task_name, recipe_params in
                                 data.get('recipe_parameters', {}).items()}),
                            workflow_parameters=ParameterSet.__stringify_and_freeze(
                                data.get('workflow_parameters', {})),
                            parameter_metadata=data.get('parameter_metadata', {})
                            )

    def get_recipe_parameters(self) -> Dict[TaskName, Dict[str, str]]:
        return self.recipe_parameters

    def get_workflow_parameters(self) -> Dict[str, str]:
        return self.workflow_parameters

    def get_parameter_metadata(self) -> Dict[str, str]:
        return self.parameter_metadata

    @staticmethod
    def __stringify_and_freeze(data: Dict[str, object]) -> Dict[str, str]:
        return frozendict.frozendict({key: str(value) for key, value in data.items()})


Empty = ParameterSet(is_default=False, tags=[], recipe_parameters={}, workflow_parameters={}, parameter_metadata={})


class ParameterSets:
    def __init__(self, config: Dict[str, List[str]]):
        self.logger = logging.getLogger("ParameterSets")
        self.parameter_sets: Dict[WorkflowName, Dict[ParameterSetName, ParameterSet]] = self.load_parameters(config)
        self.defaults: Dict[WorkflowName, ParameterSetName] = self.compute_defaults()

    def get_recipe_parameters(self, workflow_name: WorkflowName, set_name: ParameterSetName) -> Dict[TaskName, Dict]:
        return self.get_parameter_set(workflow_name, set_name).get_recipe_parameters()

    def get_workflow_parameters(self, workflow_name: WorkflowName, set_name: ParameterSetName) -> Dict:
        return self.get_parameter_set(workflow_name, set_name).get_workflow_parameters()

    def get_parameter_set(self, workflow_name: WorkflowName, set_name: ParameterSetName) -> ParameterSet:
        set_name = set_name if set_name else self.defaults.get(workflow_name, '')
        return self.parameter_sets.get(workflow_name, {}).get(set_name, Empty)

    def get_parameter_sets(self, workflow_name: str) -> List[ParameterSetDTO]:
        parameter_sets = self.parameter_sets.get(workflow_name, {})
        return [ParameterSetDTO(name=name,
                                default=parameter_set.is_default,
                                recipe_parameters=parameter_set.recipe_parameters,
                                workflow_parameters=parameter_set.workflow_parameters,
                                parameter_metadata=parameter_set.parameter_metadata)
                for name, parameter_set in parameter_sets.items()]

    def load_parameters(self, config: Dict[str, List[str]]) -> Dict[WorkflowName, Dict[ParameterSetName, ParameterSet]]:
        workflow_parameters = {wkf: self.load_workflow_parameter_sets(paths) for wkf, paths in config.items()}
        return frozendict.frozendict(workflow_parameters)

    def load_workflow_parameter_sets(self, paths: List[str]) -> Dict[ParameterSetName, ParameterSet]:
        result = {}
        for parameter_file_path in paths:
            self.logger.debug("Loading parameters from %s", parameter_file_path)
            if not os.path.exists(parameter_file_path):
                self.logger.error("Parameter file %s does not exist", parameter_file_path)
            else:
                with open(parameter_file_path) as file:
                    config = yaml.safe_load(file.read())
                    for set_name, set_data in config.items():
                        if set_name in result:
                            raise RuntimeError(f"Parameter set name '{set_name}' is duplicated, "
                                               f"second occurrence in '{parameter_file_path}'")
                        result[set_name] = ParameterSet.from_dict(set_data)
                    self.logger.info("Successfully loaded parameters from %s", parameter_file_path)
        return frozendict.frozendict(result)

    def compute_defaults(self) -> Dict[WorkflowName, ParameterSetName]:
        defaults = {}
        for workflow_name, parameter_sets in self.parameter_sets.items():
            for parameter_set_name, parameter_set in parameter_sets.items():
                if parameter_set.is_default:
                    if workflow_name in defaults:
                        raise RuntimeError(f"Duplicated default parameter set for workflow '{workflow_name}', "
                                           f"first one: '{defaults[workflow_name]}', "
                                           f"second one: '{parameter_set_name}'")
                    defaults[workflow_name] = parameter_set_name
        return frozendict.frozendict(defaults)


class JobParameters:
    def __init__(self, workflow_parameters: Dict, recipe_parameters: Dict, workflow_parameter_set: ParameterSetName,
                 recipe_parameter_set: ParameterSetName):
        # FIXME: we could also freeze those, but job parameters go into a single job, so no crosstalk
        self.workflow_parameter_set = workflow_parameter_set
        self.workflow_parameters = dict(workflow_parameters)
        self.recipe_parameter_set = recipe_parameter_set
        self.recipe_parameters = dict(recipe_parameters)

    def get_workflow_param(self, param_name: str, default_value=None) -> Optional:
        return self.workflow_parameters.get(param_name, default_value)

    def get_recipe_param(self, param_name: str, default_value=None) -> Optional:
        return self.recipe_parameters.get(param_name, default_value)

    def as_dto(self) -> JobParametersDTO:
        return JobParametersDTO(
            workflow_parameter_set=self.workflow_parameter_set,
            workflow_parameters=self.workflow_parameters,
            recipe_parameter_set=self.recipe_parameter_set,
            recipe_parameters=self.recipe_parameters
        )

    def resolve_keywords(self, keywords: List[str]) -> List[str]:
        return [self.resolve_keyword(keyword) for keyword in keywords]

    def resolve_keyword(self, keyword: str) -> str:
        return self.get_workflow_param(keyword) if keyword.startswith("$") else keyword

    def wrap_file(self, file: ClassifiedFitsFile) -> ParameterResolvingClassifiedFitsFile:
        return ParameterResolvingClassifiedFitsFile(file, self.resolve_keyword)

    def __repr__(self):
        sorted_params = [(param, self.workflow_parameters[param]) for param in sorted(self.workflow_parameters)] + \
                        [(param, self.recipe_parameters[param]) for param in sorted(self.recipe_parameters)]
        return str(sorted_params)


class Parameters:
    def __init__(self, workflow_parameters: Dict, recipe_parameters: Dict[TaskName, Dict],
                 workflow_parameter_set: ParameterSetName, recipe_parameter_set: ParameterSetName):
        self.logger = logging.getLogger("Parameters")
        self.workflow_parameter_set = workflow_parameter_set
        self.__workflow_parameters = frozendict.frozendict(workflow_parameters)
        self.recipe_parameter_set = recipe_parameter_set
        self.__recipe_parameters = frozendict.frozendict(recipe_parameters)

    def get_workflow_param(self, param_name: str) -> Optional:
        return self.__workflow_parameters.get(param_name, None)

    def get_workflow_parameters(self) -> Dict:
        return self.__workflow_parameters

    def get_recipe_parameters(self, task_name: str) -> Dict:
        return self.__recipe_parameters.get(task_name, {})

    def is_not_empty(self):
        return len(self.__workflow_parameters) + len(self.__recipe_parameters) > 0

    def resolve_keywords(self, keywords: List[str]) -> List[str]:
        return [self.resolve_keyword(keyword) for keyword in keywords]

    def resolve_keyword(self, keyword: str) -> str:
        resolved_keyword = self.get_workflow_param(keyword) if keyword.startswith("$") else keyword
        if resolved_keyword is None:
            raise RuntimeError(f"Failed to resolve variable keyword '{keyword}', "
                               f"available workflow parameters are: '{self.__workflow_parameters}'")
        return resolved_keyword

    def __repr__(self):
        return f'workflow_parameters={self.__workflow_parameters} recipe_parameters={self.__recipe_parameters}'


class ParametersProvider:
    def __init__(self, parameters_sets: ParameterSets, request_parameters: RequestParameters):
        self.request_parameters = request_parameters
        self.parameter_sets = parameters_sets
        self.logger = logging.getLogger('ParametersProvider')

    def get_request_recipe_parameters(self, workflow_name: str, task_name: str) -> Dict[str, str]:
        yaml_recipe_parameters = self.parameter_sets.get_recipe_parameters(workflow_name,
                                                                           self.request_parameters.recipe_parameter_set)
        return utils.merge_dicts(yaml_recipe_parameters.get(task_name, {}),
                                 self.request_parameters.recipe_parameters.get(task_name, {}))

    def get_parameters(self, task_name: str, workflow_names: Set[str], request_id: UUID) -> Parameters:
        if len(workflow_names) > 1:
            fmt = "[%s %s] has multiple workflows associated: %s, looping over them to find parameters"
            self.logger.debug(fmt, task_name, request_id, workflow_names)
        for workflow_name in workflow_names:
            parameters = self.get_parameters_for_workflow(workflow_name, request_id)
            if parameters.is_not_empty():
                self.logger.debug("Found parameters for workflow %s", workflow_name)
                break
        else:
            parameters = self.get_parameters_for_workflow(None, request_id)
        return parameters

    def get_parameters_for_workflow(self, workflow_name: Optional[str], request_id: UUID) -> Parameters:
        workflow_parameters = utils.merge_dicts(
            self.parameter_sets.get_workflow_parameters(workflow_name, self.request_parameters.workflow_parameter_set),
            self.request_parameters.workflow_parameters)
        recipe_parameters = self.parameter_sets.get_recipe_parameters(workflow_name,
                                                                      self.request_parameters.recipe_parameter_set)
        self.logger.debug(
            "Request %s for workflow %s and with provided parameters configuration %s have effective workflow parameters %s and recipe parameters %s",
            request_id, workflow_name, self.request_parameters, workflow_parameters, recipe_parameters)
        return Parameters(
            workflow_parameters=workflow_parameters,
            workflow_parameter_set=self.request_parameters.workflow_parameter_set,
            recipe_parameters=recipe_parameters,
            recipe_parameter_set=self.request_parameters.recipe_parameter_set
        )
