import logging
import os
import subprocess
import uuid
from dataclasses import dataclass
from typing import List


@dataclass
class RecipeParam:
    name: str
    value: str


def get_recipe_params(esorex_path: str, recipe: str) -> List[RecipeParam]:
    logger = logging.getLogger('DefaultRecipeParametersLoader')
    config_file = f"{str(uuid.uuid4())}"
    command = [esorex_path, "--create-config=" + config_file, "--log-level=off", recipe]
    try:
        res = subprocess.check_output(command)
        logger.debug("Calling esorex for default recipe parameters of %s returned %s", recipe, res)
        recipe_params = []
        if os.path.exists(config_file):
            with open(config_file) as fid:
                lines = fid.read().splitlines()
                params = [line.split("=") for line in lines if not line.startswith("#")]
                recipe_params = [RecipeParam(param[0], param[1]) for param in params if len(param) == 2]
            os.remove(config_file)
        else:
            logger.warning("Output file %s with recipe %s default parameters was not found", config_file, recipe)
        return recipe_params
    except subprocess.CalledProcessError as e:
        logger.warning("Calling esorex via %s for default recipe parameters of %s failed with: %s, %s, %s", e.cmd,
                       recipe, e.returncode, e.stderr, e.stdout)
        raise RuntimeError("Failed to get default recipe parameters")
