import inspect
from typing import List, Dict, Optional

from edps import empty
from .assoc_config import MatchKeywords, MatchFunction
from .constants import TRUE_CONDITION
from .grouping import (Grouper, KeywordGrouping, SimpleClusterGrouping, SkyPositionClusterGrouping, GroupingFunction,
                       FunctionGrouper)
from .task import AssociatedInput, AssociatedInputGroup, TaskBase, Task, DataSource
from .task import BaseClassificationRule, AssociationConfiguration, ActiveCondition, JobProcessingFunction
from .task import DynamicParameterProvider
from .task_details import ReportConfig, ReportInput, FilterMode, CommandType, CommandConfig
from .time_range import RelativeTimeRange, UNLIMITED
from .util import match
from ..executor.recipe import RecipeFunction


def create_assoc_config(match_function: Optional[MatchFunction], match_keywords: MatchKeywords, level: Optional[float],
                        index: int, time_range: RelativeTimeRange) -> AssociationConfiguration:
    return AssociationConfiguration(
        level=level if level is not None else index,
        match_function=match_function if match_function else lambda x, y: match(x, y, match_keywords),
        match_keywords=match_keywords, time_range=time_range
    )


# FIXME deprecated
class AssociatedInputBuilder:
    def __init__(self, input_task: Optional[TaskBase] = None):
        self.task = input_task
        self.classification_rules: List[BaseClassificationRule] = []
        self.min_ret = None
        self.max_ret = None
        self.condition: Optional[ActiveCondition] = None
        self.assoc_configs: List[AssociationConfiguration] = []
        self.sort_keys: List[str] = []

    def with_classification_rule(self, classification_rule: BaseClassificationRule) -> 'AssociatedInputBuilder':
        self.classification_rules.append(classification_rule)
        return self

    def with_min_ret(self, min_ret: int) -> 'AssociatedInputBuilder':
        self.min_ret = min_ret
        return self

    def with_max_ret(self, max_ret: int) -> 'AssociatedInputBuilder':
        self.max_ret = max_ret
        return self

    def with_condition(self, condition: ActiveCondition) -> 'AssociatedInputBuilder':
        self.condition = condition
        return self

    def with_match_function(self, match_function: MatchFunction, level: float = None,
                            time_range: RelativeTimeRange = UNLIMITED) -> 'AssociatedInputBuilder':
        config = create_assoc_config(match_function, [], level, len(self.assoc_configs), time_range)
        self.assoc_configs.append(config)
        return self

    def with_match_keywords(self, match_keywords: MatchKeywords, level: float = None,
                            time_range: RelativeTimeRange = UNLIMITED) -> 'AssociatedInputBuilder':
        config = create_assoc_config(None, match_keywords, level, len(self.assoc_configs), time_range)
        self.assoc_configs.append(config)
        return self

    def with_closest_by(self, sort_keys: List[str]) -> 'AssociatedInputBuilder':
        self.sort_keys = sort_keys
        return self

    # FIXME deprecated
    def build(self) -> AssociatedInput:
        min_ret = self.min_ret or 1
        max_ret = self.max_ret or 1
        condition = self.condition or TRUE_CONDITION
        return AssociatedInput(self.task, self.classification_rules, min_ret, max_ret, condition, self.assoc_configs,
                               self.sort_keys)


class MatchRules:
    def __init__(self):
        self.assoc_configs: List[AssociationConfiguration] = []

    def with_match_function(self, match_function: MatchFunction, level: float = None,
                            time_range: RelativeTimeRange = UNLIMITED) -> 'MatchRules':
        config = create_assoc_config(match_function, [], level, len(self.assoc_configs), time_range)
        self.assoc_configs.append(config)
        return self

    def with_match_keywords(self, match_keywords: MatchKeywords, level: float = None,
                            time_range: RelativeTimeRange = UNLIMITED) -> 'MatchRules':
        config = create_assoc_config(None, match_keywords, level, len(self.assoc_configs), time_range)
        self.assoc_configs.append(config)
        return self


def create_associated_input(input_task: TaskBase, classification_rules: List[BaseClassificationRule] = None,
                            min_ret: int = 1, max_ret: int = 1, condition: ActiveCondition = TRUE_CONDITION,
                            sort_keys: List[str] = None, match_rules: Optional[MatchRules] = None) -> AssociatedInput:
    assoc_configs = match_rules.assoc_configs if match_rules else None
    return AssociatedInput(input_task=input_task, classification_rules=classification_rules, min_ret=min_ret,
                           max_ret=max_ret, condition=condition, assoc_configs=assoc_configs, sort_keys=sort_keys)


class AssociatedInputGroupBuilder:
    def __init__(self, sort_keys: List[str] = None):
        self.associated_inputs: List[AssociatedInput] = []
        self.sort_keys = sort_keys or []

    def with_associated_input(self, input_task: TaskBase, classification_rules: List[BaseClassificationRule] = None,
                              min_ret: int = 1, max_ret: int = 1,
                              condition: ActiveCondition = TRUE_CONDITION,
                              sort_keys: List[str] = None,
                              match_rules: Optional[MatchRules] = None) -> 'AssociatedInputGroupBuilder':
        assoc_input = create_associated_input(input_task=input_task, classification_rules=classification_rules,
                                              min_ret=min_ret, max_ret=max_ret, condition=condition,
                                              sort_keys=sort_keys, match_rules=match_rules)
        self.associated_inputs.append(assoc_input)
        return self

    # FIXME deprecated
    def with_associated_input_builder(self, builder: AssociatedInputBuilder) -> 'AssociatedInputGroupBuilder':
        self.associated_inputs.append(builder.build())
        return self

    def build(self) -> AssociatedInputGroup:
        return AssociatedInputGroup(associated_inputs=self.associated_inputs, sort_keys=self.sort_keys)


class DataSourceBuilder:
    def __init__(self, name: str):
        self.name = name
        self.classification_rules: List[BaseClassificationRule] = []
        self.min_group_size: int = 1
        self.assoc_configs: List[AssociationConfiguration] = []
        self.grouping_keywords: List[str] = []
        self.groupers: List[Grouper] = []
        self.setup_keywords = []
        self.subworkflow_name = []

    def with_classification_rule(self, classification_rule: BaseClassificationRule) -> 'DataSourceBuilder':
        self.classification_rules.append(classification_rule)
        return self

    def with_grouping_keywords(self, grouping_keywords: List[str]) -> 'DataSourceBuilder':
        self.grouping_keywords = grouping_keywords
        return self.with_custom_grouping(KeywordGrouping(grouping_keywords))

    def with_setup_keywords(self, setup_keywords: List[str]) -> 'DataSourceBuilder':
        self.setup_keywords = setup_keywords
        return self

    def with_min_group_size(self, min_group_size: int) -> 'DataSourceBuilder':
        self.min_group_size = min_group_size
        return self

    def with_match_function(self, match_function: MatchFunction, level: float = None,
                            time_range: RelativeTimeRange = UNLIMITED) -> 'DataSourceBuilder':
        config = create_assoc_config(match_function, [], level, len(self.assoc_configs), time_range)
        self.assoc_configs.append(config)
        return self

    def with_match_keywords(self, match_keywords: MatchKeywords, level: float = None,
                            time_range: RelativeTimeRange = UNLIMITED) -> 'DataSourceBuilder':
        config = create_assoc_config(None, match_keywords, level, len(self.assoc_configs), time_range)
        self.assoc_configs.append(config)
        return self

    def with_match_rules(self, match_rules: MatchRules) -> 'DataSourceBuilder':
        self.assoc_configs.extend(match_rules.assoc_configs)
        return self

    def with_cluster(self, keyword: str, *, max_diameter: str = None,
                     max_separation: str = None) -> 'DataSourceBuilder':
        if keyword == 'SKY.POSITION':
            return self.with_custom_grouping(SkyPositionClusterGrouping(keyword, max_diameter, max_separation))
        else:
            return self.with_custom_grouping(SimpleClusterGrouping(keyword, max_diameter, max_separation))

    def with_grouping_function(self, function: GroupingFunction) -> 'DataSourceBuilder':
        return self.with_custom_grouping(FunctionGrouper(function))

    def with_custom_grouping(self, grouper: Grouper) -> 'DataSourceBuilder':
        self.groupers.append(grouper)
        return self

    def with_subworkflow_name(self, name: str) -> 'DataSourceBuilder':
        self.subworkflow_name.append(name)
        return self

    def build(self) -> 'DataSource':
        name = self.name or ", ".join([x.classification for x in self.classification_rules])
        return DataSource(classification_rules=self.classification_rules, grouping_keywords=self.grouping_keywords,
                          min_group_size=self.min_group_size, match_functions=self.assoc_configs,
                          groupers=self.groupers, name=name, setup_keywords=self.setup_keywords,
                          subworkflow_name=self.subworkflow_name)


class TaskBuilder:
    def __init__(self, name: str):
        self.name = name
        self.subworkflow_name: List[str] = []
        self.command_config: CommandConfig = CommandConfig(
            command=self.build_function_ref(empty),
            command_type=CommandType.FUNCTION,
            recipes=[]
        )
        self.main_input: Optional[TaskBase] = None
        self.associated_inputs: List[AssociatedInputGroup] = []
        self.condition: ActiveCondition = TRUE_CONDITION
        self.meta_targets: List[str] = []
        self.input_filter: List[str] = []
        self.input_filter_mode: FilterMode = FilterMode.SELECT
        self.output_filter: List[str] = []
        self.output_filter_mode: FilterMode = FilterMode.SELECT
        self.input_map: Dict[str, str] = {}
        self.grouping_keywords: List[str] = []
        self.groupers: List[Grouper] = []
        self.min_group_size = 1
        self.job_processing_function: Optional[JobProcessingFunction] = None
        self.dynamic_parameters: Dict[str, DynamicParameterProvider] = {}
        self.reports: List[ReportConfig] = []
        self.description: Optional[str] = None
        self.accepted_classification_rules: List[BaseClassificationRule] = []

    def with_recipe(self, recipe: str) -> 'TaskBuilder':
        self.command_config = CommandConfig(
            command=recipe,
            command_type=CommandType.RECIPE,
            recipes=[recipe]
        )
        return self

    def with_shell_command(self, command: str, recipes: List[str] = None) -> 'TaskBuilder':
        self.command_config = CommandConfig(
            command=command,
            command_type=CommandType.SHELL,
            recipes=recipes or []
        )
        return self

    def with_function(self, function: RecipeFunction, recipes: List[str] = None) -> 'TaskBuilder':
        self.command_config = CommandConfig(
            command=self.build_function_ref(function),
            command_type=CommandType.FUNCTION,
            recipes=recipes or []
        )
        return self

    def with_main_input(self, main_input: TaskBase,
                        accepted_classification_rules: Optional[List[BaseClassificationRule]] = None) -> 'TaskBuilder':
        self.main_input = main_input
        self.accepted_classification_rules = accepted_classification_rules or []
        self.main_input.add_classification_rules(self.accepted_classification_rules)
        return self

    def with_associated_input(self, input_task: TaskBase, classification_rules: List[BaseClassificationRule] = None,
                              min_ret: int = 1, max_ret: int = 1,
                              condition: ActiveCondition = TRUE_CONDITION,
                              sort_keys: List[str] = None,
                              match_rules: Optional[MatchRules] = None) -> 'TaskBuilder':
        assoc_input = create_associated_input(input_task=input_task, classification_rules=classification_rules,
                                              min_ret=min_ret, max_ret=max_ret, condition=condition,
                                              sort_keys=sort_keys, match_rules=match_rules)
        self.associated_inputs.append(AssociatedInputGroup(associated_inputs=[assoc_input]))
        return self

    # FIXME deprecated
    def with_associated_input_builder(self, builder: AssociatedInputBuilder) -> 'TaskBuilder':
        self.associated_inputs.append(AssociatedInputGroupBuilder().with_associated_input_builder(builder).build())
        return self

    def with_alternative_associated_inputs(self, builder: AssociatedInputGroupBuilder) -> 'TaskBuilder':
        self.associated_inputs.append(builder.build())
        return self

    # FIXME deprecated
    def with_alternatives(self, builder: AssociatedInputGroupBuilder) -> 'TaskBuilder':
        return self.with_alternative_associated_inputs(builder)

    def with_input_filter(self, *args: BaseClassificationRule, mode: FilterMode = FilterMode.SELECT):
        self.input_filter = [arg.classification for arg in args]
        self.input_filter_mode = mode
        return self

    def with_output_filter(self, *args: BaseClassificationRule, mode: FilterMode = FilterMode.SELECT):
        self.output_filter = [arg.classification for arg in args]
        self.output_filter_mode = mode
        return self

    def with_input_map(self, input_map: Dict[BaseClassificationRule, BaseClassificationRule]):
        self.input_map = {x.classification: y.classification for x, y in input_map.items()}
        return self

    def with_condition(self, condition: ActiveCondition) -> 'TaskBuilder':
        self.condition = condition
        return self

    def with_meta_targets(self, meta_targets: List[str]) -> 'TaskBuilder':
        self.meta_targets = meta_targets
        return self

    def with_inputs(self, inputs: List[Task], max_active: int = 1) -> 'TaskBuilder':
        # FIXME currently not used, this code needs fixing
        if len(inputs) == 1:
            return self.with_main_input(inputs[0])
        active = [inp for inp in inputs if inp.condition is True]
        num_active = len(active)
        if not 1 <= num_active <= max_active:
            msg = 'the number of active inputs must be between 1 and {} ({})'.format(max_active, num_active)
            raise RuntimeError(msg)
        self.main_input = active[0]
        for inp in set(inputs).difference({self.main_input}):
            self.associated_inputs.append(AssociatedInputGroupBuilder()
                                          .with_associated_input(inp).build())
        return self

    def with_grouping_keywords(self, grouping_keywords: List[str]) -> 'TaskBuilder':
        self.grouping_keywords = grouping_keywords
        return self.with_custom_grouping(KeywordGrouping(grouping_keywords))

    def with_cluster(self, keyword: str, *, max_diameter: str = None,
                     max_separation: str = None) -> 'TaskBuilder':
        if keyword == 'SKY.POSITION':
            return self.with_custom_grouping(SkyPositionClusterGrouping(keyword, max_diameter, max_separation))
        else:
            return self.with_custom_grouping(SimpleClusterGrouping(keyword, max_diameter, max_separation))

    def with_grouping_function(self, function: GroupingFunction) -> 'TaskBuilder':
        return self.with_custom_grouping(FunctionGrouper(function))

    def with_custom_grouping(self, grouper: Grouper) -> 'TaskBuilder':
        self.groupers.append(grouper)
        return self

    def with_min_group_size(self, min_group_size: int) -> 'TaskBuilder':
        self.min_group_size = min_group_size
        return self

    def with_job_processing(self, job_processing_function: JobProcessingFunction) -> 'TaskBuilder':
        self.job_processing_function = job_processing_function
        return self

    def with_dynamic_parameter(self, parameter: str, parameter_provider: DynamicParameterProvider) -> 'TaskBuilder':
        self.dynamic_parameters[parameter] = parameter_provider
        return self

    def with_report(self, name: str, inp: ReportInput, driver: str = 'png') -> 'TaskBuilder':
        self.reports.append(ReportConfig(name=name, input=inp, driver=driver))
        return self

    def with_subworkflow_name(self, name: str) -> 'TaskBuilder':
        self.subworkflow_name.append(name)
        return self

    def with_description(self, description: str) -> 'TaskBuilder':
        self.description = description
        return self

    @staticmethod
    def build_function_ref(function: RecipeFunction):
        return inspect.getmodule(function).__name__ + "." + function.__name__

    def build(self) -> 'Task':
        return Task(name=self.name, command_config=self.command_config, subworkflow_name=self.subworkflow_name,
                    main_input=self.main_input, associated_input_groups=self.associated_inputs,
                    condition=self.condition, meta_targets=self.meta_targets,
                    input_filter=self.input_filter, input_filter_mode=self.input_filter_mode,
                    output_filter=self.output_filter, output_filter_mode=self.output_filter_mode,
                    input_map=self.input_map, grouping_keywords=self.grouping_keywords, groupers=self.groupers,
                    min_group_size=self.min_group_size, job_editing_function=self.job_processing_function,
                    dynamic_parameters=self.dynamic_parameters, reports=self.reports, description=self.description,
                    accepted_classification_rules=self.accepted_classification_rules)


def task(name: str) -> TaskBuilder:
    return TaskBuilder(name)


def data_source(name: str = None) -> DataSourceBuilder:
    return DataSourceBuilder(name)


# FIXME deprecated
def alternative_association() -> AssociatedInputGroupBuilder:
    return AssociatedInputGroupBuilder()


def alternative_associated_inputs(sort_keys: List[str] = None) -> AssociatedInputGroupBuilder:
    return AssociatedInputGroupBuilder(sort_keys)


# FIXME deprecated
def associated_input(input_task: TaskBase) -> AssociatedInputBuilder:
    return AssociatedInputBuilder(input_task)


def match_rules() -> MatchRules:
    return MatchRules()
