from dataclasses import dataclass

from edps.client.WorkflowDTO import RelativeTimeRangeDTO
from .constants import INF, NEGATIVE_INF, MJD_OBS
from .fits import ClassifiedFitsFile


@dataclass
class RelativeTimeRange:
    left: float = NEGATIVE_INF
    right: float = INF

    def is_unlimited(self) -> bool:
        return self.left == NEGATIVE_INF and self.right == INF

    def is_within_range(self, ref: ClassifiedFitsFile, f: ClassifiedFitsFile) -> bool:
        return self.is_mjd_within_range(ref[MJD_OBS], f[MJD_OBS])

    def is_mjd_within_range(self, ref_mjd: float, f_mjd: float) -> bool:
        return True if self.is_unlimited() else ref_mjd + self.left <= f_mjd <= ref_mjd + self.right

    def as_dto(self) -> RelativeTimeRangeDTO:
        left = self.left if self.left != NEGATIVE_INF else None
        right = self.right if self.right != INF else None
        return RelativeTimeRangeDTO(left=left, right=right)

    def __repr__(self):
        return f"{self.left}, {self.right}"


ONE_AND_HALF_HOURS = RelativeTimeRange(-0.0625, 0.0625)
SAME_NIGHT = RelativeTimeRange(-0.4, 0.4)
NEXT_DAY = RelativeTimeRange(0, 1)
ONE_DAY = RelativeTimeRange(-1, 1)
TWO_DAYS = RelativeTimeRange(-2, 2)
THREE_DAYS = RelativeTimeRange(-3, 3)
FOUR_DAYS = RelativeTimeRange(-4, 4)
FIVE_DAYS = RelativeTimeRange(-5, 5)
ONE_WEEK = RelativeTimeRange(-7, 7)
TWO_WEEKS = RelativeTimeRange(-14, 14)
THREE_WEEKS = RelativeTimeRange(-21, 21)
ONE_MONTH = RelativeTimeRange(-30, 30)
QUARTERLY = RelativeTimeRange(-90, 90)
IN_THE_PAST = RelativeTimeRange(NEGATIVE_INF, 0)
UNLIMITED = RelativeTimeRange(NEGATIVE_INF, INF)
