import datetime
from dataclasses import dataclass, field
from typing import List, Dict, Callable, Tuple
from uuid import UUID

from edps.client.FitsFile import FitsFile
from edps.client.JobInfo import JobInfo
from edps.client.ProcessingJob import ProcessingJob, LogEntry, ReportEntry
from edps.client.ProcessingJobStatus import ProcessingJobStatus, JobStatus
from edps.client.search import SearchFilter
from edps.executor.filtering import ProductFilter


class JobNotFoundError(Exception):
    def __init__(self, message: str):
        self.message = message


@dataclass
class ExecutionResult:
    status: ProcessingJobStatus
    input_files: List[FitsFile] = field(default_factory=list)
    output_files: List[FitsFile] = field(default_factory=list)
    logs: List[LogEntry] = field(default_factory=list)
    reports: List[ReportEntry] = field(default_factory=list)
    recipe_parameters: Dict[str, str] = field(default_factory=dict)
    completion_date: str = datetime.datetime.max.isoformat()
    interrupted: bool = False

    def is_complete(self) -> bool:
        return self.status == ProcessingJobStatus.COMPLETED

    def is_failed(self) -> bool:
        return self.status == ProcessingJobStatus.FAILED

    def is_missing(self) -> bool:
        return self.status == ProcessingJobStatus.MISSING

    def is_pending(self) -> bool:
        return self.status in [ProcessingJobStatus.CREATED, ProcessingJobStatus.RUNNING]

    def is_interrupted(self) -> bool:
        return self.interrupted

    def __repr__(self):
        return f'{self.status}: {[f.model_dump_json() for f in self.output_files]}'


class JobDetails:
    def __init__(self, configuration: JobInfo, result: ExecutionResult, submission_date: str, rejected: bool = False):
        self.configuration = configuration
        self.result = result
        self.submission_date = submission_date
        self.rejected = rejected

    def __repr__(self):
        return self.to_processing_job().model_dump_json()

    def to_processing_job(self) -> ProcessingJob:
        return ProcessingJob(completion_date=self.result.completion_date,
                             submission_date=self.submission_date,
                             status=self.result.status,
                             input_files=self.result.input_files,
                             output_files=self.result.output_files,
                             reports=self.result.reports,
                             logs=self.result.logs,
                             recipe_parameters=self.result.recipe_parameters,
                             configuration=self.configuration,
                             rejected=self.rejected,
                             interrupted=self.is_interrupted())

    def to_job_status(self) -> JobStatus:
        return JobStatus(
            status=self.result.status,
            rejected=self.rejected,
            interrupted=self.is_interrupted()
        )

    def to_dict(self) -> Dict:
        return self.to_processing_job().model_dump()

    @staticmethod
    def from_dict(data: Dict, is_job_present: Callable[[str, List[FitsFile]], bool] = lambda x, y: True) -> 'JobDetails':
        info = JobInfo.from_dict(data['configuration'])
        status = ProcessingJobStatus(data['status'])
        outputs = [FitsFile.from_dict(output_file) for output_file in data['output_files']]
        return JobDetails(
            info,
            ExecutionResult(
                JobDetails.compute_effective_status(status, info.job_id, outputs, is_job_present),
                [FitsFile.from_dict(input_file) for input_file in data['input_files']],
                outputs,
                [LogEntry.from_dict(log) for log in data['logs']],
                [ReportEntry.from_dict(report) for report in data['reports']],
                data.get('recipe_parameters', {}),
                data['completion_date'],
                data.get('interrupted', False)
            ),
            data['submission_date'],
            data['rejected']
        )

    @staticmethod
    def compute_effective_status(status: ProcessingJobStatus, job_id: str, outputs: List[FitsFile],
                                 is_job_present: Callable[[str, List[FitsFile]], bool]) -> ProcessingJobStatus:
        if is_job_present(job_id, outputs) or status == ProcessingJobStatus.CREATED:
            return status
        else:
            return ProcessingJobStatus.MISSING

    def is_complete(self) -> bool:
        return self.result.is_complete()

    def is_failed(self) -> bool:
        return self.result.is_failed()

    def is_missing(self) -> bool:
        return self.result.is_missing()

    def is_missing_or_failed(self) -> bool:
        return self.is_missing() or self.is_failed()

    def is_pending(self) -> bool:
        return self.result.is_pending()

    def is_within_re_execution_window(self, reexecution_window: datetime.timedelta):
        return datetime.datetime.fromisoformat(self.submission_date) + reexecution_window > datetime.datetime.now()

    def is_outside_re_execution_window(self, reexecution_window: datetime.timedelta):
        return not self.is_within_re_execution_window(reexecution_window)

    def needs_re_execution(self, reexecution_window: datetime.timedelta) -> bool:
        return self.is_missing_or_failed() and not self.rejected and (self.is_within_re_execution_window(reexecution_window) or self.is_interrupted())

    def should_be_skipped(self, reexecution_window: datetime.timedelta) -> bool:
        return self.is_missing_or_failed() and (self.rejected or (self.is_outside_re_execution_window(reexecution_window) and self.is_not_interrupted()))

    def is_interrupted(self) -> bool:
        return self.result.is_interrupted()

    def is_not_interrupted(self) -> bool:
        return not self.is_interrupted()

    def update_status(self, is_job_present: Callable[[str, List[FitsFile]], bool]):
        self.result.status = self.compute_effective_status(self.result.status, self.configuration.job_id,
                                                           self.result.output_files, is_job_present)

    def clone(self) -> 'JobDetails':
        return self.from_dict(self.to_dict())


@dataclass
class RejectionResult:
    rejected_jobs: List[UUID]
    incomplete_jobs: List[UUID]
    files_to_resubmit: List[str]


class JobsRepository(object):

    def get_job_details(self, job_id: UUID) -> JobDetails:
        raise NotImplementedError

    def get_job_details_simple(self, job_id: UUID) -> JobDetails:
        raise NotImplementedError

    def find_complete_job(self, command: str, inputs: List[FitsFile], recipe_parameters: Dict,
                          input_filter: ProductFilter, output_filter: ProductFilter) -> UUID:
        raise NotImplementedError

    def find_all_pending_jobs(self) -> List[JobDetails]:
        raise NotImplementedError

    def find_all_non_rejected_jobs(self) -> List[JobDetails]:
        raise NotImplementedError

    def get_jobs_list(self, pattern, offset, limit) -> List[JobDetails]:
        raise NotImplementedError

    def set_job_status(self, job_id: UUID, result: ExecutionResult):
        raise NotImplementedError

    def set_jobs_submitted(self, jobs: List[JobInfo], submission_date: str):
        raise NotImplementedError

    def insert_job(self, job: JobDetails) -> UUID:
        raise NotImplementedError

    def get_jobs_list_filter(self, search_filter: SearchFilter) -> List[JobDetails]:
        raise NotImplementedError

    def reject_job(self, job_id: UUID) -> RejectionResult:
        raise NotImplementedError

    def find_related_jobs(self, job_id: UUID) -> Tuple[List[UUID], List[UUID]]:
        raise NotImplementedError

    def remove_jobs(self, job_ids: List[UUID]) -> List[UUID]:
        raise NotImplementedError

    def find_jobs_to_remove(self, cleanup_older_than: datetime.timedelta) -> List[UUID]:
        raise NotImplementedError

    def clear(self):
        raise NotImplementedError

    def close(self):
        raise NotImplementedError
