import threading
from typing import List, Dict, Callable, Optional, Tuple
from uuid import UUID

from edps.client.FitsFile import FitsFile
from edps.client.search import SearchFilter
from edps.config.configuration import Configuration
from edps.executor.filtering import ProductFilter
from edps.interfaces.JobsRepository import JobDetails
from edps.jobs.InMemoryJobsDB import InMemoryJobsDB
from edps.jobs.JobsDB import JobsDB, JobExistenceFunction
from edps.jobs.CommandQueueProcessor import CommandQueueProcessor
from edps.jobs.TinyJobsDB import TinyJobsDB


class CachingJobsDB(JobsDB):
    """
    Caching wrapper over some `JobsDB`.
    The general idea to have completely lock-free read and write with low-latency on top of persistent db.

    Persistent storage is handled by underlying `self.persistent_db`, quickly accessible current state is held in `self.memory_cache`.

    All persistent write operations are stored in a queue, and dedicated thread is processing them one by one. This ensures there is only one writer at a time.
    Each write operation consists of 2 steps -> updating in-memory caches and updating underlying persistent storage.

    To ensure consistency, action goes into db-writer-queue only once it's being processed by the memory cache.
    This way we can be certain that order of elements in both queues is the same, even though those actions can be consumed at a different rate.

    Depending on the cache setting the updates might block until changes are visible in the cache, or might have eventual consistency.

    In case we reach a situation when there are lots of writes, enabling TinyDB caching middleware can be useful to cut down I/O time,
    however the db writes are handled by separate thread, so they don't actually slow down reads or writes from the cache.
    """

    def __init__(self, config: Configuration):
        self.persistent_db: JobsDB = TinyJobsDB(config)
        self.memory_cache: InMemoryJobsDB = InMemoryJobsDB(self.persistent_db.get_all)
        self.async_persistent_writer = CommandQueueProcessor()
        threading.Thread(target=self.async_persistent_writer.run).start()
        self.lock = threading.RLock()

    def get_all(self) -> List[JobDetails]:
        return self.memory_cache.get_all()

    def get_job_details_simple(self, job_id: UUID,
                               is_job_present: JobExistenceFunction = lambda x, y: True) -> JobDetails:
        return self.memory_cache.get_job_details_simple(job_id, is_job_present)

    def get_jobs_list(self, pattern: str, offset: int, limit: int) -> List[JobDetails]:
        return self.memory_cache.get_jobs_list(pattern, offset, limit)

    def get_jobs_list_filter_with_limits(self, search_filter: SearchFilter,
                                         start: int, end: Optional[int]) -> List[JobDetails]:
        return self.memory_cache.get_jobs_list_filter_with_limits(search_filter, start, end)

    def find_complete_job(self, command: str, inputs: List[FitsFile], recipe_parameters: Dict,
                          input_filter: ProductFilter, output_filter: ProductFilter,
                          is_job_present: JobExistenceFunction = lambda x, y: True) -> UUID:
        return self.memory_cache.find_complete_job(command, inputs, recipe_parameters, input_filter, output_filter, is_job_present)

    def find_associated_jobs(self, parent_jobs: List[UUID]) -> List[UUID]:
        return self.memory_cache.find_associated_jobs(parent_jobs)

    def find_children_jobs(self, parent_jobs: List[UUID]) -> Tuple[List[UUID], List[UUID]]:
        return self.memory_cache.find_children_jobs(parent_jobs)

    def insert_multiple(self, data: List[JobDetails]):
        self.consistent_operation(lambda db: db.insert_multiple(data))

    def set_job_details(self, job_id: UUID, new_value: JobDetails):
        self.consistent_operation(lambda db: db.set_job_details(job_id, new_value))

    def remove_jobs(self, job_ids: List[UUID]):
        self.consistent_operation(lambda db: db.remove_jobs(job_ids))

    def consistent_operation(self, operation: Callable[[JobsDB], None]):
        with self.lock:
            operation(self.memory_cache)
            self.async_persistent_writer.schedule_action(lambda: operation(self.persistent_db))

    def clear(self):
        self.memory_cache.clear()
        self.persistent_db.clear()

    def close(self):
        self.memory_cache.close()
        self.async_persistent_writer.stop()
        self.persistent_db.close()
