import threading
from queue import Queue, Empty
from typing import Callable


class CommandQueueProcessor:
    """
    Class responsible for unloading operations queue.
    Only one thread should run for given commands_queue, because there is no locking involved.
    Once a signal to stop is received, it will continue to operate until the queue is empty, to ensure all operations are handled.
    """

    def __init__(self):
        self.stopped = False
        self.command_queue = Queue()

    def run(self):
        while not self.stopped:
            try:
                action = self.command_queue.get(True, timeout=1)
                action()
                self.command_queue.task_done()
            except Empty:
                pass
        try:
            while True:
                action = self.command_queue.get_nowait()
                action()
                self.command_queue.task_done()
        except Empty:
            pass

    def schedule_action(self, action: Callable):
        self.command_queue.put(action)

    def block_until_converged(self):
        event = threading.Event()
        self.command_queue.put(lambda: event.set())
        event.wait()

    def stop(self):
        self.stopped = True
        self.command_queue.join()
