import datetime
import logging
from typing import List, Iterable

import requests
from requests.auth import HTTPBasicAuth

from edps.metrics.meter import Meter


class Publisher:
    def publish(self, meters: Iterable[Meter]):
        raise NotImplementedError()


class NopPublisher:
    def publish(self, meters: Iterable[Meter]):
        pass


class ElasticPublisher(Publisher):
    def __init__(self, url: str, index_prefix: str, username: str = None, password: str = None):
        self.index_prefix = index_prefix
        self.url = url
        self.logger = logging.getLogger("ElasticPublisher")
        self.auth = None
        if username and password:
            self.auth = HTTPBasicAuth(username, password)

    def publish(self, meters: List[Meter]):
        index_name = self.index_prefix + "-" + datetime.datetime.now().strftime("%Y-%m")
        for meter in meters[::]:
            payload = meter.get_and_reset()
            endpoint = f"{self.url}/{index_name}/_doc"
            r = requests.post(endpoint, json=payload, auth=self.auth)
            self.logger.debug("Indexing %s resulted in %s %s", payload, r.status_code, r.content)
