import argparse
import configparser
import datetime
import logging
import logging.config
import os.path
import re
import shutil
import signal
import sys
import time
import uuid
from pathlib import Path
from typing import List, Optional, Dict
from uuid import UUID

import fastapi.responses
import uvicorn
import yaml
from fastapi import FastAPI, Request, status, Query
from fastapi.responses import JSONResponse
from fastapi.responses import PlainTextResponse

from edps import __version__, __banner__
from edps.EDPS import EDPS
from edps.client.BreakpointsStateDTO import BreakpointsStateDTO, LoopOnceRequestDTO, LoopRequestDTO, \
    BreakpointInspectResultDTO
from edps.client.CalselectorRequest import CalselectorRequest
from edps.client.EDPSClient import EDPSClient
from edps.client.FitsFile import FitsFile
from edps.client.FlatOrganization import DatasetsDTO
from edps.client.GraphType import GraphType
from edps.client.JobInfo import JobInfo
from edps.client.ParameterSetDTO import ParameterSetDTO
from edps.client.ProcessingJob import ProcessingJob
from edps.client.ProcessingJobStatus import JobStatus
from edps.client.ProcessingRequest import ProcessingRequest
from edps.client.ProcessingResponse import ProcessingResponse
from edps.client.Rejection import Rejection
from edps.client.RunReportsRequestDTO import RunReportsRequestDTO
from edps.client.WorkflowDTO import CalselectorJob
from edps.client.WorkflowDTO import WorkflowDTO
from edps.client.WorkflowStateDTO import WorkflowStateDTO
from edps.client.search import SearchFilter
from edps.config.configuration import Configuration, AppConfig
from edps.generator.constants import WorkflowName, WorkflowPath
from edps.generator.workflow_manager import WorkflowNotFoundError
from edps.interfaces.JobsRepository import JobNotFoundError
from edps.metrics.meter_registry import MeterRegistry
from edps.phase3.phase3 import Phase3Configuration
from edps.utils import JSONResponseWithoutValidation


async def on_shutdown():
    edps.shutdown()


server_logger: Optional[logging.Logger] = None
app = FastAPI(on_shutdown=[on_shutdown])
edps: Optional[EDPS] = None


class OctetStreamResponse(fastapi.responses.Response):
    media_type = "application/octet-stream"


@app.middleware("http")
async def log_requests(request: Request, call_next):
    request_id = uuid.uuid4()
    server_logger.debug(f"rid {request_id} start request path {request.url.path}")
    start_time = time.time()
    response = await call_next(request)
    process_time = (time.time() - start_time) * 1000
    formatted_process_time = '{0:.2f}'.format(process_time)
    server_logger.debug(f"rid {request_id} completed_in {formatted_process_time}ms status_code {response.status_code}")
    uri = "/" + str(request.url).replace(str(request.base_url), "")
    uri = uri[:uri.find("?")] if '?' in uri else uri
    for k, v in request.path_params.items():
        uri = uri.replace(v, f'{{{k}}}')
    meter = MeterRegistry.instance.get_meter("http_server_requests",
                                             {'method': request.method, 'status': response.status_code, 'uri': uri})
    meter.record(process_time)
    return response


@app.exception_handler(FileNotFoundError)
async def file_not_found_exception_handler(request: Request, exc: FileNotFoundError):
    server_logger.warning(str(exc), exc_info=exc)
    return JSONResponse(
        status_code=status.HTTP_404_NOT_FOUND,
        content={"message": str(exc)},
    )


@app.exception_handler(JobNotFoundError)
async def job_not_found_exception_handler(request: Request, exc: JobNotFoundError):
    server_logger.debug(str(exc))
    return JSONResponse(
        status_code=status.HTTP_404_NOT_FOUND,
        content={"message": str(exc)},
    )


@app.exception_handler(WorkflowNotFoundError)
async def workflow_not_found_exception_handler(request: Request, exc: WorkflowNotFoundError):
    server_logger.debug(str(exc))
    return JSONResponse(
        status_code=status.HTTP_400_BAD_REQUEST,
        content={"message": exc.message},
    )


@app.exception_handler(Exception)
async def global_exception_handler(request: Request, exc: Exception):
    server_logger.error(str(exc), exc_info=exc)
    return JSONResponse(
        status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
        content={"message": "Internal Server Error " + str(exc)},
    )


@app.get(EDPSClient.JOBS_PATH, response_model=List[ProcessingJob])
def search_jobs(search: str = '.*', skip: int = 0, limit: int = 50) -> JSONResponse:
    return JSONResponseWithoutValidation(edps.get_jobs(search, skip, limit))


@app.get(EDPSClient.JOBS_FILTER_PATH, response_model=List[ProcessingJob])
def filter_jobs(
        instrument: str,
        targets: Optional[List[str]] = Query(default=None),
        meta_targets: Optional[List[str]] = Query(default=None),
        completion_time_from: datetime.datetime = datetime.datetime.min,
        completion_time_to: datetime.datetime = datetime.datetime.max,
        mjdobs_from: float = 0,
        mjdobs_to: float = sys.float_info.max,
        submission_time_from: datetime.datetime = datetime.datetime.min,
        submission_time_to: datetime.datetime = datetime.datetime.max
) -> JSONResponse:
    search_filter = SearchFilter(instrument=instrument,
                                 targets=targets or [],
                                 meta_targets=meta_targets or [],
                                 completion_time_from=completion_time_from,
                                 completion_time_to=completion_time_to,
                                 mjdobs_from=mjdobs_from,
                                 mjdobs_to=mjdobs_to,
                                 submission_time_from=submission_time_from,
                                 submission_time_to=submission_time_to)
    return JSONResponseWithoutValidation(edps.get_jobs_filter(search_filter))


@app.get(EDPSClient.SCHEDULED_JOBS_PATH, response_model=List[UUID])
def get_scheduled_jobs() -> List[UUID]:
    return edps.get_scheduled_jobs()


@app.post(EDPSClient.REQUESTS_PATH, status_code=status.HTTP_202_ACCEPTED, response_model=ProcessingResponse)
def process_data(request: ProcessingRequest) -> JSONResponse:
    return JSONResponseWithoutValidation(edps.process_data(request))


@app.post(EDPSClient.CLASSIFY_PATH, response_model=List[FitsFile])
def classify_files(request: ProcessingRequest) -> List[FitsFile]:
    return edps.classify_files(request)


@app.post(EDPSClient.CALSELECTOR_PATH, response_model=List[CalselectorJob])
def create_calselector_jobs(request: CalselectorRequest) -> JSONResponse:
    return JSONResponseWithoutValidation(edps.create_calselector_jobs(request))


@app.post(EDPSClient.ORGANISE_PATH, response_model=List[JobInfo])
def organise_data(request: ProcessingRequest) -> JSONResponse:
    return JSONResponseWithoutValidation(edps.organise_data(request))


@app.post(EDPSClient.ORGANISE_FLAT_PATH, response_model=DatasetsDTO)
def organise_data_flatten(request: ProcessingRequest) -> JSONResponse:
    return JSONResponseWithoutValidation(edps.organise_data_flatten(request))


@app.get(EDPSClient.REPORT_PATH, response_class=OctetStreamResponse)
def get_job_report(job_id: UUID, report_name: str) -> fastapi.responses.Response:
    content_type, content = edps.get_job_report(job_id, report_name)
    return fastapi.responses.Response(
        content=content,
        media_type=content_type,
    )


@app.get(EDPSClient.LOG_PATH, response_class=PlainTextResponse, response_model=str)
def get_job_log(job_id: UUID, log_name: str) -> bytes:
    return edps.get_job_log(job_id, log_name)


@app.get(EDPSClient.JOB_DETAILS_PATH, response_model=ProcessingJob)
def get_job_details(job_id: UUID) -> JSONResponse:
    return JSONResponseWithoutValidation(edps.get_job_details(job_id))


@app.get(EDPSClient.JOB_STATUS_PATH, response_model=JobStatus)
def get_job_status(job_id: UUID) -> JSONResponse:
    return JSONResponseWithoutValidation(edps.get_job_status(job_id))


@app.get(EDPSClient.WORKFLOWS_PATH, response_model=List[str])
def list_workflows() -> List[str]:
    return edps.list_workflows()


@app.get(EDPSClient.WORKFLOW_PATH, response_model=WorkflowDTO)
def get_workflow(workflow_name: str) -> JSONResponse:
    return JSONResponseWithoutValidation(edps.get_workflow(workflow_name))


@app.get(EDPSClient.WORKFLOW_DIR_PATH, response_class=PlainTextResponse, response_model=str)
def get_workflow_dir(workflow_name: str) -> str:
    if workflow_name not in edps.workflows:
        raise WorkflowNotFoundError(f"Workflow {workflow_name} not found")
    return edps.workflows.get(workflow_name)


@app.get(EDPSClient.WORKFLOW_STATE_PATH, response_model=WorkflowStateDTO)
def get_workflow_state(workflow_name: str) -> WorkflowStateDTO:
    return edps.get_workflow_state(workflow_name)


@app.get(EDPSClient.WORKFLOW_JOBS_PATH, response_model=List[JobInfo])
def get_workflow_jobs(workflow_name: str) -> List[JobInfo]:
    return edps.get_workflow_jobs(workflow_name)


@app.get(EDPSClient.WORKFLOW_JOB_PATH, response_model=JobInfo)
def get_workflow_job(workflow_name: str, job_id: str) -> JobInfo:
    return edps.get_workflow_job(workflow_name, job_id)


@app.get(EDPSClient.GRAPH_PATH, response_class=PlainTextResponse, response_model=str)
def get_graph(workflow_name: str, graph_type: GraphType = GraphType.SIMPLE) -> str:
    return edps.get_graph(workflow_name, graph_type)


@app.get(EDPSClient.ASSOC_MAP_PATH, response_class=PlainTextResponse, response_model=str)
def get_assoc_map(workflow_name: str) -> str:
    return edps.get_assoc_map(workflow_name)


@app.post(EDPSClient.RESET_PATH, response_model=Optional[str])
def reset_workflow(workflow_name: str):
    edps.reset_workflow(workflow_name)


@app.get(EDPSClient.TARGETS_PATH, response_class=PlainTextResponse, response_model=str)
def get_targets(workflow_name: str, targets: str, meta_targets: str) -> str:
    return edps.get_targets(workflow_name, targets.split(","), meta_targets.split(","))


@app.get(EDPSClient.PARAMETER_SETS_PATH, response_model=List[ParameterSetDTO])
def get_parameter_sets(workflow_name: str) -> JSONResponseWithoutValidation:
    return JSONResponseWithoutValidation(edps.get_parameter_sets(workflow_name))


@app.get(EDPSClient.DEFAULT_PARAMS_PATH, response_model=Dict[str, str])
def get_default_params(workflow_name: str, task_name: str) -> JSONResponseWithoutValidation:
    return JSONResponseWithoutValidation(edps.get_default_params(workflow_name, task_name))


@app.get(EDPSClient.RECIPE_PARAMS_PATH, response_model=Dict[str, str])
def get_recipe_params(workflow_name: str, task_name: str,
                      parameter_set: str = Query(default=None)) -> JSONResponseWithoutValidation:
    return JSONResponseWithoutValidation(edps.get_recipe_params(workflow_name, task_name, parameter_set))


@app.post(EDPSClient.REJECT_JOB_PATH, response_model=Rejection)
def reject_job(job_id: UUID) -> Rejection:
    return edps.reject_job(job_id)


@app.post(EDPSClient.DELETE_JOB_PATH, response_model=List[UUID])
def delete_job(job_id: UUID) -> List[UUID]:
    return edps.delete_job_cascade(job_id)


@app.post(EDPSClient.DELETE_JOBS_PATH, response_model=List[UUID])
def delete_independent_jobs_subset(job_ids: List[UUID]) -> List[UUID]:
    return edps.delete_independent_jobs_subset(job_ids)


@app.post(EDPSClient.RESUBMIT_JOB_PATH, response_model=UUID)
def resubmit_job(job_id: UUID) -> UUID:
    return edps.resubmit_job_with_id(job_id)


@app.get(EDPSClient.ASSOC_REPORT_PATH, response_class=PlainTextResponse, response_model=str)
def get_association_report(job_id: UUID) -> str:
    return edps.get_association_report(job_id)


@app.post(EDPSClient.UPDATE_CALIB_PATH, response_model=Optional[str])
def update_calib_db(workflow_name: str):
    edps.load_calibrations(workflow_name)


@app.post(EDPSClient.LOOP_EXECUTION, response_model=UUID)
def loop_execution(request: LoopRequestDTO) -> UUID:
    return edps.breakpoint_manager.loop_execution(UUID(request.job_id), request.inputs, request.parameters)


@app.post(EDPSClient.LOOP_EXECUTION_ONCE, response_model=Optional[str])
def loop_once(request: LoopOnceRequestDTO):
    edps.breakpoint_manager.loop_execution_once([UUID(job_id) for job_id in request.job_ids], request.parameters)


@app.post(EDPSClient.CREATE_BREAKPOINT, response_model=Optional[str])
def create_breakpoint(job_id: UUID):
    edps.breakpoint_manager.create_breakpoint(job_id)


@app.delete(EDPSClient.REMOVE_BREAKPOINT, response_model=Optional[str])
def remove_breakpoint(job_id: UUID):
    edps.breakpoint_manager.remove_breakpoint(job_id)


@app.post(EDPSClient.CONTINUE_EXECUTION, response_model=UUID)
def continue_execution(job_id: UUID) -> UUID:
    return edps.breakpoint_manager.continue_execution(job_id)


@app.get(EDPSClient.INSPECT_BREAKPOINT, response_model=BreakpointInspectResultDTO)
def inspect_execution(job_id: UUID) -> Optional[BreakpointInspectResultDTO]:
    return edps.breakpoint_manager.get_results(job_id)


@app.get(EDPSClient.BREAKPOINTS_PATH, response_model=BreakpointsStateDTO)
def list_breakpoints_status() -> BreakpointsStateDTO:
    return edps.breakpoint_manager.get_breakpoints_state()


@app.get(EDPSClient.PIPELINES_PATH, response_model=List[str])
def list_pipelines() -> List[str]:
    return edps.list_pipelines()


@app.post(EDPSClient.STOP_PROCESSING_PATH, response_model=str)
def stop_processing() -> str:
    edps.halt_executions()
    return "0"


@app.post(EDPSClient.SHUTDOWN_PATH, response_model=str)
def shutdown() -> str:
    signal.raise_signal(signal.SIGTERM)
    return "0"


@app.get(EDPSClient.VERSION_PATH, response_model=str)
def version() -> str:
    return __version__


@app.post(EDPSClient.REPORT_EXECUTION_PATH, response_model=List[str])
def run_reports(request: RunReportsRequestDTO) -> List[str]:
    return edps.run_reports(request)


@app.post(EDPSClient.PHASE3_PACKAGE_PATH, response_model=List[str])
def package_phase3(request: Phase3Configuration) -> List[str]:
    return edps.package_phase3(request)


def load_logging_config(config: str) -> Dict:
    with open(config, 'rb') as file:
        return yaml.safe_load(file.read())


def setup_logging(config: Dict):
    global server_logger
    logging.config.dictConfig(config)
    server_logger = logging.getLogger("ServerController")


def load_config(config_file: str, overrides: Dict[str, Dict[str, str]]) -> Configuration:
    config = configparser.ConfigParser()
    config.read(config_file)
    for section, params in overrides.items():
        for option, value in params.items():
            config.set(section, option, value)
    return Configuration(config)


def setup_edps(config_file: str, logging_file: str, overrides: Dict[str, Dict[str, str]] = None) -> EDPS:
    logging_config = load_logging_config(logging_file)
    config = load_config(config_file, overrides or {})
    return setup_edps_from_dict(config, logging_config)


def setup_edps_from_dict(config: Configuration, logging_config: Dict) -> EDPS:
    global edps
    if edps is not None:
        edps.shutdown()
    setup_logging(logging_config)
    logging.info(__banner__)
    logging.info("Starting EDPS version %s", __version__)
    config.log()
    esorex_path = Path(shutil.which(config.esorex_path) or config.esorex_path)
    workflow_dir = config.workflow_dir or esorex_path.parent.parent / 'share/esopipes/workflows'
    workflows = workflow_autodiscovery(str(workflow_dir))
    sys.path.extend({str(Path(path).parent) for path in workflows.values()})
    edps = EDPS(config, workflows)
    return edps


def workflow_autodiscovery(workflows_path: str) -> Dict[WorkflowName, WorkflowPath]:
    workflows_paths = [os.path.abspath(p.strip()) for p in workflows_path.split(",")] if workflows_path.strip() else []
    server_logger.info("Workflows directories resolved as %s", workflows_paths)
    workflows = {}
    for workflows_path in workflows_paths:
        for (dirpath, dirnames, filenames) in os.walk(workflows_path, followlinks=True):
            package = os.path.basename(dirpath)
            for filename in filenames:
                if re.match(rf'{package}.*_wkf.py$', filename):
                    server_logger.info("Found workflow %s/%s", dirpath, filename)
                    workflows[f'{package}.{Path(filename).stem}'] = dirpath
    return workflows


def main():
    parser = argparse.ArgumentParser(description=f"EDPS server version {__version__}")
    parser.add_argument('-c', '--application-config', required=False, default=AppConfig.APPLICATION_CONFIG)
    parser.add_argument('-l', '--logging-config', required=False, default=AppConfig.LOGGING_CONFIG)
    args = parser.parse_args()
    config = AppConfig(args.application_config, args.logging_config)
    if not config.exists():
        print(f"FATAL: Could not find EDPS configuration files: {config.application_config} {config.logging_config}")
        sys.exit(1)
    edps = setup_edps(config.application_config, config.logging_config)
    uvicorn.run(app, port=edps.configuration.port, host=edps.configuration.host)


if __name__ == "__main__":
    main()
