import os
import random
import shutil
import threading
import unittest
import uuid
from time import sleep
from typing import Dict

import uvicorn.server

from edps.EDPS import EDPS
from edps.client.BreakpointsStateDTO import BreakpointedJobState
from edps.client.EDPSClient import EDPSClient
from edps.client.JobSummary import JobSummary
from edps.client.ProcessingJobStatus import ProcessingJobStatus
from edps.config.configuration import Configuration
from edps.scripts.server import setup_edps
from edps.test.dsl.configuration import TestHelper


class BaseIT(unittest.TestCase):
    app_config: Configuration = None
    server: uvicorn.server.Server = None
    base_dir: str = None
    package_base_dir: str = None

    def __init__(self, args):
        super().__init__(args)

    def setUp(self) -> None:
        from edps.scripts.server import edps
        self.reconfigure_edps()
        self.client.reset_workflow("all")
        edps.repository.clear()
        shutil.rmtree(BaseIT.base_dir)
        shutil.rmtree(BaseIT.package_base_dir)
        os.makedirs(BaseIT.base_dir)
        os.makedirs(BaseIT.package_base_dir)

    @classmethod
    def setUpClass(cls):
        from edps.scripts.server import app
        from uvicorn.config import Config
        from uvicorn import Server
        if os.getcwd() not in os.environ["PATH"]:
            os.environ["PATH"] += os.pathsep + os.getcwd()
        edps = setup_edps("application-test.properties", "logging-test.yaml")
        BaseIT.app_config = edps.configuration
        BaseIT.base_dir = os.path.join(BaseIT.app_config.base_dir, str(uuid.uuid4()))
        BaseIT.package_base_dir = os.path.join(BaseIT.app_config.base_dir, str(uuid.uuid4()))
        os.makedirs(BaseIT.base_dir)
        os.makedirs(BaseIT.package_base_dir)
        uvicorn_config = Config(app, port=BaseIT.app_config.port, host=BaseIT.app_config.host)
        BaseIT.server = Server(config=uvicorn_config)
        threading.Thread(target=BaseIT.server.run).start()
        sleep(1)

    @classmethod
    def tearDownClass(cls):
        if isinstance(BaseIT.server.should_exit, bool):
            BaseIT.server.should_exit = True
        else:
            BaseIT.server.should_exit.set()
        # For CI, giving it sometime to properly close server
        sleep(5)
        shutil.rmtree(BaseIT.package_base_dir)
        shutil.rmtree(BaseIT.base_dir)

    @staticmethod
    def wait_until(condition, timeout):
        ticks = 0
        while not (condition() or (ticks > timeout)):
            sleep(1)
            ticks += 1
        if not condition():
            raise RuntimeError("Timeout reached without condition fulfilled")

    @staticmethod
    def remove_job(job: JobSummary):
        job_dir = os.path.join(BaseIT.base_dir, job.instrument, job.task_name, job.job_id)
        shutil.rmtree(job_dir)

    @staticmethod
    def extract_job_ids(links):
        return [link[link.rfind("/") + 1:] for link in links]

    def job_complete(self, job_summary: JobSummary) -> bool:
        status = self.client.get_job_status(job_summary.job_id)
        return status.is_right() and status.get().status in (ProcessingJobStatus.COMPLETED,
                                                             ProcessingJobStatus.FAILED)

    def job_complete_successfully(self, job_summary: JobSummary) -> bool:
        status = self.client.get_job_status(job_summary.job_id)
        return status.is_right() and status.get().status in ProcessingJobStatus.COMPLETED

    def job_halted(self, count: int) -> bool:
        breakpoints = self.client.list_breakpoints_state()
        return breakpoints.is_right() and len([brk for brk in breakpoints.get().breakpoints if
                                               brk.state == BreakpointedJobState.STOPPED_ON_BREAKPOINT]) == count

    @staticmethod
    def get_random_mjd_obs() -> float:
        return 5e5 + random.random() * 1e4

    def reconfigure_edps(self, config_file: str = "application-test.properties",
                         logging_file: str = "logging-test.yaml", overrides: Dict[str, Dict[str, str]] = None) -> EDPS:
        overrides = overrides or {}
        if 'executor' not in overrides:
            overrides['executor'] = {}
        overrides['executor'].update({'base_dir': BaseIT.base_dir})
        if 'packager' not in overrides:
            overrides['packager'] = {}
        if 'package_base_dir' not in overrides['packager']:
            overrides['packager'].update({'package_base_dir': BaseIT.package_base_dir})
        edps = setup_edps(config_file, logging_file, overrides)
        BaseIT.app_config = edps.configuration
        from edps.scripts.server import edps
        self.client = EDPSClient(BaseIT.app_config.host, BaseIT.app_config.port)
        self.helper = TestHelper(BaseIT.app_config.test_tmpdir, BaseIT.app_config.workflow_dir,
                                 BaseIT.app_config.test_calibdir, BaseIT.base_dir, BaseIT.package_base_dir, edps)
        return edps
