import configparser
import json
import logging
import os
import re
from typing import List, Dict, Tuple

from edps.config.configuration import Configuration
from edps.generator.association_preference import AssociationPreference
from edps.test.base import BaseIT
from edps.test.dsl.configuration import TestTemplate


def get_workflow_dir(properties: str) -> str:
    config_parser = configparser.ConfigParser()
    config_parser.read(properties)
    config = Configuration(config_parser)
    return config.workflow_dir


def list_configs(properties:str) -> List[str]:
    found_configs = []
    for workflow_directory in get_workflow_dir(properties).split(","):
        for (dirpath, dirnames, filenames) in os.walk(workflow_directory):
            for filename in filenames:
                if filename.endswith(".json"):
                    found_file = os.path.join(dirpath, filename)
                    found_configs.append(found_file)
    logging.info('Found json configs: "%s"', found_configs)
    return found_configs


def merge_keywords(template: Dict, common_keywords: Dict) -> Dict:
    result = template.copy()
    keywords = {}
    for name in template.get('common_keywords', []):
        if name in common_keywords:
            keywords.update(common_keywords[name])
        else:
            logging.warning('common keywords %s used in input %s not found', name, template['name_prefix'])
    keywords.update(template['keywords'])
    result['keywords'] = keywords
    return result


def load_scenarios(config: str) -> Tuple[List[Dict], Dict[str, Dict]]:
    match_pattern = os.environ.get('JSON_TEST_MATCH_PATTERN', '.*')
    with open(config, 'rb') as input_file:
        try:
            configuration = json.load(input_file)
            scenarios = [scenario for scenario in configuration.get('scenarios', []) if
                         scenario.get('skip', False) is False and re.match(match_pattern, scenario['description'])]
            common_keywords = configuration.get('keywords', {})
            common_inputs = {template['name_prefix']: merge_keywords(template, common_keywords) for template in
                             configuration.get('inputs', [])}
            print(config, scenarios)
            return scenarios, common_inputs
        except Exception as e:
            logging.warning('failed to load %s as JSON test: %s', config, e, exc_info=e)
            return [], {}


def create_templates(scenario: Dict, common_inputs: Dict[str, Dict]) -> List[TestTemplate]:
    mjd_obs = BaseIT.get_random_mjd_obs()
    result = []
    for common_input in scenario.get('common_inputs', []):
        if common_input in common_inputs:
            count, tpl = create_template(mjd_obs, common_inputs[common_input])
            mjd_obs -= count * 0.02
            result.append(tpl)
        else:
            logging.warning(
                "Test scenario '%s' declares to use common input '%s' but the only configured common inputs are %s",
                scenario['description'], common_input, common_inputs.keys())

    for template in scenario.get('input_files', []):
        count, tpl = create_template(mjd_obs, template)
        mjd_obs -= count * 0.02
        result.append(tpl)
    return result


def create_template(mjd_obs, template) -> Tuple[int, TestTemplate]:
    count = template["count"] if "count" in template else 1
    tpl = TestTemplate.builder().with_template_files(template['name_prefix'], template["keywords"], mjd_obs,
                                                     count).build()
    return count, tpl


class JSONMetaTest(type):
    def __new__(mcs, name, bases, attrs, config_dir : str = "."):
        def gen_func(scenario: Dict, common_inputs: Dict[str, Dict]):
            def func(self):
                with(self.helper.create_configuration()
                        .with_templates(create_templates(scenario, common_inputs))
                        .build()) as configuration:
                    request = self.helper.create_request(configuration, scenario['targets'], scenario['meta_targets'],
                                                         scenario['workflow'],
                                                         scenario.get('workflow_parameters', None),
                                                         scenario.get('workflow_parameter_set', None),
                                                         association_preference=scenario.get('association_preference', AssociationPreference.MASTER))
                    result = self.client.submit_to_organise(request)
                    expected = [(s['recipe'], s['inputs_prefixes'], s.get('assoc_prefixes', [])) for s in
                                scenario['results']]
                    (
                        self.helper.assert_on_organization(self, result)
                            .is_success()
                            .has_complete_jobs(len(expected))
                            .has_jobs_matching(expected)
                    )

            return func
        properties = os.path.join(config_dir, 'application-test.properties')
        for config in list_configs(properties):
            scenarios, inputs = load_scenarios(config)
            for test_scenario in scenarios:
                description = test_scenario['description']
                test_name = "test_" + os.path.basename(config).replace(".", "_") + "-" + description.replace(' ', '_')
                attrs[test_name] = gen_func(test_scenario, inputs)

        return super(JSONMetaTest, mcs).__new__(mcs, name, bases, attrs)
