import json
import logging
import os
import re
from typing import List, Dict

from edps.test.dsl.configuration import TestTemplate
from edps.test.base import BaseIT


class JSONTestsBase(BaseIT):
    def __init__(self, args):
        super(JSONTestsBase, self).__init__(args)
        self.match_pattern = os.environ.get('JSON_TEST_MATCH_PATTERN', '.*')
        self.logger = logging.getLogger('JSONTest')

    def synthetic_json_organization(self):
        for file in self.list_configs():
            with open(file, 'rb') as input_file:
                configuration = json.load(input_file)
                for scenario in configuration['scenarios']:
                    description = scenario['description']
                    skip = scenario.get('skip', False) or re.match(self.match_pattern, description) is None
                    if skip:
                        self.logger.info('skipping test scenario "%s"', description)
                        continue
                    self.logger.info('executing test scenario "%s"', description)
                    self.setUp()
                    with self.subTest(description):
                        # given
                        with(self.helper.create_configuration()
                                .with_templates(self.create_templates(scenario))
                                .build()) as configuration:
                            request = self.helper.create_request(configuration, scenario['targets'],
                                                                 scenario['meta_targets'], scenario['workflow'],
                                                                 scenario.get('workflow_parameters', None),
                                                                 scenario.get('workflow_parameter_set', None))

                            # when
                            result = self.client.submit_to_organise(request)
                            expected = [(s['recipe'], s['inputs_prefixes'], s.get('assoc_prefixes', [])) for s in
                                        scenario['results']]

                            # then
                            (
                                self.helper.assert_on_organization(self, result)
                                .is_success()
                                .has_complete_jobs(len(expected))
                                .has_jobs_matching(expected)
                            )

    def list_configs(self) -> List[str]:
        found_configs = []
        for (dirpath, dirnames, filenames) in os.walk(self.helper.workflow_dir):
            for filename in filenames:
                if filename.endswith(".json"):
                    found_file = os.path.join(dirpath, filename)
                    found_configs.append(found_file)
        self.logger.info('Found json configs: "%s"', found_configs)
        return found_configs

    def create_templates(self, scenario: Dict) -> List[TestTemplate]:
        result = []
        mjd_obs = self.get_random_mjd_obs()
        for f in scenario['input_files']:
            count = f["count"] if "count" in f else 1
            tpl = TestTemplate.builder().with_template_files(f['name_prefix'], f["keywords"], mjd_obs, count).build()
            mjd_obs -= count * 0.02
            result.append(tpl)
        return result
