from pathlib import Path

import bokeh
import hvplot
import panel as pn
import panel_material_ui as pmui
import sys
from edps import __version__ as edps_version
from panel.viewable import Viewer

from edpsgui import __version__ as edpsgui_version
from edpsgui.gui.classifier import Classifier
from edpsgui.gui.dataset_creator import DatasetCreator
from edpsgui.gui.edps_ctl import EDPSControl
from edpsgui.gui.input_selector import InputSelector
from edpsgui.gui.job_viewer import JobViewer
from edpsgui.gui.parameter_selector import ParameterSelector
from edpsgui.gui.reduction_monitor import ReductionMonitor
from edpsgui.gui.reduction_table import ReductionQueue, ReductionArchive
from edpsgui.gui.settings_editor import SettingsEditor
from edpsgui.gui.target_selector import TargetSelector
from edpsgui.gui.tooltips import (
    SELECT_INPUT_DATA, INSPECT_INPUT_DATA, SELECT_TARGET, CREATE_DATASETS, WORKFLOW_PARAMETERS
)
from edpsgui.gui.tooltips import TOPBAR_TOOLTIP
from edpsgui.gui.workflow import Workflow
from edpsgui.gui.workflow_selector import WorkflowSelector
from edpsplot import __version__ as edpsplot_version

pn.extension('tabulator', 'terminal', 'codeeditor', 'modal',
             css_files=["https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.2/css/all.min.css"],
             disconnect_notification='Connection lost, try reloading the page!',
             ready_notification='Application fully loaded.',
             notifications=True)

pn.state.notifications.position = 'top-center'
# Initialize the cache for tracking running report generations
pn.state.cache['running_report_job_ids'] = set()


class EDPSDashboard(Viewer):

    def __panel__(self):
        input_selector = InputSelector()
        edps_ctl = EDPSControl()
        workflow_selector = WorkflowSelector(edps_status=edps_ctl.param.edps_status)
        workflow_graph = Workflow(workflow=workflow_selector.param.workflow)
        target_selector = TargetSelector(workflow=workflow_selector.param.workflow)
        parameter_selector = ParameterSelector(workflow=workflow_selector.param.workflow)
        dataset_creator = DatasetCreator(edps_status=edps_ctl.param.edps_status,
                                         workflow=workflow_selector.param.workflow,
                                         inputs=input_selector.param.inputs,
                                         targets=target_selector.param.targets,
                                         parameter_set=parameter_selector.param.parameter_set,
                                         workflow_parameters=parameter_selector.param.custom_workflow_parameters)
        classifier = Classifier(edps_status=edps_ctl.param.edps_status,
                                workflow=workflow_selector.param.workflow,
                                inputs=input_selector.param.inputs)
        reduction_table = ReductionQueue(edps_status=edps_ctl.param.edps_status,
                                         workflow=workflow_selector.param.workflow)
        reduction_monitor = ReductionMonitor(edps_status=edps_ctl.param.edps_status,
                                             workflow=workflow_selector.param.workflow)
        reduction_archive = ReductionArchive(edps_status=edps_ctl.param.edps_status,
                                             workflow=workflow_selector.param.workflow)

        select_data_tooltip = pn.widgets.TooltipIcon(value=SELECT_INPUT_DATA)
        inspect_data_tooltip = pn.widgets.TooltipIcon(value=INSPECT_INPUT_DATA)
        select_target_tooltip = pn.widgets.TooltipIcon(value=SELECT_TARGET)
        select_parameters_tooltip = pn.widgets.TooltipIcon(value=WORKFLOW_PARAMETERS)
        create_datasets_tooltip = pn.widgets.TooltipIcon(value=CREATE_DATASETS)

        select_data_text = pmui.Typography('1. Select the input data', variant='h5')
        inspect_data_text = pmui.Typography('2. Inspect the input data (optional)', variant='h5')
        select_target_text = pmui.Typography('3. Select the reduction target', variant='h5')
        select_parameters_text = pmui.Typography('4. Select the workflow parameters', variant='h5')
        create_datasets_text = pmui.Typography('5. Create datasets', variant='h5')

        data_explorer = pn.Column(
            pn.Row(select_data_text, select_data_tooltip),
            input_selector,
            pn.layout.Divider(),
            pn.Row(inspect_data_text, inspect_data_tooltip),
            classifier,
            pn.layout.Divider(),
            pn.Row(select_target_text, select_target_tooltip),
            target_selector,
            pn.layout.Divider(),
            pn.Row(select_parameters_text, select_parameters_tooltip),
            parameter_selector,
            pn.layout.Divider(),
            pn.Row(create_datasets_text, create_datasets_tooltip),
            dataset_creator,
        )

        python_version = f'{sys.version_info.major}.{sys.version_info.minor}.{sys.version_info.micro}'
        edps_aa_article = 'https://www.aanda.org/articles/aa/full_html/2024/01/aa47651-23/aa47651-23.html'
        edps_homepage = 'https://www.eso.org/sci/software/edps/'
        online_documentation = 'https://edps.readthedocs.io/en/latest/'
        about_edps = pn.pane.HTML(f"""<font size="3">
        If you make use of EDPS or its workflows for your research, we request to acknowledge<br>
        <a href="{edps_aa_article}" target="_blank">Freudling, Zampieri, Coccato et al 2024, A&A 681, A93.</a>
        <p>
        For more information, please visit the
        <a href="{edps_homepage}" target="_blank">EDPS Homepage</a> or the
        <a href="{online_documentation}" target="_blank">Online Documentation</a>. 
        </font>
        <h3> Version Information </h3>
        <font size="-1">
        <ul>
        <li>edps {edps_version}</li>
        <li>edpsgui {edpsgui_version}</li>
        <li>edpsplot {edpsplot_version}</li>
        <li>Panel {pn.__version__}</li>
        <li>Bokeh {bokeh.__version__}</li>
        <li>hvPlot {hvplot.__version__}</li>
        <li>Python {python_version}</li>
        </ul>
        </font>
        """)

        reduction_queue = pn.Column(
            reduction_table,
            reduction_monitor
        )

        tabs = pn.Tabs(
            ('Workflow', workflow_graph),
            ('Raw Data', data_explorer),
            ('Reduction Queue', reduction_queue),
            ('Reduction Archive', reduction_archive),
            stylesheets=[":host {font-size: 20px;}"]
        )

        def update_tabs(event):
            if event == 1:
                dataset_creator.update_dataset_df()
            elif event == 2:
                reduction_table.update_table()
            elif event == 3:
                reduction_archive.update_table()

        pn.bind(update_tabs, tabs.param.active, watch=True)

        settings_editor = SettingsEditor()
        settings_btn = pmui.IconButton(icon='settings', color='light', size='large', description='Settings')
        settings_btn.on_click(settings_editor.open_modal)
        modal = pn.Modal(about_edps)

        def menu_actions(event):
            if event.new == 'about':
                modal.open = True
            elif event.new == 'settings':
                settings_editor.open_modal(event)

        menu_items = [('About EDPS', 'about'), None, ('Settings', 'settings')]
        help_menu = pn.widgets.MenuButton(name='Help', items=menu_items, icon='help',
                                          button_type='primary', button_style='outline')
        help_menu.on_click(menu_actions)

        logo_path = Path(__file__).resolve().parent / 'eso-logo-blue.jpg'
        template = pn.template.FastListTemplate(title='EDPS Dashboard', logo=str(logo_path), main_layout=None)
        template.main.append(tabs)
        template.main.append(settings_editor)
        template.main.append(modal)
        template.header.append(pn.pane.HTML(f'<font size="2">v{edpsgui_version}</font>'))
        template.header.append(edps_ctl)
        template.header.append(pn.widgets.TooltipIcon(value=TOPBAR_TOOLTIP))
        template.header.append(workflow_selector)
        template.header.append(help_menu)
        return template


job_id = pn.state.location.query_params.get('job_id')
obs_target = pn.state.location.query_params.get('obs_target')
if job_id:
    JobViewer(job_id, str(obs_target)).servable()
else:
    EDPSDashboard().servable()
