import asyncio
from urllib.parse import urlparse

import pandas as pd
import panel as pn
from asyncer import asyncify
from panel.viewable import Viewer, Viewable

from edpsgui.domain.reduction_repository import ClassifiedFile
from edpsplot import get_plot_for_task, Plotter
from .edps_ctl import get_edps
from .file_table import FileTable


class JobViewer(Viewer):

    def __init__(self, job_id: str, obs_target: str = None, **params):
        super().__init__(**params)
        self.edps = get_edps()
        self.job_id = job_id
        self.obs_target = obs_target
        self.job = self.edps.get_job_details(job_id)
        self.plotter = get_plot_for_task(self.job.configuration.instrument, self.job.configuration.task_name)
        self.has_edps_plot = False if type(self.plotter) is Plotter else True
        self.has_graphical_reports = [panel for report in self.job.reports for panel in report.panels] != []
        self.create_reports_btn = pn.widgets.Button(name='Create reports', button_type='primary')
        self.report_inputs_selector = pn.widgets.RadioBoxGroup(name='Report inputs', inline=True)

    def json_viewer(self):
        return pn.Column(
            pn.pane.JSON(self.job.model_dump_json(), depth=-1),
            height=800, scroll=True
        )

    def log_viewer(self):
        logs = {
            log.file_name: self.edps.get_log(self.job_id, log.file_name) for log in self.job.logs
        }
        tabs = [
            (name, pn.widgets.Terminal(content, height=800, sizing_mode='stretch_width'))
            for name, content in logs.items()
        ]
        if tabs:
            return pn.Tabs(*tabs, sizing_mode='scale_both')
        else:
            return pn.pane.Markdown('### No logs available')

    def report_panel_selector(self):
        options = [panel.file_name for report in self.job.reports for panel in report.panels]
        if not options:
            return pn.pane.Markdown('### No reports available')
        select = pn.widgets.Select(name='Select report panel', options=options)
        return pn.Column(
            select,
            pn.bind(lambda file_name: pn.pane.PNG(self.edps.get_report_panel(self.job_id, file_name)),
                    select.param.value),
            sizing_mode='stretch_width'
        )

    def add_to_running_reports(self):
        pn.state.cache['running_report_job_ids'].add(self.job_id)

    def remove_from_running_reports(self):
        pn.state.cache['running_report_job_ids'].discard(self.job_id)

    def has_running_reports(self) -> bool:
        return self.job_id in pn.state.cache['running_report_job_ids']

    async def create_reports(self, event):
        if not event:
            return
        self.create_reports_btn.disabled = True
        self.add_to_running_reports()
        yield pn.indicators.LoadingSpinner(value=True, size=40, name=f"Creating reports ...")
        try:
            # https://panel.holoviz.org/how_to/concurrency/sync_to_async.html
            await asyncify(self.edps.run_reports)(self.job_id, self.report_inputs_selector.value)
        except Exception as e:
            pass
        self.create_reports_btn.disabled = False
        self.remove_from_running_reports()
        self.job = self.edps.get_job_details(self.job_id)
        if all(report.return_code != 0 for report in self.job.reports):
            yield pn.pane.Markdown(f'<span style="color:red">Failed to create reports</span>')
        else:
            yield self.report_panel_selector()

    async def wait_for_reports(self):
        yield pn.indicators.LoadingSpinner(value=True, size=40, name=f"Waiting for reports ...")
        while self.has_running_reports():
            await asyncio.sleep(1)
        self.job = self.edps.get_job_details(self.job_id)
        if all(report.return_code != 0 for report in self.job.reports):
            yield pn.pane.Markdown(f'<span style="color:red">Failed to create reports</span>')
        else:
            yield self.report_panel_selector()

    def configure_report_inputs(self):
        options, value = {}, None
        report_inputs = {report.input for report in self.job.configuration.reports}
        if 'RECIPE_INPUTS' in report_inputs:
            options['Raw data'] = 'raw'
            value = 'raw'
        if 'RECIPE_INPUTS_OUTPUTS' in report_inputs:
            options['Reduced data'] = 'reduced'
            value = 'reduced'
        if len(options) == 2:
            options['Both'] = 'both'
        self.report_inputs_selector.options = options
        self.report_inputs_selector.value = value

    def report_viewer(self):
        if not self.job.configuration.reports:
            return pn.pane.Markdown('### No reports configured for this task')
        if self.job.status.value != 'COMPLETED':
            return pn.pane.Markdown('### Reports are only available for completed jobs')
        if self.has_running_reports():
            return pn.bind(self.wait_for_reports)
        if not self.has_graphical_reports:
            self.create_reports_btn.disabled = False
            self.configure_report_inputs()
            return pn.Column(
                pn.Row(self.report_inputs_selector, self.create_reports_btn),
                pn.bind(self.create_reports, self.create_reports_btn.param.clicks)
            )
        return self.report_panel_selector()

    def plot_viewer(self):
        fits_files = self.job.input_files + self.job.output_files
        try:
            return self.plotter.plot(fits_files)
        except Exception as e:
            return pn.pane.Markdown(f'<span style="color:red">**Error**: {e}</span>')

    def parameters_viewer(self):
        workflow = self.job.configuration.workflow_names[0]
        task = self.job.configuration.task_name
        parameter_set = self.job.configuration.parameters.recipe_parameter_set
        recipe_parameters = self.edps.get_combined_parameters(workflow, parameter_set, task)
        actual_parameters = self.job.recipe_parameters
        parameter_data = []
        for param in recipe_parameters:
            default_value = param.value or param.default
            actual_value = actual_parameters.get(param.param_name, '')
            parameter_data.append((param.param_name, str(default_value), str(actual_value)))
        data_frame = pd.DataFrame(parameter_data, columns=['Parameter', 'Default', 'Custom'])
        editors = {
            'Parameter': None,
            'Default': None,
            'Custom': None,
        }
        titles = {
            'Default': 'Default value',
            'Custom': 'Custom value',
        }
        formatters = {
            'Default': 'textarea',
            'Custom': 'textarea',
        }
        widths = {
            'Default': 300,
            'Custom': 300,
        }
        parameter_table = pn.widgets.Tabulator(data_frame, show_index=False, theme='default', editors=editors,
                                               titles=titles, formatters=formatters, widths=widths)
        parameter_selector = pn.widgets.RadioBoxGroup(
            options={'All parameters': 'all', 'Modified parameters': 'modified'},
            value='modified', inline=True)

        def filter_parameters(df, filter_value):
            if filter_value == 'modified':
                return df[(df['Custom'] != '') & (df['Custom'] != df['Default'])]
            return df

        parameter_table.add_filter(pn.bind(filter_parameters, filter_value=parameter_selector.param.value))

        return pn.Column(
            parameter_selector,
            parameter_table
        )

    def job_summary(self) -> pn.widgets.Tabulator:
        task = self.job.configuration.task_name
        status = self.job.status.value
        setup = self.job.configuration.setup
        setup_values = ','.join(setup.values())
        df = pd.DataFrame({
            'Task': [task],
            'Object': [self.obs_target or 'N/A'],
            'Setup': [setup_values],
            'Status': [status],
        })
        return pn.widgets.Tabulator(df, show_index=False)

    def update_params(self, mpl_pane, new_params):
        # self.recipe_params = str(new_params)
        print(new_params)
        mpl_pane.param.trigger('object')

    @property
    def job_url(self) -> str:
        o = urlparse(pn.state.location.href)
        return f'{o.scheme}://{o.netloc}{o.path}?job_id={self.job_id}&obs_target={self.obs_target or ""}'

    @property
    def active_tab(self) -> int:
        PLOT = 0
        REPORTS = 1
        INPUT_FILES = 2
        OUTPUT_FILES = 3
        LOGS = 4
        if self.job.status.value == 'FAILED':
            return LOGS
        elif self.job.status.value == 'COMPLETED':
            if self.has_edps_plot:
                return PLOT
            elif self.has_graphical_reports:
                return REPORTS
            else:
                return OUTPUT_FILES
        return INPUT_FILES

    def __panel__(self) -> Viewable:
        input_file_table = FileTable()
        output_file_table = FileTable()
        input_file_table.update([ClassifiedFile(f.name, f.category) for f in self.job.input_files])
        output_file_table.update([ClassifiedFile(f.name, f.category) for f in self.job.output_files])
        tabs = pn.Tabs(
            ('Interactive plot', self.plot_viewer()),
            ('Graphical reports', self.report_viewer()),
            ('Input files', input_file_table),
            ('Output files', output_file_table),
            ('Logs', self.log_viewer()),
            ('Parameters', self.parameters_viewer()),
            ('Details', self.json_viewer()),
            active=self.active_tab,
            stylesheets=[":host {font-size: 16px;}"]
        )
        msg = "Click here to open this panel in a new browser tab"
        return pn.Column(
            pn.pane.HTML(f'<a href="{self.job_url}" target="_blank">{msg}</a>'),
            self.job_summary(),
            tabs
        )
