import shutil
from datetime import datetime

import panel as pn
from panel.viewable import Viewer, Viewable

from edpsgui.domain.edpsw import EDPSWrapper
from .edps_ctl import get_edps


class SettingsEditor(Viewer):
    EDPS_CONFIG = EDPSWrapper.APPLICATION_CONFIG

    def __init__(self, **params):
        super().__init__(**params)
        self.layout = pn.Row(
            pn.Spacer(width=1)  # Placeholder for modal
        )
        self.modal = None
        self.file_content = ''
        self.code_editor = pn.widgets.CodeEditor(
            language='properties',
            sizing_mode='stretch_both',
        )

    def reset_layout(self):
        self.layout.pop(-1)
        self.layout.append(pn.Spacer(width=5))

    def save_config(self, event):
        self.reset_layout()
        if self.modal:
            self.modal.open = False
        if self.file_content == self.code_editor.value:
            pn.state.notifications.info('No changes to save')
            return
        backup_file = self.EDPS_CONFIG + '.' + datetime.now().isoformat(timespec='milliseconds')
        shutil.copy(self.EDPS_CONFIG, backup_file)
        with open(self.EDPS_CONFIG, 'w') as f:
            f.write(self.code_editor.value)
        message = f'Configuration saved to {self.EDPS_CONFIG}. Backup created at {backup_file}'
        pn.state.notifications.success(message, duration=10_000)

    def close_modal(self, event):
        self.reset_layout()
        if self.modal:
            self.modal.open = False

    def open_modal(self, event):
        self.file_content = ''
        with open(self.EDPS_CONFIG) as f:
            self.file_content = f.read()
        self.code_editor.value = self.file_content
        save_btn = pn.widgets.Button(name='Save', button_type='primary')
        save_btn.on_click(self.save_config)
        cancel_btn = pn.widgets.Button(name='Cancel', button_type='default')
        cancel_btn.on_click(self.close_modal)
        if get_edps().is_running():
            self.code_editor.readonly = True
            mode = 'view'
            save_btn.icon = 'lock'
            save_btn.disabled = True
            save_btn.description = 'Stop EDPS to edit settings'
        else:
            self.code_editor.readonly = False
            mode = 'edit'
        modal_content = pn.Column(
            f"## EDPS Settings ({mode}) `{self.EDPS_CONFIG}`",
            self.code_editor,
            pn.Row(save_btn, cancel_btn),
            height=800, width=1000
        )
        self.modal = pn.Modal(modal_content)
        self.layout.pop(-1)
        self.layout.append(self.modal)
        self.modal.open = True

    def __panel__(self) -> Viewable:
        return self.layout
