import logging
import os
import re
from pathlib import Path
from typing import List
from urllib.parse import urlparse

import graphviz
import panel as pn
import param
import pypdf
import requests
from asyncer import asyncify
from panel.viewable import Viewer, Viewable

from edpsgui import pipeline_by_workflow, instrument_by_workflow
from .edps_ctl import get_edps

NO_WORKFLOW_MSG = "No workflow"


class Workflow(Viewer):
    workflow = param.String(default=None, allow_refs=True)
    simple_graph_format = param.Selector(default='PDF')
    detailed_graph_format = param.Selector(default='PDF')
    page_reloads = param.Integer(default=0)

    def __init__(self, **params):
        super().__init__(**params)
        self.edps = get_edps()
        self.logger = logging.getLogger('Workflow')
        self.pipeline_documentation_urls = self.get_pipeline_documentation_urls()
        self.simple_graph_format_selector = pn.widgets.RadioButtonGroup.from_param(
            self.param.simple_graph_format,
            options=['PDF', 'PNG']
        )
        self.detailed_graph_format_selector = pn.widgets.RadioButtonGroup.from_param(
            self.param.detailed_graph_format,
            options=['PDF', 'PNG']
        )
        # Hack to trigger recreation of detailed graph when the page is reloaded
        pn.state.onload(lambda: setattr(self, 'page_reloads', self.page_reloads + 1))

    def get_pipeline_documentation_urls(self):
        url = 'https://www.eso.org/sci/software/pipe_aem_table.html'
        pipeline_manual_pattern = r'https://ftp\.eso\.org/pub/dfs/pipelines/instruments/[^/]+/[^/]+-pipeline-manual-[^"\s]+\.pdf'
        reflex_tutorial_pattern = r'https://ftp\.eso\.org/pub/dfs/pipelines/instruments/[^/]+/[^/]+-reflex-tutorial-[^"\s]+\.pdf'
        try:
            response = requests.get(url)
            response.raise_for_status()
            pipeline_manual_urls = re.findall(pipeline_manual_pattern, response.text)
            reflex_tutorial_urls = re.findall(reflex_tutorial_pattern, response.text)
            return pipeline_manual_urls + reflex_tutorial_urls
        except requests.RequestException as e:
            self.logger.error("Failed to fetch URL: %s", e)
            return []

    def get_graph_directory(self) -> str:
        graph_dir = Path(self.edps.base_dir) / 'graph'
        graph_dir.mkdir(parents=True, exist_ok=True)
        return str(graph_dir)

    def get_pdf_url(self, filename) -> str:
        os.environ['EDPSGUI_GRAPH_DIR'] = self.get_graph_directory()
        o = urlparse(pn.state.location.href)
        return f'{o.scheme}://{o.netloc}/pdf?file={os.path.basename(filename)}'

    @staticmethod
    def merge_pdf_files(input_files: List[str], output_file: str) -> None:
        merger = pypdf.PdfWriter()
        for file in input_files:
            merger.append(file)
        Path(output_file).unlink(missing_ok=True)
        merger.write(output_file)
        merger.close()

    def create_simple_graph(self, fmt: str) -> str:
        filename = os.path.join(self.get_graph_directory(), f'{self.workflow}_simple')
        graph = get_edps().get_simple_graph(self.workflow)
        graphviz.Source(graph).render(filename, format=fmt, cleanup=True)
        return f'{filename}.{fmt}'

    def create_detailed_graph(self, fmt: str) -> List[str]:
        graph_dir = self.get_graph_directory()
        pathname = os.path.join(graph_dir, self.workflow)
        graph = get_edps().get_detailed_graph(self.workflow)
        graphviz.Source(graph).render(pathname, format=fmt, cleanup=True)
        paths = [Path(f'{self.workflow}.{fmt}')]
        paths += sorted(Path(graph_dir).glob(f'{self.workflow}.[0-9].{fmt}'), reverse=True)
        files = [os.path.join(graph_dir, p.name) for p in paths]
        if fmt == 'pdf':
            merged_file = f'{pathname}_merged.pdf'
            self.merge_pdf_files(files, merged_file)
            return [merged_file]
        else:
            return files

    @pn.depends('workflow', 'detailed_graph_format', 'page_reloads')
    async def detailed_graph(self):
        if not self.workflow:
            yield NO_WORKFLOW_MSG
            return
        yield pn.indicators.LoadingSpinner(value=True, size=40, name=f"Creating graph for workflow {self.workflow}...")
        try:
            width_slider = pn.widgets.IntSlider(name='Width', start=1000, end=2000, value=1000)
            height_slider = pn.widgets.IntSlider(name='Height', start=800, end=1600, value=800)
            graph_format = self.detailed_graph_format.lower()
            # https://panel.holoviz.org/how_to/concurrency/sync_to_async.html
            graph_files = await asyncify(self.create_detailed_graph)(graph_format)
            if graph_format == 'png':
                width_slider = ""
                graph_widgets = [pn.pane.PNG(f, height=height_slider) for f in graph_files]
                html = ""
            else:
                url = self.get_pdf_url(graph_files[0])
                graph_widgets = [pn.pane.PDF(url, width=width_slider, height=height_slider)]
                html = f'<a href="{url}" target="_blank">Open in a new browser tab</a>'
            yield pn.Column(
                pn.Row(width_slider, height_slider, self.detailed_graph_format_selector),
                pn.pane.HTML(html),
                *graph_widgets
            )
        except Exception as e:
            msg = f"Failed to create graph for workflow {self.workflow}: {e}"
            yield pn.pane.Markdown(f'<span style="color:red">{msg}</span>')

    @pn.depends('workflow', 'simple_graph_format')
    def simple_graph(self):
        if not self.workflow:
            return NO_WORKFLOW_MSG
        try:
            width_slider = pn.widgets.IntSlider(name='Width', start=1000, end=2000, value=1000)
            height_slider = pn.widgets.IntSlider(name='Height', start=800, end=1600, value=800)
            graph_format = self.simple_graph_format.lower()
            graph_file = self.create_simple_graph(graph_format)
            if graph_format == 'png':
                height_slider = ""
                graph_viewer = pn.pane.PNG(graph_file, width=width_slider)
                html = ""
            else:
                url = self.get_pdf_url(graph_file)
                graph_viewer = pn.pane.PDF(url, width=width_slider, height=height_slider)
                html = f'<a href="{url}" target="_blank">Open in a new browser tab</a>'
            return pn.Column(
                pn.Row(width_slider, height_slider, self.simple_graph_format_selector),
                pn.pane.HTML(html),
                graph_viewer
            )
        except Exception as e:
            return f"Failed to create graph for workflow {self.workflow}: {e}"

    @pn.depends('workflow')
    def pipeline_documentation(self):
        if not self.workflow:
            return NO_WORKFLOW_MSG
        workflow = self.workflow.split('.')[0]
        pipeline = pipeline_by_workflow.get(workflow, workflow)
        urls = [url for url in self.pipeline_documentation_urls if pipeline in url]

        # Create HTML links that open in new window
        links = [pn.pane.HTML(f'<a href="{url}" target="_blank">{Path(url).name}</a>')
                 for url in urls]

        instrument = instrument_by_workflow.get(workflow, workflow)
        instrument_page = f'https://www.eso.org/sci/facilities/paranal/instruments/{instrument}.html'

        return pn.Column(
            f"## {workflow.upper()} pipeline documentation",
            *(links if links else [pn.pane.Markdown("No documentation found")]),
            pn.Spacer(height=20),
            pn.pane.HTML(
                f'<font size="3">For more information visit the <a href="{instrument_page}" target="_blank">instrument page</a></font>.'
            )
        )

    @pn.depends('workflow')
    def association_map(self):
        if not self.workflow:
            return NO_WORKFLOW_MSG
        return pn.pane.Markdown(self.edps.get_assoc_map(self.workflow), renderer='markdown', height=800)

    @staticmethod
    def file_content(filename) -> str:
        with open(filename) as f:
            return f.read()

    @pn.depends('workflow')
    def code_viewer(self):
        if not self.workflow:
            return NO_WORKFLOW_MSG
        workflow_path = Path(self.edps.get_workflow_path(self.workflow))
        # workflow_files = {f.name: str(f.resolve()) for f in workflow_path.iterdir() if f.is_file()}
        workflow_files = [f.resolve() for f in workflow_path.iterdir() if f.is_file()]
        workflow_name = self.workflow.split(".")[1] + ".py"
        try:
            selected_file = next(f for f in workflow_files if f.name == workflow_name)
        except StopIteration:
            selected_file = None
        file_selector = pn.widgets.Select(name='Select file', options=workflow_files, value=selected_file)
        code_editor = pn.widgets.CodeEditor(
            value=pn.bind(self.file_content, file_selector),
            language='python', readonly=True, sizing_mode='stretch_both'
        )
        return pn.Column(file_selector, code_editor)

    def __panel__(self) -> Viewable:
        return pn.layout.Tabs(
            ('Task dependencies', self.detailed_graph),
            ('Dataflow', self.simple_graph),
            ('Data sources', pn.Row(self.association_map, sizing_mode='stretch_width', scroll=True)),
            ('Documentation', self.pipeline_documentation),
            ('Code', self.code_viewer),
        )
