import importlib
import logging
import sys
import time
from typing import List, Dict

from adari_core.backends.base import BaseRenderer
from adari_core.backends.bokeh import BokehRenderer
from adari_core.backends.matplotlib import MatplotlibRenderer
from adari_core.report import AdariReportBase

LOGGER = logging.getLogger('adari-wrapper')


def set_adari_reports_path():
    workflow_paths = {path for path in sys.path if 'workflows' in path}
    report_paths = {path.replace('workflows', 'reports') for path in workflow_paths}
    sys.path = list(set(sys.path).union(report_paths))
    LOGGER.info(sys.path)


def get_renderer(name: str) -> BaseRenderer:
    if name.lower() == 'bokeh':
        return BokehRenderer()
    else:
        return MatplotlibRenderer()


def run_report(report_name: str, files: List[Dict[str, str]], which_renderer: str):
    set_adari_reports_path()
    LOGGER.debug('importing %s', report_name)
    report_mod = importlib.import_module(report_name)
    report: AdariReportBase = getattr(report_mod, 'rep')
    report.files = [f['name'] for f in files]
    report.inputs = [(f['name'], f['category']) for f in files]
    renderer = get_renderer(which_renderer)
    report.set_renderer(renderer)
    file_lists = report.parse_sof()
    report.set_hdus(file_lists)
    t0 = time.time()
    panel_list = report.generate_panels(report.files)
    t1 = time.time()
    LOGGER.info('generate_panels %s %.3f', report_name, t1 - t0)
    panel = list(panel_list.keys())[0]
    fig = renderer.render_panel(panel, "a", "b", "c")
    t2 = time.time()
    LOGGER.info('render_panel %s %.3f', report_name, t2 - t1)
    return fig
