import os

import panel as pn
import param
from panel.viewable import Viewer, Viewable
from panel_modal import Modal

from edpsgui.gui.edps_ctl import get_edps


class Inputs(Viewer):
    edps_status = param.Boolean(default=None, allow_refs=True)
    workflow = param.String(default=None, allow_refs=True)
    meta_target = param.Selector()
    target = param.Selector()
    inputs = param.ListSelector()

    def __init__(self, **params):
        super().__init__(**params)
        self.edps = get_edps()
        directory = os.environ.get('EDPSGUI_INPUT_DIR') or '/'
        file_selector = pn.widgets.FileSelector.from_param(self.param.inputs, directory=directory)
        self.inputs_modal = Modal(pn.Column(pn.Spacer(height=20),
                                            pn.pane.Markdown("### Select Inputs"),
                                            file_selector, width=800))


    @pn.depends('workflow')
    def update_meta_target(self):
        if self.workflow:
            meta_targets = self.edps.get_meta_targets(self.workflow)
            if 'science' in meta_targets:
                meta_targets = ['science'] + [mt for mt in meta_targets if mt != 'science']
            self.meta_target = meta_targets[0] if meta_targets else None
            return meta_targets
        else:
            return []

    @pn.depends('workflow', 'meta_target')
    def update_target(self):
        if self.workflow:
            targets = ['all'] + sorted(self.edps.get_targets(self.workflow, self.meta_target))
            self.target = targets[0]
            return targets
        else:
            return []

    def open_modal(self, event):
        self.inputs_modal.is_open = True

    def __panel__(self) -> Viewable:
        meta_target_select = pn.widgets.Select.from_param(self.param.meta_target,
                                                          name='Reduction Target', align='start', width=200,
                                                          options=self.update_meta_target)
        target_select = pn.widgets.Select.from_param(self.param.target,
                                                     name='Reduction Task', align='start', width=200,
                                                     options=self.update_target)

        inputs_btn = pn.widgets.Button(name='Select Inputs', button_type='primary', align='start', width=200)
        inputs_btn.on_click(self.open_modal)

        workflow_text = pn.widgets.StaticText.from_param(self.param.workflow, name='Selected Workflow')
        meta_target_text = pn.widgets.StaticText.from_param(self.param.meta_target, name='Selected Reduction Target')
        target_text = pn.widgets.StaticText.from_param(self.param.target, name='Selected Reduction Task')
        inputs_text = pn.widgets.StaticText.from_param(self.param.inputs, name='Selected Inputs')

        return pn.Column(
            meta_target_select,
            target_select,
            pn.HSpacer(height=10),
            inputs_btn,
            workflow_text,
            meta_target_text,
            target_text,
            inputs_text,
            self.inputs_modal
        )
