// This file is part of the PyHDRL Python language bindings
// Copyright (C) 2020-2024 European Southern Observatory
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

#ifndef PYHDRL_VERSION
#define PYHDRL_VERSION "unknown"
#endif

#include <cpl_error.h>
#include <cpl_init.h>
#include <pybind11/pybind11.h>

#include "hdrlcore/error_bindings.hpp"
#include "hdrlcore/image_bindings.hpp"
#include "hdrlcore/imagelist_bindings.hpp"
#include "hdrldebug/types_bindings.hpp"
#include "hdrlfunc/airmass_bindings.hpp"
#include "hdrlfunc/barycorr_bindings.hpp"
#include "hdrlfunc/bpm_bindings.hpp"
#include "hdrlfunc/catalogue_bindings.hpp"
#include "hdrlfunc/collapse_bindings.hpp"
#include "hdrlfunc/dar_bindings.hpp"
#include "hdrlfunc/flat_bindings.hpp"
#include "hdrlfunc/fpn_bindings.hpp"
#include "hdrlfunc/fringe_bindings.hpp"
#include "hdrlfunc/lacosmic_bindings.hpp"
#include "hdrlfunc/maglim_bindings.hpp"
#include "hdrlfunc/overscan_bindings.hpp"
#include "hdrlfunc/resample_bindings.hpp"
#include "hdrlfunc/strehl_bindings.hpp"

// not yet implemented
// #include "hdrlcore/spectrum_bindings.hpp"
// #include "hdrlfunc/bias_bindings.hpp"

namespace py = pybind11;

PYBIND11_MODULE(hdrl, m)
{
  m.add_object("__version__", py::str(PYHDRL_VERSION),
               "str: package version string");

  cpl_init(CPL_INIT_DEFAULT);
  // in case cpl_init has already been called elsewhere (e.g. PyCPL)
  cpl_error_reset();

  // Placeholder
  py::module_ hdrlcore = m.def_submodule("core", R"pydoc(HDRL Core submodule
  This module provides the features to implement the HDRL core capabilities.
  )pydoc");
  bind_errors(hdrlcore);
  bind_image(hdrlcore);
  bind_imagelist(hdrlcore);


  py::module_ hdrlfunc =
      m.def_submodule("func", R"pydoc(HDRL Functionalities submodule
  This module provides the features to implement the HDRL functionalities.
  )pydoc");
  bind_airmass(hdrlfunc);
  bind_barycorr(hdrlfunc);
  bind_bpm(hdrlfunc);
  bind_catalogue(hdrlfunc);
  bind_collapse(hdrlfunc);
  bind_dar(hdrlfunc);
  bind_flat(hdrlfunc);
  bind_fpn(hdrlfunc);
  bind_fringe(hdrlfunc);
  bind_lacosmic(hdrlfunc);
  bind_maglim(hdrlfunc);
  bind_overscan(hdrlfunc);
  bind_resample(hdrlfunc);
  bind_strehl(hdrlfunc);

  py::module_ hdrldebug = m.def_submodule("debug", R"pydoc(HDRL Debug submodule
  This module provides the features to test PyCPL to PyHDRL custom type converters.
  It is not intended for use by pipeline developers.
  )pydoc");
  bind_types(hdrldebug);
}
