/*
 * This file is part of the ESO Telluric Correction Library
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "mf_gdas.h"
#include "mf_molecules.h"

#include "mf_atm_combined_gdas.h"

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/

/* Routine to copy a CPL table column to a CPL array */
static cpl_error_code mf_atm_col_to_arr(cpl_array *arr, const cpl_table *tab, const char *colname);

/* Merges an atmospheric standard and a GDAS-like profile */
static cpl_error_code mf_atm_merge_gdas(
    cpl_table           *merged_profile,
    const cpl_table     *atm_profile,
    const cpl_table     *gdas_profile,
    const cpl_array     *molecs,
    const mf_parameters *params
);

/* Merge atmospheric standard and GDAS profiles */
static cpl_error_code mf_atm_merge_gdas_grid_fixed(
    cpl_table       *merged_profile,
    const cpl_table *atm_profile,
    const cpl_table *gdas_profile,
    const cpl_array *molecs,
    double          geoelev
);

/* Merges an atmospheric standard and a GDAS-like profile for a given grid of layers (column MF_COL_ATM_HGT) */
static cpl_error_code mf_atm_merge_gdas_grid_natural(
    cpl_table       *merged_profile,
    const cpl_table *atm_profile,
    const cpl_table *gdas_profile,
    const cpl_array *molecs
);

/* Merges the height layers of an atmospheric standard and a GDAS-like profile */
static cpl_error_code mf_atm_get_natural_grid(
    cpl_table           *merged_profile,
    const cpl_table     *atm_profile,
    const cpl_table     *gdas_profile,
    const mf_parameters *params
);

/* Adapt atmospheric profiles using the mf_parameters */
static cpl_error_code mf_atm_adapt(cpl_table *profile, const mf_parameters *params);

/* Scales the water vapor profile of an atmospheric profile table */
static cpl_error_code mf_atm_scale_to_pwv(cpl_table *profile, const mf_parameters *params);

/* Calculates water vapor column pwv in [mm] from profile in [ppmv] */
static void mf_atm_calpwv(double *pwv, cpl_table *prof, const double *geoelev);

/* Convert relative humidity to ppmv */
static void mf_atm_rhum_to_ppmv(double *ppmv, const double *tem, const double *p, const double *hum);

/* Convert ppmv to relative humidity */
static void mf_atm_ppmv_to_rhum(double *hum, const double *tem, const double *p, const double *ppmv);

/* Fill vector with values from cpl_table */
static void mf_atm_fill_vector(double *v, cpl_size nrows, const cpl_table *tab, const char *str);

/* Interpolate columns from two CPL_TABLE */
static cpl_error_code mf_atm_interpolate_linear_column(
    const cpl_table *in_tab,
    const char      *in_str_x,
    const char      *in_str_y,
    cpl_table       *out_tab,
    const char      *out_str_x,
    const char      *out_str_y,
    cpl_size        start,
    cpl_size        end
);

/* Logarithmic linear interpolation */
static double mf_atm_interpolate_linear_log_column(
    const double *x,
    cpl_table    *profile,
    cpl_size     start,
    const char   *column1,
    const char   *column2
);

/* Linear interpolation */
static cpl_error_code mf_atm_interpolate_linear(
    const double *x_out,
    double       *y_out,
    long         n_out,
    const double *x_ref,
    const double *y_ref,
    long         n_ref
);

/*----------------------------------------------------------------------------*/
/**
 *                 Functions
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup mf_atm   Calculate atmospheric profile
 *
 * @brief
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/* ---------------------------------------------------------------------------*/
/**
 * @brief Check user ATM_PROFILE_STANDARD input cpl_table
 *
 * @param table              .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_error_code mf_atm_profile_standard_check(const cpl_table *table)
{
    cpl_ensure(table, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

    cpl_error_ensure(
        cpl_table_has_column(table, MF_COL_ATM_HGT) != 0 && cpl_table_has_column(table, MF_COL_ATM_PRE) != 0 &&
            cpl_table_has_column(table, MF_COL_ATM_TEM) != 0 && cpl_table_get_ncol(table) > 3 &&
            cpl_table_get_nrow(table) > 0,
        CPL_ERROR_INCOMPATIBLE_INPUT, return CPL_ERROR_INCOMPATIBLE_INPUT,
        "cpl_table *%s does not have the correct columns [%s, %s, %s] + n-[molecule]-columns",
        MF_INPUT_ATM_PROFILE_STANDARD, MF_COL_ATM_HGT, MF_COL_ATM_PRE, MF_COL_ATM_TEM
    );

    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Check user ATM_PROFILE_STANDARD input cpl_table
 *
 * @param table              .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_error_code mf_atm_profile_combined_check(const cpl_table *table)
{
    cpl_ensure(table, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

    cpl_error_ensure(
        cpl_table_has_column(table, MF_COL_ATM_HGT) != 0 && cpl_table_has_column(table, MF_COL_ATM_PRE) != 0 &&
            cpl_table_has_column(table, MF_COL_ATM_TEM) != 0 && cpl_table_get_ncol(table) > 3 &&
            cpl_table_get_nrow(table) > 0,
        CPL_ERROR_INCOMPATIBLE_INPUT, return CPL_ERROR_INCOMPATIBLE_INPUT,
        "cpl_table *%s does not have the correct columns [%s, %s, %s] + n-[molecule]-columns",
        MF_INPUT_ATM_PROFILE_COMBINED, MF_COL_ATM_HGT, MF_COL_ATM_PRE, MF_COL_ATM_TEM
    );

    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief reates a combined atmospheric profile standard with GDAS interpolate data.
 *
 * @param params                    mf_parameters parameter configuration structure
 * @param gdas_user                 NULL or user GDAS data provide.
 * @param atm_profile_standard_user
 * @param gdas_profile1             ESO GDAS file below of image date
 * @param gdas_profile2             ESO GDAS file above of image date
 * @param gdas_interpolate          ESO GDAS file interpolate used in the atmospheric profile calculation
 * @param atm_profile_standard
 *
 * @return cpl_table         with the atm profile data calculate or NULL in ERROR.
 *
 * @note The GDAS data be downloaded from a web server if necessary, and a standard atmospheric profile (from MIPAS)
 *              based on the parameters specified in the mf_parameters parameter structure.
 *              If the parameter gdas_prof is set to the default MF_PARAMETERS_GDAS_PROF_AUTO, this profile building mode is performed.
 *              The selection will only use the standard atmospheric profile in other case.
 *              In all other cases, the file path and name of a specific GDAS-like profile is expected.
 *              This will then be merged with the standard profile.
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_table *mf_atm_combined_gdas(
    const mf_parameters *params,
    const cpl_table     *gdas_user,
    const cpl_table     *atm_profile_standard_user,
    cpl_table          **gdas_profile1,
    cpl_table          **gdas_profile2,
    cpl_table          **gdas_interpolate,
    cpl_table          **atm_profile_standard,
    const char         **gdas_src
)
{
    cpl_error_code err = CPL_ERROR_NONE;

    /* Get molecules from parameter list */
    int    nmolec  = params->config->internal.molecules.n_molec;
    char **mol_arr = cpl_table_get_data_string(params->molectab, MF_COL_LIST_MOLECULES);

    /* LBLRTM molecules : Check whether LBLRTM can handle the selected molecules */
    cpl_array *all_molecs = mf_molecules_create_array();
    cpl_size   nrow_all   = cpl_array_get_size(all_molecs);

    for (cpl_size j = 0; j < nmolec; j++) {
        cpl_boolean exmol = CPL_FALSE;
        for (cpl_size i = 0; i < nrow_all; i++) {
            if (strcmp(cpl_array_get_string(all_molecs, i), mol_arr[j]) == 0) {
                exmol = CPL_TRUE;
                break;
            }
        }

        if (!exmol) {
            cpl_array_delete(all_molecs);
            cpl_error_set_message(
                cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                "Invalid object value(s): cpl_table params->molectab (molecule %s cannot be handled)", mol_arr[j]
            );
            return NULL;
        }
    }
    cpl_array_delete(all_molecs);

    /* Check user atm_profile_standard */
    if (atm_profile_standard_user) {
        /* Use user provide ATM_PROFILE_STANDARD */
        cpl_msg_info(cpl_func, "(mf_atm       ) ATM_PROFILE_STANDARD user provided");
        *atm_profile_standard = cpl_table_duplicate(atm_profile_standard_user);
    }
    else {
        /* Get file name of standard profile */
        char *atm_profile_reference_file = cpl_sprintf(
            "%s/%s/%s", params->config->directories.telluriccorr_data_path, MF_PROFILES_MIPAS_PATH,
            params->config->atmospheric.ref_atm
        );

        cpl_msg_info(
            cpl_func, "(mf_atm       ) Read ATM_PROFILE_STANDARD by telluriccorr data : %s", atm_profile_reference_file
        );
        *atm_profile_standard = cpl_table_load(atm_profile_reference_file, 1, 0);
        if (err != CPL_ERROR_NONE) {
            cpl_error_set_message(cpl_func, err, "Could not read reference profile: %s", atm_profile_reference_file);
            cpl_free(atm_profile_reference_file);
            return NULL;
        }

        cpl_free(atm_profile_reference_file);
    }

    /* Column labels from standard profile : Check existence of selected molecules */
    cpl_array *atm_molecs = cpl_table_get_column_names(*atm_profile_standard);
    cpl_size   nrow       = cpl_array_get_size(atm_molecs);
    for (cpl_size j = 0; j < nmolec; j++) {
        cpl_boolean exmol = CPL_FALSE;
        for (cpl_size i = 0; i < nrow; i++) {
            if (strcmp(cpl_array_get_string(atm_molecs, i), mol_arr[j]) == 0) {
                exmol = CPL_TRUE;
                break;
            }
        }

        if (exmol == CPL_FALSE) {
            cpl_error_set_message(
                cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                "Invalid object value(s): cpl_table params->molectab (molecule %s not found in ATM_PROFILE_STANDARD)",
                mol_arr[j]
            );
            cpl_array_delete(atm_molecs);
            return NULL;
        }
    }
    cpl_array_delete(atm_molecs);

    cpl_table *out_profile;

    //*gdas_src = "NONE";
    /* The none/NONE added here ensure that we redirect the NONE case to mf_gdas() in the else clause*/
    if (!gdas_user && strcmp(params->config->atmospheric.gdas_prof, MF_PARAMETERS_GDAS_PROFILE_AUTO) != 0 &&
        !(!strcmp(params->config->atmospheric.gdas_prof, "none") ||
          !strcmp(params->config->atmospheric.gdas_prof, "NONE"))) {
        /*** Take standard profile as output profile (no GDAS profile) ***/
        cpl_msg_info(cpl_func, "(mf_atm       ) Do not consider GDAS profiles");
        out_profile = cpl_table_duplicate(*atm_profile_standard);
        /* gdas_src = "NONE"; */
    }
    else {
        /* Copy selected molecules to CPL array */
        cpl_array *molecs = cpl_array_new(0, CPL_TYPE_STRING);
        mf_atm_col_to_arr(molecs, params->molectab, MF_COL_LIST_MOLECULES);
        if (!err) {
            /* Get GDAS profile */
            if (gdas_user) {
                cpl_msg_info(cpl_func, "(mf_atm       ) GDAS user profile provided");
                *gdas_src         = "USER";
                *gdas_interpolate = cpl_table_duplicate(gdas_user);
            }
            else {
                cpl_msg_info(cpl_func, "(mf_atm       ) GDAS profiles automatically get from ESO GDAS DB");
                *gdas_src                = "REPOSITORY";
                cpl_boolean average_used = CPL_FALSE;
                *gdas_interpolate        = mf_gdas(params, gdas_profile1, gdas_profile2, &average_used);
                if (average_used == CPL_TRUE) {
                    *gdas_src = "AVERAGE";
                }
                /*if mf_gdas uses average profile: It could not find GDAS profile on server -> Using average profile by telluriccorr data : */
                /* gdas_src = "AVERAGE"; */
            }

            /* Merge GDAS and standard atmosphere profiles */
            out_profile = cpl_table_new(0);
            if (*gdas_interpolate) {
                err = mf_atm_merge_gdas(out_profile, *atm_profile_standard, *gdas_interpolate, molecs, params);
            }
            else {
                err = CPL_ERROR_NULL_INPUT;
            }
        }

        /* Cleanup */
        cpl_array_delete(molecs);
    }

    /* Adapt atmospheric profiles to ESO meteo monitor data */
    if (!err) {
        err = mf_atm_adapt(out_profile, params);
    }

    /* Scale H2O profile to given PWV value if requested */
    if (!err) {
        err = mf_atm_scale_to_pwv(out_profile, params);
    }

    /* If error cleanup out_profile */
    if (err != CPL_ERROR_NONE && out_profile) {
        cpl_table_delete(out_profile);
        return NULL;
    }

    return out_profile;
}


/** @cond PRIVATE */

/* ---------------------------------------------------------------------------*/
/**
 * @brief Routine to copy a CPL table column to a CPL array.
 *
 * @param arr                input/output: Filled CPL array, empty at the beginning
 * @param tab                CPL table containing selected column 'colname'.
 * @param colname            Column name of table to be copied to array.
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *
 * @description The target array must have size = 0, otherwise existing array data will be overwritten.
 *              Data types of source array and target table column must match.
 *
 * @note Supported data types: CPL_TYPE_INT, CPL_TYPE_DOUBLE, CPL_TYPE_STRING.
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code mf_atm_col_to_arr(cpl_array *arr, const cpl_table *tab, const char *colname)
{
    /* Input table size check */
    cpl_size nrow_tab = cpl_table_get_nrow(tab);
    if (nrow_tab == 0) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT, "CPL input table has size = 0!\n Cannot continue. Emergency stop."
        );
    }

    /* Existence of source column ? */
    if (cpl_table_has_column(tab, colname) == 0) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT,
            "CPL input table does not contain a column '%s'!\n Cannot continue. Emergency stop.", colname
        );
    }

    /* Size of target array >0 ? */
    cpl_size nrow_arr = cpl_array_get_size(arr);
    if (nrow_arr > 0) {
        cpl_msg_warning(cpl_func, "(mf_atm       ) Input CPL array size > 0, existing data will be overwritten!");
    }

    /* CPL types match ? */
    cpl_type tabcoltype = cpl_table_get_column_type(tab, colname);
    cpl_type arrcoltype = cpl_array_get_type(arr);
    if (tabcoltype != arrcoltype) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT,
            "CPL types of table column and array do not match!\nCannot continue. Emergency stop."
        );
    }

    /* Copying data */
    cpl_array_set_size(arr, nrow_tab);

    for (cpl_size loop = 0; loop < nrow_tab; loop++) {
        switch (tabcoltype) {
            case CPL_TYPE_INT:
                cpl_array_set_int(arr, loop, cpl_table_get_int(tab, colname, loop, NULL));
                break;
            case CPL_TYPE_DOUBLE:
                cpl_array_set_double(arr, loop, cpl_table_get_double(tab, colname, loop, NULL));
                break;
            case CPL_TYPE_STRING:
                cpl_array_set_string(arr, loop, cpl_table_get_string(tab, colname, loop));
                break;
            default:
                return cpl_error_set_message(
                    cpl_func, CPL_ERROR_ILLEGAL_INPUT, "CPL type not supported!\n Cannot continue. Emergency stop."
                );
        }
    }

    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Merges an atmospheric standard and a GDAS-like profile.
 *
 * @param merged_profile     output: merged profile.
 * @param atm_profile        Atmospheric standard profile.
 * @param gdas_profile       GDAS profile.
 * @param molecs             List of molecules.
 * @param params             mf_parameters parameter structure.
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - Error in subroutine (see subroutines).
 *
 * @description Depending on the parameter layers, either a fixed (= 1) or
 *              a natural (= 0) grid of heights is used for the profile.
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code mf_atm_merge_gdas(
    cpl_table           *merged_profile,
    const cpl_table     *atm_profile,
    const cpl_table     *gdas_profile,
    const cpl_array     *molecs,
    const mf_parameters *params
)
{
    /* Perform profile merging */
    cpl_error_code status;
    cpl_boolean    grid_fixed = params->config->atmospheric.layers;
    if (grid_fixed) {
        /* Get elevation of observing site in km */
        double elevation = params->config->ambient.elevation.value * MF_CONV_M_2_KM;

        /* Merging for fixed grid of heights */
        cpl_msg_info(cpl_func, "(mf_atm       ) Take fixed grid of layers for merged profile");
        status = mf_atm_merge_gdas_grid_fixed(merged_profile, atm_profile, gdas_profile, molecs, elevation);
    }
    else {
        /* Get natural grid of heights */
        mf_atm_get_natural_grid(merged_profile, atm_profile, gdas_profile, params);

        /* Merging for natural grid of heights */
        cpl_msg_info(cpl_func, "(mf_atm       ) Take natural grid of layers for merged profile");
        status = mf_atm_merge_gdas_grid_natural(merged_profile, atm_profile, gdas_profile, molecs);
    }

    return status;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Merge atmospheric standard and GDAS profiles.
 *
 * @param merged_profile.    output: Merged profile.
 * @param atm_profile        Atmospheric standard profile.
 * @param gdas_profile       GDAS profile.
 * @param molecs             List of molecules.
 * @param geoelev            Geo elevation in km.
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - Error in subroutine (see subroutines).
 *
 * @description Information contained in GDAS profiles is incorporated into the
 *              standard atmospheric profile. The resulting merged profile contains all
 *              columns from the standard profile and has pressure, temperature and
 *              H2O columns replaced with the GDAS values. The merged profile is
 *              interpolated to a new irregular height grid with 50 levels, the first
 *              29 of which come from GDAS. The four height levels from 20 to 26 km are
 *              a weighted mix of GDAS and standard profile. The influence of the GDAS
 *              profile decreases with increasing height: 80%, 60%, 40%, 20% at 20km,
 *              22km, 24km, 26km. Beyond 26km no GDAS information is available.
 *
 * @note        GDAS does not contain values for molecules other than H2O.
 *              At heights <26km only standard profile information is available for all
 *              other molecules.
 *
 *              All information is in the ATM profile
 *              to introduce some variability at lower altitudes GDAS overwrites
 *              the following parameters:
 *                height, pressure, temperature, H2O via relative humidity
 *
 *              Column names in GDAS:
 *                press height temp relhum
 *
 *              Column names in ATM:
 *                HGT PRE TEM N2 O2 CO2 O3 H2O CH4 N2O HNO3 OH CO NO2 N2O5 ClO HOCl
 *                ClONO2 NO HNO4 HCN NH3 F11 F12 F14 F22 CCl4 COF2 H2O2 C2H2 C2H6
 *                OCS SO2 SF6
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code mf_atm_merge_gdas_grid_fixed(
    cpl_table       *merged_profile,
    const cpl_table *atm_profile,
    const cpl_table *gdas_profile,
    const cpl_array *molecs,
    const double     geoelev
)
{
    double low_hgt[] = { 0.,  0.5, 1.,  1.5, 2.,  2.5, 3.,  3.5, 4.,  4.5, 5.,  5.5, 6.,  6.5, 7.,  7.5, 8.,
                         8.5, 9.,  9.5, 10., 11., 12., 13., 14., 15., 16., 17., 18., 20., 22., 24., 26. };

    /* Number of levels in low_hgt */
    cpl_size n_low_hgt = 33;

    /* Number of lowest layer */
    cpl_size i0 = 0;

    /* height interval */
    double dh = 1.;

    /* Find lowest hgt level */
    for (cpl_size i = 0; i < n_low_hgt; i++) {
        if (low_hgt[i] > geoelev - dh) {
            i0 = i;
            break;
        }
    }

    /* Get numbers of rows for GDAS, ATM and merged profile */
    cpl_size gdas_nrows   = cpl_table_get_nrow(gdas_profile);
    cpl_size merged_nrows = cpl_table_get_nrow(merged_profile);

    /* Ensure that output table has (at least) 50 rows for Cerro Paranal */
    if (merged_nrows < 54 - i0) {
        cpl_table_set_size(merged_profile, 54 - i0);
        merged_nrows = 54 - i0;
    }

    /* Define array for height levels */
    double *hgt_levels = cpl_calloc(merged_nrows, sizeof(double));

    /* Fill hgt vector with values for GDAS range */
    for (cpl_size i = 0; i < n_low_hgt - i0; i++) {
        hgt_levels[i] = low_hgt[i + i0];
    }

    /* Calculate step size for higher hgt levels (spacing of upper levels) */
    double step = (120 - low_hgt[n_low_hgt - 1]) / (double)(merged_nrows - (n_low_hgt - i0));

    /* Fill hgt vector with higher level values */
    for (cpl_size i = n_low_hgt - i0; i < merged_nrows; i++) {
        hgt_levels[i] = low_hgt[n_low_hgt - 1] + (double)(i - n_low_hgt + i0 + 1) * step;
    }

    /* Get number of GDAS levels below height of 20 km in new grid */
    cpl_size n_gdas_max = n_low_hgt - i0 - 4;


    /*** Insert new columns including unit and format ***/

    /* Height in [km] */
    cpl_table_new_column(merged_profile, MF_COL_ATM_HGT, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(merged_profile, MF_COL_ATM_HGT, MF_UNIT_DIST);
    cpl_table_set_column_format(merged_profile, MF_COL_ATM_HGT, "%10.3e");

    /* Pressure in [mb] */
    cpl_table_new_column(merged_profile, MF_COL_ATM_PRE, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(merged_profile, MF_COL_ATM_PRE, MF_UNIT_PRESS);
    cpl_table_set_column_format(merged_profile, MF_COL_ATM_PRE, "%10.3e");

    /* Temperature in [k] */
    cpl_table_new_column(merged_profile, MF_COL_ATM_TEM, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(merged_profile, MF_COL_ATM_TEM, MF_UNIT_TEMP);
    cpl_table_set_column_format(merged_profile, MF_COL_ATM_TEM, "%10.3e");

    /* Existence of H2O column */
    cpl_boolean exh2o = CPL_FALSE;

    /* Loop over all molecules */
    cpl_size n_molecs = cpl_array_get_size(molecs);
    for (cpl_size i = 0; i < n_molecs; i++) {
        /* String for a single molecule */
        const char *mol = cpl_array_get_string(molecs, i);

        cpl_table_new_column(merged_profile, mol, CPL_TYPE_DOUBLE);
        cpl_table_set_column_unit(merged_profile, mol, MF_UNIT_VOL);
        cpl_table_set_column_format(merged_profile, mol, "%10.3e");

        /* Existence of H2O column */
        if (strcmp(mol, MF_MOLECULES_H2O) == 0) {
            exh2o = CPL_TRUE;
        }
    }

    /* Write height column */
    for (cpl_size i = 0; i < merged_nrows; i++) {
        cpl_table_set_double(merged_profile, MF_COL_ATM_HGT, i, *(hgt_levels + i));
    }
    cpl_free(hgt_levels);

    /* Initialize error code */
    cpl_error_code err_code = CPL_ERROR_NONE;

    /* Interpolate pressure column (GDAS range) */
    err_code += mf_atm_interpolate_linear_column(
        gdas_profile, MF_COL_GDAS_HEIGHT, MF_COL_GDAS_PRESS, merged_profile, MF_COL_ATM_HGT, MF_COL_ATM_PRE, 0,
        n_gdas_max + 4
    );

    /* Interpolate temperature column (GDAS range) */
    err_code += mf_atm_interpolate_linear_column(
        gdas_profile, MF_COL_GDAS_HEIGHT, MF_COL_GDAS_TEMP, merged_profile, MF_COL_ATM_HGT, MF_COL_ATM_TEM, 0,
        n_gdas_max + 4
    );

    if (err_code != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, err_code, "Problem in interpolation");
    }

    /* Interpolate H2O column (GDAS range) */
    if (exh2o) {
        /* Temporary table for storage GDAS rel_hum and calculating the overlap region */
        cpl_table *tmp_gdas = cpl_table_new(gdas_nrows);

        /* Height in [km] */
        cpl_table_new_column(tmp_gdas, MF_COL_GDAS_HEIGHT, CPL_TYPE_DOUBLE);
        cpl_table_set_column_unit(tmp_gdas, MF_COL_GDAS_HEIGHT, MF_UNIT_DIST);
        cpl_table_set_column_format(tmp_gdas, MF_COL_GDAS_HEIGHT, "%10.3e");

        /* H2O */
        cpl_table_new_column(tmp_gdas, MF_MOLECULES_H2O, CPL_TYPE_DOUBLE);
        cpl_table_set_column_unit(tmp_gdas, MF_MOLECULES_H2O, MF_UNIT_VOL);
        cpl_table_set_column_format(tmp_gdas, MF_MOLECULES_H2O, "%10.3e");

        /* Convert relative humidity from GDAS to ppmv (stored in tmp_gdas) */
        for (cpl_size i = 0; i < gdas_nrows; i++) {
            double rel_hum = cpl_table_get_double(gdas_profile, MF_COL_GDAS_RELHUM, i, NULL);
            double h       = cpl_table_get_double(gdas_profile, MF_COL_GDAS_HEIGHT, i, NULL);
            double p       = cpl_table_get_double(gdas_profile, MF_COL_GDAS_PRESS, i, NULL);
            double t       = cpl_table_get_double(gdas_profile, MF_COL_GDAS_TEMP, i, NULL);

            double ppmv = 0.;
            mf_atm_rhum_to_ppmv(&ppmv, &t, &p, &rel_hum);

            cpl_table_set_double(tmp_gdas, MF_MOLECULES_H2O, i, ppmv);
            cpl_table_set_double(tmp_gdas, MF_COL_GDAS_HEIGHT, i, h);
        }

        /* Interpolate H2O column (overlap region) */
        err_code = mf_atm_interpolate_linear_column(
            tmp_gdas, MF_COL_GDAS_HEIGHT, MF_MOLECULES_H2O, merged_profile, MF_COL_ATM_HGT, MF_MOLECULES_H2O, 0,
            n_gdas_max + 4
        );

        cpl_table_delete(tmp_gdas);

        if (err_code != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, err_code, "Problem in interpolation");
        }
    }

    /* New temporary table for storage GDAS values in overlap region and calculating it */
    cpl_table *overlap_region = cpl_table_new(merged_nrows);

    /* Pressure in [mb] */
    cpl_table_new_column(overlap_region, MF_COL_ATM_PRE, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(overlap_region, MF_COL_ATM_PRE, MF_UNIT_PRESS);
    cpl_table_set_column_format(overlap_region, MF_COL_ATM_PRE, "%10.3e");

    /* Temperature in [K] */
    cpl_table_new_column(overlap_region, MF_COL_ATM_TEM, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(overlap_region, MF_COL_ATM_TEM, MF_UNIT_TEMP);
    cpl_table_set_column_format(overlap_region, MF_COL_ATM_TEM, "%10.3e");

    /* If column H2O */
    if (exh2o) {
        cpl_table_new_column(overlap_region, MF_MOLECULES_H2O, CPL_TYPE_DOUBLE);
        cpl_table_set_column_unit(overlap_region, MF_MOLECULES_H2O, MF_UNIT_VOL);
        cpl_table_set_column_format(overlap_region, MF_MOLECULES_H2O, "%10.3e");
    }

    /* Set TEM, PRE, & H2O columns */
    for (cpl_size i = 0; i < n_gdas_max + 4; i++) {
        double p = cpl_table_get_double(merged_profile, MF_COL_ATM_PRE, i, NULL);
        cpl_table_set_double(overlap_region, MF_COL_ATM_PRE, i, p);

        double t = cpl_table_get_double(merged_profile, MF_COL_ATM_TEM, i, NULL);
        cpl_table_set_double(overlap_region, MF_COL_ATM_TEM, i, t);

        if (exh2o) {
            double ppmv = cpl_table_get_double(merged_profile, MF_MOLECULES_H2O, i, NULL);
            cpl_table_set_double(overlap_region, MF_MOLECULES_H2O, i, ppmv);
        }
    }

    /* Interpolate pressure column (ATM range) */
    err_code += mf_atm_interpolate_linear_column(
        atm_profile, MF_COL_ATM_HGT, MF_COL_ATM_PRE, merged_profile, MF_COL_ATM_HGT, MF_COL_ATM_PRE, n_gdas_max,
        merged_nrows
    );

    /* Interpolate temperature column (ATM range) */
    err_code += mf_atm_interpolate_linear_column(
        atm_profile, MF_COL_ATM_HGT, MF_COL_ATM_TEM, merged_profile, MF_COL_ATM_HGT, MF_COL_ATM_TEM, n_gdas_max,
        merged_nrows
    );

    /* Insert molecules if requested */
    for (cpl_size i = 0; i < n_molecs; i++) {
        const char *mol = cpl_array_get_string(molecs, i);

        /* Insert H2O only above GDAS data, rest from 0km */
        if (strcmp(mol, MF_MOLECULES_H2O) == 0) {
            err_code += mf_atm_interpolate_linear_column(
                atm_profile, MF_COL_ATM_HGT, MF_MOLECULES_H2O, merged_profile, MF_COL_ATM_HGT, MF_MOLECULES_H2O,
                n_gdas_max, merged_nrows
            );
        }
        else {
            err_code += mf_atm_interpolate_linear_column(
                atm_profile, MF_COL_ATM_HGT, mol, merged_profile, MF_COL_ATM_HGT, mol, 0, merged_nrows
            );
        }
    }

    /* Insert overlap region */
    for (cpl_size i = n_gdas_max, j = 1; i < n_gdas_max + 4; i++, j++) {
        double val_overlap;
        double val_merged;
        double dj = (double)j;

        val_merged  = cpl_table_get_double(merged_profile, MF_COL_ATM_PRE, i, NULL);
        val_overlap = cpl_table_get_double(overlap_region, MF_COL_ATM_PRE, i, NULL);
        cpl_table_set_double(merged_profile, MF_COL_ATM_PRE, i, val_merged * 0.2 * dj + val_overlap * (1 - 0.2 * dj));

        val_merged  = cpl_table_get_double(merged_profile, MF_COL_ATM_TEM, i, NULL);
        val_overlap = cpl_table_get_double(overlap_region, MF_COL_ATM_TEM, i, NULL);
        cpl_table_set_double(merged_profile, MF_COL_ATM_TEM, i, val_merged * 0.2 * dj + val_overlap * (1 - 0.2 * dj));

        if (exh2o) {
            val_merged  = cpl_table_get_double(merged_profile, MF_MOLECULES_H2O, i, NULL);
            val_overlap = cpl_table_get_double(overlap_region, MF_MOLECULES_H2O, i, NULL);
            cpl_table_set_double(
                merged_profile, MF_MOLECULES_H2O, i, val_merged * 0.2 * dj + val_overlap * (1 - 0.2 * dj)
            );
        }
    }
    cpl_table_delete(overlap_region);

    if (err_code != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, err_code, "Problem in interpolation");
    }

    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Merges an atmospheric standard and a GDAS-like profile for a given grid of layers (column MF_COL_ATM_HGT).
 *
 * @param merged_profile     input/output: CPL table merged profile with grid of layers in km in column MF_COL_ATM_HGT
 * @param atm_profile        Atmospheric standard profile.
 * @param gdas_profile       GDAS profile.
 * @param molecs             List of molecules.
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - Invalid object structure.
 *                           - Error in subroutine (see subroutines).
 *
 * @description The resulting merged profile contains all columns from
 *              the standard profile and has pressure, temperature, and H2O columns
 *              replaced with the GDAS values.
 *              Above the uppermost GDAS layer, the relative deviation between
 *              the standard and the GDAS profile (as measured for the highest
 *              GDAS layer) is gradually decreased to be zero at a 1 + mf_MERGEFRAC
 *              times higher altitude. Above that limiting height the unmodified
 *              standard profile is written into the output profile.
 *
 * @note Column names in GDAS:
 *       - press height temp relhum
 *
 * @note Column names in ATM:
 *       - HGT PRE TEM N2 O2 CO2 O3 H2O CH4 N2O HNO3 OH CO NO2 N2O5 ClO HOCl
 *         ClONO2 NO HNO4 HCN NH3 F11 F12 F14 F22 CCl4 COF2 H2O2 C2H2 C2H6
 *         OCS SO2 SF6
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code mf_atm_merge_gdas_grid_natural(
    cpl_table       *merged_profile,
    const cpl_table *atm_profile,
    const cpl_table *gdas_profile,
    const cpl_array *molecs
)
{
    /* Check existence of HGT column */
    if (cpl_table_has_column(merged_profile, MF_COL_ATM_HGT) != 1) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
            "Invalid object structure: cpl_table *merged_profile (no HGT column)"
        );
    }

    /* Get numbers of rows for GDAS and merged profile */
    cpl_size gdas_nrows   = cpl_table_get_nrow(gdas_profile);
    cpl_size merged_nrows = cpl_table_get_nrow(merged_profile);

    /* Get pointer to HGT column */
    double *hgt_levels = cpl_table_get_data_double(merged_profile, MF_COL_ATM_HGT);

    /* Get critical heights for profile merging */
    double h_gdas_max = cpl_table_get_double(gdas_profile, MF_COL_GDAS_HEIGHT, gdas_nrows - 1, NULL);
    double h_merge    = h_gdas_max * (1. + MF_MERGE_FRAC);
    double dh         = h_merge - h_gdas_max;

    /* Loop over new grid and find critical rows */
    cpl_size n_gdas_max = -1;
    cpl_size n_merge    = -1;
    for (cpl_size i = 0; i < merged_nrows; i++) {
        if (n_gdas_max < 0 && hgt_levels[i] > h_gdas_max) {
            n_gdas_max = i;
        }
        if (n_merge < 0 && hgt_levels[i] > h_merge) {
            n_merge = i;
        }
    }


    /*** Insert new columns including unit and format ***/

    /* Height in [km] (must exist) */
    cpl_table_set_column_unit(merged_profile, MF_COL_ATM_HGT, MF_UNIT_DIST);
    cpl_table_set_column_format(merged_profile, MF_COL_ATM_HGT, "%10.3e");

    /* pressure in [mb] */
    cpl_table_new_column(merged_profile, MF_COL_ATM_PRE, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(merged_profile, MF_COL_ATM_PRE, MF_UNIT_PRESS);
    cpl_table_set_column_format(merged_profile, MF_COL_ATM_PRE, "%10.3e");

    /* temperature in [k] */
    cpl_table_new_column(merged_profile, MF_COL_ATM_TEM, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(merged_profile, MF_COL_ATM_TEM, MF_UNIT_TEMP);
    cpl_table_set_column_format(merged_profile, MF_COL_ATM_TEM, "%10.3e");

    /* Get the number of molecules */
    cpl_size n_molecs = cpl_array_get_size(molecs);

    /* Loop over all molecules */
    cpl_boolean exh2o = CPL_FALSE;
    for (cpl_size i = 0; i < n_molecs; i++) {
        /* String for a single molecule */
        const char *mol = cpl_array_get_string(molecs, i);

        cpl_table_new_column(merged_profile, mol, CPL_TYPE_DOUBLE);
        cpl_table_set_column_unit(merged_profile, mol, MF_UNIT_VOL);
        cpl_table_set_column_format(merged_profile, mol, "%10.3e");

        /* Existence of H2O column */
        if (strcmp(mol, MF_MOLECULES_H2O) == 0) {
            exh2o = CPL_TRUE;
        }
    }

    /* Initialize error */
    cpl_error_code err_code = CPL_ERROR_NONE;

    /* Interpolate pressure column (GDAS range) */
    err_code += mf_atm_interpolate_linear_column(
        gdas_profile, MF_COL_GDAS_HEIGHT, MF_COL_GDAS_PRESS, merged_profile, MF_COL_ATM_HGT, MF_COL_ATM_PRE, 0,
        n_gdas_max
    );

    /* Interpolate temperature column (GDAS range) */
    err_code += mf_atm_interpolate_linear_column(
        gdas_profile, MF_COL_GDAS_HEIGHT, MF_COL_GDAS_TEMP, merged_profile, MF_COL_ATM_HGT, MF_COL_ATM_TEM, 0,
        n_gdas_max
    );

    /* Check for errors */
    if (err_code != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, err_code, "Problem in interpolation");
    }

    /* Interpolate H2O column (GDAS range) */

    if (exh2o) {
        /* New temporary table for calculating H2O in ppmv */
        cpl_table *tmp_gdas = cpl_table_new(gdas_nrows);
        cpl_table_new_column(tmp_gdas, MF_COL_GDAS_HEIGHT, CPL_TYPE_DOUBLE); /* in km   */
        cpl_table_new_column(tmp_gdas, MF_MOLECULES_H2O, CPL_TYPE_DOUBLE);   /* in ppmv */

        /* Convert relative humidity from GDAS to ppmv (stored in tmp_gdas) */
        for (cpl_size i = 0; i < gdas_nrows; i++) {
            double rel_hum = cpl_table_get_double(gdas_profile, MF_COL_GDAS_RELHUM, i, NULL);
            double h       = cpl_table_get_double(gdas_profile, MF_COL_GDAS_HEIGHT, i, NULL);
            double p       = cpl_table_get_double(gdas_profile, MF_COL_GDAS_PRESS, i, NULL);
            double t       = cpl_table_get_double(gdas_profile, MF_COL_GDAS_TEMP, i, NULL);

            double ppmv = 0.;
            mf_atm_rhum_to_ppmv(&ppmv, &t, &p, &rel_hum);

            cpl_table_set_double(tmp_gdas, MF_MOLECULES_H2O, i, ppmv);
            cpl_table_set_double(tmp_gdas, MF_COL_GDAS_HEIGHT, i, h);
        }

        /* Interpolate H2O column (GDAS range) */
        err_code = mf_atm_interpolate_linear_column(
            tmp_gdas, MF_COL_GDAS_HEIGHT, MF_MOLECULES_H2O, merged_profile, MF_COL_ATM_HGT, MF_MOLECULES_H2O, 0,
            n_gdas_max
        );

        /* Delete temporary table */
        cpl_table_delete(tmp_gdas);

        /* Check for errors */
        if (err_code != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, err_code, "Problem in interpolation");
        }
    }

    /* New temporary table for storing the GDAS values and calculating the overlap region */
    cpl_table *overlap_region = cpl_table_new(merged_nrows);
    cpl_table_new_column(overlap_region, MF_COL_ATM_PRE, CPL_TYPE_DOUBLE); /* in mb   */
    cpl_table_new_column(overlap_region, MF_COL_ATM_TEM, CPL_TYPE_DOUBLE); /* in K    */
    if (exh2o) {
        cpl_table_new_column(overlap_region, MF_MOLECULES_H2O, CPL_TYPE_DOUBLE); /* in ppmv */
    }

    /* Set TEM, PRE, & H2O columns */
    for (cpl_size i = 0; i < n_gdas_max; i++) {
        double p = cpl_table_get_double(merged_profile, MF_COL_ATM_PRE, i, NULL);
        cpl_table_set_double(overlap_region, MF_COL_ATM_PRE, i, p);

        double t = cpl_table_get_double(merged_profile, MF_COL_ATM_TEM, i, NULL);
        cpl_table_set_double(overlap_region, MF_COL_ATM_TEM, i, t);

        if (exh2o) {
            double ppmv = cpl_table_get_double(merged_profile, MF_MOLECULES_H2O, i, NULL);
            cpl_table_set_double(overlap_region, MF_MOLECULES_H2O, i, ppmv);
        }
    }

    /* Interpolate pressure column (ATM range) */
    err_code += mf_atm_interpolate_linear_column(
        atm_profile, MF_COL_ATM_HGT, MF_COL_ATM_PRE, merged_profile, MF_COL_ATM_HGT, MF_COL_ATM_PRE, n_gdas_max - 1,
        merged_nrows
    );

    /* Interpolate temperature column (ATM range) */
    err_code += mf_atm_interpolate_linear_column(
        atm_profile, MF_COL_ATM_HGT, MF_COL_ATM_TEM, merged_profile, MF_COL_ATM_HGT, MF_COL_ATM_TEM, n_gdas_max - 1,
        merged_nrows
    );

    /* Insert molecules (Insert H2O only above GDAS data, rest from 0km) */
    for (cpl_size i = 0; i < n_molecs; i++) {
        const char *mol = cpl_array_get_string(molecs, i);

        if (strcmp(mol, MF_MOLECULES_H2O) == 0) {
            err_code += mf_atm_interpolate_linear_column(
                atm_profile, MF_COL_ATM_HGT, MF_MOLECULES_H2O, merged_profile, MF_COL_ATM_HGT, MF_MOLECULES_H2O,
                n_gdas_max - 1, merged_nrows
            );
        }
        else {
            err_code += mf_atm_interpolate_linear_column(
                atm_profile, MF_COL_ATM_HGT, mol, merged_profile, MF_COL_ATM_HGT, mol, 0, merged_nrows
            );
        }
    }

    /* Insert overlap region */
    double dp   = 0.;
    double dt   = 0.;
    double dh2o = 0.;
    for (cpl_size i = n_gdas_max - 1; i < n_merge; i++) {
        double hfrac      = (h_merge - hgt_levels[i]) / dh;
        double val_merged = cpl_table_get_double(merged_profile, MF_COL_ATM_PRE, i, NULL);

        if (i == n_gdas_max - 1) {
            double val_overlap = cpl_table_get_double(overlap_region, MF_COL_ATM_PRE, i, NULL);
            dp                 = val_overlap / val_merged - 1;
        }

        cpl_table_set_double(merged_profile, MF_COL_ATM_PRE, i, val_merged * (1 + dp * hfrac));
        val_merged = cpl_table_get_double(merged_profile, MF_COL_ATM_TEM, i, NULL);

        if (i == n_gdas_max - 1) {
            double val_overlap = cpl_table_get_double(overlap_region, MF_COL_ATM_TEM, i, NULL);
            dt                 = val_overlap / val_merged - 1;
        }

        cpl_table_set_double(merged_profile, MF_COL_ATM_TEM, i, val_merged * (1 + dt * hfrac));

        if (exh2o) {
            val_merged = cpl_table_get_double(merged_profile, MF_MOLECULES_H2O, i, NULL);

            if (i == n_gdas_max - 1) {
                double val_overlap = cpl_table_get_double(overlap_region, MF_MOLECULES_H2O, i, NULL);
                dh2o               = val_overlap / val_merged - 1;
            }

            cpl_table_set_double(merged_profile, MF_MOLECULES_H2O, i, val_merged * (1 + dh2o * hfrac));
        }
    }

    /* Delete temporary table */
    cpl_table_delete(overlap_region);

    /* Check for errors */
    if (err_code != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, err_code, "Problem in interpolation");
    }

    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Merges the height layers of an atmospheric standard and a GDAS-like profile.
 *
 * @param merged_profile     CPL table with grid of layers in km in column MF_COL_ATM_HGT.
 * @param atm_profile        Atmospheric standard profile.
 * @param gdas_profile       GDAS profile.
 * @param params             mf_parameters parameter structure.
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - Error in subroutine (see subroutines).
 *
 * @description All layers of both profiles are combined and written into the
 *              MF_COL_ATM_HGT column of the output profile. If local meteo data are
 *              considered, the geo elevation will added as well.
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code mf_atm_get_natural_grid(
    cpl_table           *merged_profile,
    const cpl_table     *atm_profile,
    const cpl_table     *gdas_profile,
    const mf_parameters *params
)
{
    /* Check existence of HGT column in output table */
    if (cpl_table_has_column(merged_profile, MF_COL_ATM_HGT) != 1) {
        cpl_table_new_column(merged_profile, MF_COL_ATM_HGT, CPL_TYPE_DOUBLE);
        cpl_table_set_column_unit(merged_profile, MF_COL_ATM_HGT, MF_UNIT_DIST);
        cpl_table_set_column_format(merged_profile, MF_COL_ATM_HGT, "%10.3e");
    }

    /* Get number of layers */
    cpl_size n_atm  = cpl_table_get_nrow(atm_profile);
    cpl_size n_gdas = cpl_table_get_nrow(gdas_profile);
    cpl_size n_max  = n_atm + n_gdas;

    double emix      = params->config->atmospheric.emix; /* If emix > gelev, add geoelevation for local meteo data */
    double elevation = params->config->ambient.elevation.value *
                       MF_CONV_M_2_KM; /* Get geoelevation of observing site in km               */
    if (emix > elevation) {
        n_max += 1;
    }

    /* Set size of output table */
    cpl_table_set_size(merged_profile, n_max);

    /* Get lowest layer height */
    double dh    = 1.;
    double h_min = elevation - dh;
    double h0    = h_min;

    /* Get pointers to height columns */
    const double *hgt_atm  = cpl_table_get_data_double_const(atm_profile, MF_COL_ATM_HGT);
    const double *hgt_gdas = cpl_table_get_data_double_const(gdas_profile, MF_COL_GDAS_HEIGHT);

    /* Initialize output grid column and get pointer to it */
    cpl_table_fill_column_window_double(merged_profile, MF_COL_ATM_HGT, 0, n_max, 0.);
    double *hgt_merged = cpl_table_get_data_double(merged_profile, MF_COL_ATM_HGT);

    /* Build grid */
    cpl_size new_size = 0;
    cpl_size i_obs    = 0;
    cpl_size i_atm    = 0;
    cpl_size i_gdas   = 0;
    for (cpl_size i = 0; i < n_max; i++) {
        double h;

        if ((emix > elevation) && (i_obs == 0) && (elevation < hgt_atm[i_atm]) && (elevation < hgt_gdas[i_gdas])) {
            h = elevation;
            i_obs++;
        }
        else if (i_atm < n_atm && (i_gdas >= n_gdas || hgt_atm[i_atm] < hgt_gdas[i_gdas])) {
            h = hgt_atm[i_atm];
            i_atm++;
        }
        else {
            h = hgt_gdas[i_gdas];
            i_gdas++;
        }

        if (h > h0) {
            hgt_merged[new_size] = h;
            h0                   = h;
            new_size++;
        }
    }

    /* Resize output table */
    cpl_table_set_size(merged_profile, new_size);

    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief  Adapt atmospheric profiles using the mf_parameters to local conditions (temperature/pressure/humidity).
 *
 * @param profile            in/out: Atmospheric profile (adapted to local conditions).
 * @param params             mf_parameters parameter structure
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - Error in subroutine (see subroutines).
 *
 * @description The input profile gets overwritten on output.
                This routine adapts the pressure, temperature, and humidity to local on-sight measurements:
 *              Up to gelev, the ratio of the on-site value and the profile value at gelev is used as scaling factor.
 *              Profile values above emix are left untouched. In between, a smooth transition is created by linear interpolation.
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code mf_atm_adapt(cpl_table *profile, const mf_parameters *params)
{
    double elevation = params->config->ambient.elevation.value * MF_CONV_M_2_KM; /* Geo elevation       in [km   ] */
    double emix      = params->config->atmospheric.emix;                         /* Upper mixing height in [km   ] */

    /* If emix <= gelev, do not consider data of local meteo station */
    if (emix <= elevation) {
        cpl_msg_info(cpl_func, "(mf_atm       ) Do not consider local meteo data");
        return CPL_ERROR_NONE;
    }

    /* Get data of local meteo station */
    double ghum  = 0.;                                        /* Geo humidity        in [ppmv ] */
    double gpres = params->config->ambient.pressure.value;    /* Geo pressure        in [mbar ] */
    double gtemp = params->config->ambient.temperature.value; /* Geo temperature     in [deg C] */

    /* Convert relative humdity into ppmv */
    double hum_orig  = params->config->ambient.relative_humidity.value;
    double temp_orig = gtemp + MF_CONV_T_2_K;
    mf_atm_rhum_to_ppmv(&ghum, &temp_orig, &gpres, &hum_orig);

    /* Modify atmospheric profile */
    cpl_msg_info(cpl_func, "(mf_atm       ) Consider local meteo data");

    /* Find starting height */
    cpl_size init_height;
    for (init_height = 1; elevation >= cpl_table_get(profile, MF_COL_ATM_HGT, init_height, NULL); init_height++)
        ;

    /* Check existence of H2O column */
    cpl_boolean exh2o = CPL_TRUE;
    if (cpl_table_has_column(profile, MF_MOLECULES_H2O) != 1) {
        exh2o = CPL_FALSE;
    }

    /* Interpolate pressure / temperature / humidity value at gelev */
    double pres =
        mf_atm_interpolate_linear_log_column(&elevation, profile, init_height, MF_COL_ATM_PRE, MF_COL_ATM_HGT);
    double temp =
        mf_atm_interpolate_linear_log_column(&elevation, profile, init_height, MF_COL_ATM_TEM, MF_COL_ATM_HGT);
    double hum = ghum;
    if (exh2o) {
        hum = mf_atm_interpolate_linear_log_column(&elevation, profile, init_height, MF_MOLECULES_H2O, MF_COL_ATM_HGT);
        if (hum < MF_TOL) {
            hum = ghum;
            cpl_msg_warning(
                cpl_func, "(mf_atm       ) Do not consider local rel. humidity due to zero value in model profile at "
                          "altitude of meteo station"
            );
        }
    }

    /* Get the number nint of init_height levels up to emix */
    cpl_size nint;
    for (nint = init_height; emix >= cpl_table_get(profile, MF_COL_ATM_HGT, nint, NULL); nint++)
        ;

    /* Relative pressure / temperature / humidity shifts at gelev */
    double dpres = gpres / pres - 1;
    double dtemp = (gtemp + MF_CONV_T_2_K) / temp - 1;
    double dhum  = ghum / hum - 1;

    /* Adapt pressure / temperature / humidity profiles to local on-site measurements */
    for (cpl_size i = 0; i < nint; i++) {
        /* Calculate change of scaling factor depending on init_height */
        double hgt = cpl_table_get(profile, MF_COL_ATM_HGT, i, NULL);

        double frac = (hgt - elevation) / (emix - elevation);
        if (frac > 1) {
            frac = 1;
        }
        else if (frac < 0) {
            frac = 0;
        }

        /* Read pressure / temperature / humidity */
        double t = cpl_table_get(profile, MF_COL_ATM_TEM, i, NULL);
        double p = cpl_table_get(profile, MF_COL_ATM_PRE, i, NULL);

        /* Check for reasonable relative humidity (no correction so far) */
        double ppmv = 0.;
        if (exh2o) {
            double rhum = 0.;
            ppmv        = cpl_table_get(profile, MF_MOLECULES_H2O, i, NULL);
            mf_atm_ppmv_to_rhum(&rhum, &t, &p, &ppmv);

            if (rhum * (1 + dhum * (1 - frac)) > 100.) {
                cpl_msg_info(cpl_func, "(mf_atm       ) Rel. humidity > 100%% for layer %lld", i);
            }
        }

        /* Modify pressure / temperature / humidity */
        cpl_table_set(profile, MF_COL_ATM_PRE, i, p * (1 + dpres * (1 - frac)));
        if (hgt < elevation) {
            /* Constant values below gelev improve LBLRTM stability */
            cpl_table_set(profile, MF_COL_ATM_TEM, i, gtemp + MF_CONV_T_2_K);
            if (exh2o) {
                cpl_table_set(profile, MF_MOLECULES_H2O, i, ghum);
            }
        }
        else {
            cpl_table_set(profile, MF_COL_ATM_TEM, i, t * (1 + dtemp * (1 - frac)));
            if (exh2o) {
                cpl_table_set(profile, MF_MOLECULES_H2O, i, ppmv * (1 + dhum * (1 - frac)));
            }
        }
    }

    return cpl_error_get_code();
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Scales the water vapor profile of an atmospheric profile table.
 *
 * @param profile            input/ouput: atmospheric profile with scaled water vapor if requested.
 * @param params             mf_parameters parameter structure.
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - Error in subroutine (see subroutines).
 *
 * @description Scales the water vapor profile of an atmospheric profile table to a
 *              PWV value in mm given by the parameter pwv. If pwv is set to -1,
 *              no scaling is performed..
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code mf_atm_scale_to_pwv(cpl_table *profile, const mf_parameters *params)
{
    /* Only apply scaling if it is requested (PWV > 0) */
    double outpwv = params->config->atmospheric.pwv;
    if (outpwv > 0) {
        /* Get altitude of observer */
        double elevation = params->config->ambient.elevation.value * MF_CONV_M_2_KM;

        /* Calculate PWV for input profile : Return if no H2O column exists (PWV = 0) */
        double inpwv = 0.;
        mf_atm_calpwv(&inpwv, profile, &elevation);
        if (inpwv == 0.) {
            return CPL_ERROR_NONE;
        }

        /* Scale H2O profile */
        cpl_table_multiply_scalar(profile, MF_MOLECULES_H2O, outpwv / inpwv);
    }

    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Calculates water vapor column pwv in [mm] from profile in [ppmv].
 *
 * @param pwv                Water vapor column in [mm].
 * @param prof               Atmospheric profile.
 * @param geoelev            Geo elevation of observing site in [km].
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *
 * @description The input profiles are interpreted as functions.
 *              The starting height is the altitude of the observing site.
 *              The profile information is read from the output profiles file.
 *
 */
/* ---------------------------------------------------------------------------*/
static void mf_atm_calpwv(double *pwv, cpl_table *prof, const double *geoelev)
{
    /* Default water vapor column */
    *pwv = 0.;

    /* Check for H2O column in profile table and return if not present */
    if (cpl_table_has_column(prof, MF_MOLECULES_H2O) == 1) {
        /* Number of layers, initialisation of upper layer height, and total column height */
        cpl_size nlayer = cpl_table_get_nrow(prof);
        double   uhgt   = cpl_table_get(prof, MF_COL_ATM_HGT, 0, NULL);

        /* Sum up water vapor columns of all layers */
        for (cpl_size i = 0; i < nlayer - 1; i++) {
            /* Lower and upper limit of layer */
            double lhgt = uhgt;

            uhgt = cpl_table_get(prof, MF_COL_ATM_HGT, i + 1, NULL);

            /* Skip layers below height of observing site */
            if (uhgt <= *geoelev) {
                continue;
            }

            /* Thickness of layer in m */
            double dlayer;
            if (*pwv == 0. && uhgt > *geoelev) {
                dlayer = (uhgt - *geoelev) * MF_CONV_KM_2_M;
            }
            else {
                dlayer = (uhgt - lhgt) * MF_CONV_KM_2_M;
            }

            /* Average pressure for layer */
            double pressure =
                (cpl_table_get(prof, MF_COL_ATM_PRE, i, NULL) + cpl_table_get(prof, MF_COL_ATM_PRE, i + 1, NULL)) / 2;

            /* Average temperature for layer */
            double temp =
                (cpl_table_get(prof, MF_COL_ATM_TEM, i, NULL) + cpl_table_get(prof, MF_COL_ATM_TEM, i + 1, NULL)) / 2;
            if (temp <= 0) {
                temp = MF_TOL;
            }


            /* Average ppmv of H2O for layer */
            double molcol =
                (cpl_table_get(prof, MF_MOLECULES_H2O, i, NULL) + cpl_table_get(prof, MF_MOLECULES_H2O, i + 1, NULL)) /
                2;


            /*** H2O column in mm ***/

            /* Column height [m] for layer */
            double ch = 1e-6 * molcol * dlayer;

            /* Number of mols per unit area [mol m^-2] for layer */
            double nmol = ch * pressure * MF_CONV_MBAR_2_PA / (MF_R * temp);

            /* Mass per unit area [kg m^-2] */
            *pwv += nmol * MF_MOL_MASS_H2O;
        }
    }
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Convert relative humidity to ppmv.
 *
 * @param ppmv               output: Volume mixing ratio in ppmv.
 * @param tem                Temperature in [Kelvin].
 * @param p                  Pressure in [mbar].
 * @param hum                Relative humidity in [%].
 *
 * @description A single relative humidity value is converted to a volume mixing ratio
 *              using the prescription by Murphy and Koop, Review of the vapor
 *              pressure of ice and supercooled water for atmospheric applications,
 *              Q. J. R. Meteorol. Soc (2005), 131, pp. 1539-1565, for a given
 *              temperature and pressure.
 *
 * @note Valid Temperature range is 123K - 332K.
 *
 */
/* ---------------------------------------------------------------------------*/
static void mf_atm_rhum_to_ppmv(double *ppmv, const double *tem, const double *p, const double *hum)
{
    /* Ensure that temperature is not absolute zero */
    double t;
    if (*tem <= 0.) {
        t = DBL_MIN * 2;
        cpl_msg_warning(cpl_func, "(mf_atm       ) Temperatures must be larger than 0K");
    }
    else {
        t = *tem;
    }

    double logt = log(t);

    /* Vapor pressure over water */
    double log_ew = 54.842763 - 6763.22 / t - 4.210 * logt + 0.000367 * t +
                    tanh(0.0415 * (t - 218.8)) * (53.878 - 1331.22 / t - 9.44523 * logt + 0.014025 * t);

    /* Vapor pressure over hexagonal ice */
    double log_ei = 9.550426 - 5723.265 / t + 3.53068 * logt - 0.00728332 * t;

    /* Vapor pressure */
    double log_e = CPL_MIN(log_ew, log_ei);

    /* Water vapor pressure : Avoid overflow*/
    double p_sat;
    if (log_e < log(DBL_MAX)) {
        p_sat = exp(log_e) / 100.;
    }
    else {
        p_sat = DBL_MAX / 100.;
        cpl_msg_warning(cpl_func, "(mf_atm       ) Temperature value results in overflow");
    }

    /* Partial pressure of water in [hPa] */
    double p_h2o = CPL_MIN(*hum, 100) / 100. * p_sat;

    /* Calculate volume mixing ratio in ppmv */
    *ppmv = CPL_MAX(p_h2o / *p * 1e6, 0);
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Convert ppmv to relative humidity.
 *
 * @param hum                output: Relative humidity in [%].
 * @param tem                Temperature in [Kelvin].
 * @param p                  Pressure in [mbar].
 * @param ppmv               Volume mixing ratio in ppmv.
 *
 * @description A single volume mixing ratio is converted to a relative humidity value
 *              using the prescription by Murphy and Koop, Review of the vapor
 *              pressure of ice and supercooled water for atmospheric applications,
 *              Q. J. R. Meteorol. Soc (2005), 131, pp. 1539-1565, for a given
 *              temperature and pressure.
 *
 * @note Valid Temperature range is 123K - 332K.
 *
 */
/* ---------------------------------------------------------------------------*/
static void mf_atm_ppmv_to_rhum(double *hum, const double *tem, const double *p, const double *ppmv)
{
    /* Ensure that temperature is not absolute zero */
    double t;
    if (*tem <= 0.) {
        t = DBL_MIN * 2;
        cpl_msg_warning(cpl_func, "(mf_atm       ) Temperatures must be larger than 0K");
    }
    else {
        t = *tem;
    }

    double logt = log(t);

    /* Vapor pressure over water */
    double log_ew = 54.842763 - 6763.22 / t - 4.210 * logt + 0.000367 * t +
                    tanh(0.0415 * (t - 218.8)) * (53.878 - 1331.22 / t - 9.44523 * logt + 0.014025 * t);

    /* Vapor pressure over hexagonal ice */
    double log_ei = 9.550426 - 5723.265 / t + 3.53068 * logt - 0.00728332 * t;

    /* Vapor pressure */
    double log_e = CPL_MIN(log_ew, log_ei);

    /* Water vapor pressure : Avoid overflow*/
    double p_sat;
    if (log_e < log(DBL_MAX)) {
        p_sat = exp(log_e) / 100.;
    }
    else {
        p_sat = DBL_MAX / 100.;
        cpl_msg_warning(cpl_func, "(mf_atm       ) Temperature value results in overflow");
    }

    /* Partial pressure of water in [hPa] */
    double p_h2o = CPL_MAX(1e-6 * *ppmv * *p, 0);

    /* Calculate realive humidity in [%] */
    *hum = 100. * CPL_MIN(p_h2o / p_sat, 1.);
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Fill vector with values from cpl_table.
 *
 * @param v                  value array
 * @param nrows              number of rows in tab
 * @param tab                CPL table
 * @param str                column identifier
 *
 * @description Reads a CPL table column into a double array.
 *
 */
/* ---------------------------------------------------------------------------*/
static void mf_atm_fill_vector(double *v, const cpl_size nrows, const cpl_table *tab, const char *str)
{
    double *p = v;
    for (int i = 0; i < nrows; i++, p++) {
        *p = cpl_table_get(tab, str, i, NULL);
    }
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Interpolate columns from two CPL_TABLE.
 *
 * @param in_tab             Input CPL table
 * @param in_str_x           Identifier string for x-column in in_tab
 * @param in_str_y           Identifier string for y-column in in_tab
 * @param out_tab            out: CPL table
 * @param out_str_x          Identifier string for x-column in out_tab
 * @param out_str_y          Identifier string for y-column in out_tab
 * @param start              Starting row in output table
 * @param end                Ending row in output table + 1
 *
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - CPL_ERROR_DIVISION_BY_ZERO
 *
 * @description Interpolate two pairs of CPL_TABLE columns.
 *              The input and output columns are specified by their identifier strings.
 *              The starting and ending row numbers are additional input row numbers.
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code mf_atm_interpolate_linear_column(
    const cpl_table *in_tab,
    const char      *in_str_x,
    const char      *in_str_y,
    cpl_table       *out_tab,
    const char      *out_str_x,
    const char      *out_str_y,
    const cpl_size   start,
    const cpl_size   end
)
{
    /* Check inputs */
    if (end <= 0) {
        return CPL_ERROR_NONE;
    }

    cpl_size in_nrows  = cpl_table_get_nrow(in_tab);
    cpl_size out_nrows = cpl_table_get_nrow(out_tab);

    /* Memory reservation for interpolation pointers */
    double *xr = cpl_calloc(in_nrows, sizeof(double));
    double *yr = cpl_calloc(in_nrows, sizeof(double));
    double *xo = cpl_calloc(out_nrows, sizeof(double));
    double *yo = cpl_calloc(out_nrows, sizeof(double));

    /* Fill interpolation variables */
    mf_atm_fill_vector(xr, in_nrows, in_tab, in_str_x);
    mf_atm_fill_vector(yr, in_nrows, in_tab, in_str_y);
    mf_atm_fill_vector(xo, out_nrows, out_tab, out_str_x);

    /* Set limits */
    cpl_size imin = (start < 0) ? 0 : start;
    cpl_size imax = (end > out_nrows) ? out_nrows : end;

    /* Initialize */
    double *p = yo;
    for (cpl_size i = 0; i < imax; i++, p++) {
        *p = 0;
    }

    /* Call linear interpolation */
    cpl_error_code err_code = mf_atm_interpolate_linear(xo, yo, imax, xr, yr, in_nrows);

    /* Storage interpolate values */
    for (cpl_size i = imin; i < imax; i++) {
        cpl_table_set(out_tab, out_str_y, i, *(yo + i));
    }

    /* Cleanup */
    cpl_free(xr);
    cpl_free(yr);
    cpl_free(xo);
    cpl_free(yo);

    return err_code;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Logarithmic linear interpolation.
 *
 * @param x                  X-value for interpolation.
 * @param profile            Reference table.
 * @param start              Index of lower value in profile surrounding x.
 * @param column1            Column-id of y-values for interpolation.
 * @param column2            The x-values reference column.
 *
 * @return double            Interpolated Y-value.
 *
 * @description This routine reads the two values supposedly surrounding x,
 *              as defined by the starting index start and logarithmically
 *              interpolates the value at x. The values, which are to be
 *              interpolated are taken from profile.
 *
 */
/* ---------------------------------------------------------------------------*/
static double mf_atm_interpolate_linear_log_column(
    const double   *x,
    cpl_table      *profile,
    const cpl_size  start,
    const char     *column1,
    const char     *column2
)
{
    /* Interpolate pressure value at gelev */
    double xref[2] = { 0, 0 };
    double yref[2] = { 0, 0 };
    for (int j = 0; j < 2; j++) {
        yref[j] = cpl_table_get_double(profile, column1, start - 1 + j, NULL);

        /* ensure that yref >0 */
        if (yref[j] <= 0.) {
            yref[j] = DBL_MIN * 2;
            cpl_msg_warning(cpl_func, "(mf_interpolat) %s must be larger than 0", column1);
        }

        yref[j] = log(yref[j]);

        xref[j] = cpl_table_get_double(profile, column2, start - 1 + j, NULL);
    }

    /* Call linear interpolation */
    double y = 0;
    mf_atm_interpolate_linear(x, &y, 1, xref, yref, 2);

    /* Avoid overflow */
    double result = (y >= DBL_MAX_EXP) ? DBL_MAX : exp(y);

    return result;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Linear interpolation.
 *
 * @param x_out              Desired output x-spacing
 * @param y_out              Desired output y-spacing
 * @param n_out              Length of x_out / y_out
 * @param x_ref              Reference x-spacing
 * @param y_ref              Reference y-values at x_ref
 * @param n_ref              Length of x_ref / y_ref
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - CPL_ERROR_DIVISION_BY_ZERO.
 *
 * @description This function calculates the interpolated y-values y_out at the
 *              positions x_out with respect to the reference x/y pairs (x_ref / y_ref).
 *
 * @note        Points outside the range of the reference vectors will be extrapolated based
 *              on the first / last values in the reference vectors for the low / high end, respectively.
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code mf_atm_interpolate_linear(
    const double *x_out,
    double       *y_out,
    const long    n_out,
    const double *x_ref,
    const double *y_ref,
    const long    n_ref
)
{
    /* Pointers for looping through the reference arrays */
    const double *xr = x_ref;
    const double *yr = y_ref;
    const double *xo = x_out;
    double       *yo = y_out;

    /* At least one value has a divide by zero */
    int divide_by_zero = 0;

    /* Counter for length of output & reference array */
    long ref = 0;

    for (long out = 0; out < n_out; out++, xo++, yo++) {
        /* Find first element in ref that is larger than current out and ensure that ref does not overshoot valid range  */
        while (*xr < *xo && ref < n_ref) {
            xr++;
            yr++;
            ref++;
        }

        /* If current out is larger than ref go one step back */
        if (xr - x_ref > 0) {
            xr--;
            yr--;
            ref--;
        }

        /* if out is larger than last ref use last two points */
        if (ref == n_ref - 1) {
            xr = x_ref + n_ref - 2;
            yr = y_ref + n_ref - 2;
        }

        /* Calculate linear interpolation */
        if (*(xr + 1) - *xr == 0) {
            *yo            = *yr;
            divide_by_zero = 1;
        }
        else {
            *yo = (*xo - *xr) * (*(yr + 1) - *yr) / (*(xr + 1) - *xr) + *yr;
        }
    }

    if (divide_by_zero == 1) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_DIVISION_BY_ZERO, "Duplicate x-values");
    }
    else {
        return CPL_ERROR_NONE;
    }
}

/** @endcond */


/**@}*/
