/*
 * This file is part of the ESO Telluric Correction Library
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "mf_io.h"

#include "mf_gdas.h"

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/

/* Check local GDAS database for two closest in time data files */
static cpl_boolean mf_gdas_extract_files(
    const char *gdas_file_search,
    const char *gdas_path_tarball,
    const char *local_gdas_path,
    double      lon,
    double      lat,
    int         year,
    int         month,
    int         day,
    int         utch,
    cpl_array  *timestamps,
    cpl_array  *gdas_files
);

/* Hour closest in the file */
static cpl_boolean mf_gdas_closest_file(
    char      **gdas_file,
    double     *utc,
    const char *global_gdas_tarball,
    const char *local_gdas_path,
    double      lon,
    double      lat,
    int         year,
    int         month,
    int         day,
    int         utch,
    cpl_boolean positive_direction
);

/* Convert Gregorian to Julian date */
static cpl_error_code mf_gdas_greg_to_julian(long *jd, int year, int month, int day);

/* This function converts a Julian date into a sequence of year/month/day */
static cpl_error_code mf_gdas_julian_to_greg(int *year, int *month, int *day, long jd);

/* Attempt to update gdas tarball from eso ftp server : Not send error in any case */
static cpl_error_code
mf_gdas_update_db(const char *gdas_tarball_name, const char *global_gdas_tarball, const char *local_gdas_tarball);

/* Interpolate profiles linearly between two points in time */
static cpl_table *
mf_gdas_interpolate_profile(const cpl_table *profile1, const cpl_table *profile2, const cpl_array *timestamps);

/*----------------------------------------------------------------------------*/
/**
 *                 Functions
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup mf_gdas   Manage the GDAS files
 *
 * @brief
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/* ---------------------------------------------------------------------------*/
/**
 * @brief Check user GDAS input cpl_table
 *
 * @param table              .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_error_code mf_gdas_check(const cpl_table *table)
{
    cpl_ensure(table, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

    cpl_error_ensure(
        cpl_table_has_column(table, MF_COL_GDAS_RELHUM) != 0 && cpl_table_has_column(table, MF_COL_GDAS_HEIGHT) != 0 &&
            cpl_table_has_column(table, MF_COL_GDAS_PRESS) != 0 && cpl_table_has_column(table, MF_COL_GDAS_TEMP) != 0 &&
            cpl_table_get_ncol(table) == 4 && cpl_table_get_nrow(table) > 0,
        CPL_ERROR_INCOMPATIBLE_INPUT, return CPL_ERROR_INCOMPATIBLE_INPUT,
        "cpl_table *%s does not have the correct columns [%s, %s, %s, %s]", MF_INPUT_GDAS, MF_COL_GDAS_RELHUM,
        MF_COL_GDAS_HEIGHT, MF_COL_GDAS_PRESS, MF_COL_GDAS_TEMP
    );

    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Gets the two ESO GDAS DB profiles that are closest in requested time.
 *
 * @param params             mf_parameters parameter structure.
 * @param gdas_profile1      output: CPL table with first GDAS profile.
 * @param gdas_profile2      output: CPL table with second GDAS profile.
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - Illegal input.
 *                           - File I/O.
 *                           - Unexpected file structure.
 *                           - Error in subroutine (see subroutines).
 *
 * @note The profiles are either taken from a local TAR archive or downloaded
 *              from a web server if required. If even the latter fails, an average
 *              profile is provided. The routines also provides the time stamps for the
 *              two output GDAS profiles and the requested time.
 */
/* ---------------------------------------------------------------------------*/
cpl_table *
mf_gdas(const mf_parameters *params, cpl_table **gdas_profile1, cpl_table **gdas_profile2, cpl_boolean *average_used)
{
    /* Get UTC in h : time of observation */
    double utch = params->config->ambient.utc.value / 3600.;

    /*** Decompose UTC into h, m, and s ***/
    //char *hh = "00";  /* minute */
    //char *mm = "00";  /* minute */
    //char *ss = "00";  /* second */

    long h = (long)utch;
    long m = (long)((utch - (double)h) * 60.);
    long s = (long)(((utch - (double)h) * 60. - (double)m) * 60.);

    /* splitting single check to 3 dedicated checks - one for h, one for m and
     * one for s as recent compiler report problems in debug mode */
    cpl_error_ensure(h >= 0 && h <= 24, CPL_ERROR_ILLEGAL_INPUT, return NULL, "UTC (h) out of range");
    cpl_error_ensure(m >= 0 && m <= 60, CPL_ERROR_ILLEGAL_INPUT, return NULL, "UTC (min) out of range");
    cpl_error_ensure(s >= 0 && s <= 60, CPL_ERROR_ILLEGAL_INPUT, return NULL, "UTC (sec) out of range");

    /* Write h:m:s */
    //char *hh = cpl_sprintf("%02ld", h);
    //char *mm = cpl_sprintf("%02ld", m);
    //char *ss = cpl_sprintf("%02ld", s);

    /* Get observing date and longitude/latitude (geolocation of telescope) */
    double observing_date = params->config->ambient.observing_date.value;
    double lon            = params->config->ambient.longitude.value;
    double lat            = params->config->ambient.latitude.value;

    /* YYYYMMDD */
    int year, month, day;
    mf_gdas_frac_year_to_date(&year, &month, &day, NULL, NULL, NULL, &observing_date);

    /* Local variables */
    char *global_gdas_path =
        cpl_sprintf("%s/%s", params->config->directories.telluriccorr_data_path, MF_PROFILES_GDAS_PATH);
    char *local_gdas_path = cpl_sprintf("%s", params->config->internal.tmp_folder);

    /* Name of tarball and paths that contain all of GDAS files and the file to search (i.e. Paranal: gdas_profiles_C-70.4-24.6.tar.gz) */
    char *gdas_tarball_name   = cpl_sprintf("gdas_profiles_C%+.1f%+.1f.tar.gz", lon, lat);
    char *global_gdas_tarball = cpl_sprintf("%s/%s", global_gdas_path, gdas_tarball_name);
    char *local_gdas_tarball  = cpl_sprintf("%s/%s", local_gdas_path, gdas_tarball_name);
    char *gdas_file_search    = cpl_sprintf("C%+.1f%+.1fD%4i-%02i-%02iT%g", lon, lat, year, month, day, utch);

    /* Build GDAS file names and time stamps */
    cpl_array *timestamps = cpl_array_new(3, CPL_TYPE_DOUBLE);
    cpl_array_set_double(timestamps, 0, utch);
    cpl_array_set_double(timestamps, 1, utch);
    cpl_array_set_double(timestamps, 2, utch);

    cpl_array *gdas_files = cpl_array_new(2, CPL_TYPE_STRING);

    //if GDAS_PROF is NONE/none, then only consider the average file corresponding to the month of the observation.
    if (!strcmp(params->config->atmospheric.gdas_prof, "none") ||
        !strcmp(params->config->atmospheric.gdas_prof, "NONE")) {
        /* No GDAS profile was found by grib -> Retrieve best matching profile from library */
        int lmonth = (month % 6) + 1;

        char *gdas_default_file = cpl_sprintf(
            "%s/%s/GDAS_t0_s%1i.fits", params->config->directories.telluriccorr_data_path, MF_PROFILES_LIB_PATH, lmonth
        );

        *average_used = CPL_TRUE;
        cpl_msg_warning(
            cpl_func,
            "(mf_gdas      ) It could not find GDAS profile on server -> Using average profile by telluriccorr data : "
            "%s",
            gdas_default_file
        );
        *gdas_profile1 = cpl_table_load(gdas_default_file, 1, 0);

        if (!(*gdas_profile1)) {
            cpl_error_set_message(cpl_func, CPL_ERROR_FILE_IO, "Could not load profile: %s", gdas_default_file);
            cpl_free(gdas_default_file);
            cpl_array_delete(gdas_files);
            return NULL;
        }

        /* Cleanup */
        cpl_free(gdas_default_file);

        /* Replace gdas_profile */
        cpl_table_delete(*gdas_profile2);
        *gdas_profile2 = cpl_table_duplicate(*gdas_profile1);
    }
    else {
        /* Extract GDAS files from tarball */
        /* TODO: Why is utch passed here and not h? */
        cpl_boolean found = mf_gdas_extract_files(
            gdas_file_search, global_gdas_tarball, local_gdas_path, lon, lat, year, month, day, (int)utch, timestamps,
            gdas_files
        );

        if (!found) {
            /* Attempt to update the ESO GDAS DB to the latest version on the ftp site */
            /* Failing that extract again so as to use an interpolation                */

            cpl_msg_info(
                cpl_func,
                "(mf_gdas      ) Not found valid GDAS files from the input image [%s] in the local ESO GDAS DB",
                gdas_file_search
            );

            const char *gdas_path_tarball;

            /* Attempt to find an update via ftp but annul any errors from the stack trace if the ftp fails */
            cpl_errorstate err_state  = cpl_errorstate_get(); /* Store current error state*/
            cpl_error_code update_err = mf_gdas_update_db(gdas_tarball_name, global_gdas_tarball, local_gdas_tarball);
            cpl_errorstate_set(err_state); /* Reset error state        */

            /* If update was successful then we will use this */
            if (!update_err) {
                gdas_path_tarball = global_gdas_tarball;
            }
            else {
                gdas_path_tarball = local_gdas_tarball;
            }


            /* Try to extract again --> Not throw any error in any case */
            mf_gdas_extract_files(
                gdas_file_search, gdas_path_tarball, local_gdas_path, lon, lat, year, month, day, (int)utch, timestamps,
                gdas_files
            );
        }

        /* Cleanup */
        cpl_free(gdas_file_search);
        cpl_free(local_gdas_tarball);
        cpl_free(global_gdas_tarball);
        cpl_free(gdas_tarball_name);
        cpl_free(local_gdas_path);
        cpl_free(global_gdas_path);

        /* Save error state */
        cpl_errorstate err_state = cpl_errorstate_get();

        /* Read GDAS profiles */
        *gdas_profile1 = NULL;
        if (cpl_array_get_string(gdas_files, 0) != NULL) {
            *gdas_profile1 = mf_io_read_gdas_file_and_create_table(cpl_array_get_string(gdas_files, 0), "m");
        }

        *gdas_profile2 = NULL;
        if (cpl_array_get_string(gdas_files, 1) != NULL) {
            *gdas_profile2 = mf_io_read_gdas_file_and_create_table(cpl_array_get_string(gdas_files, 1), "m");
        }

        /* Restore error state */
        cpl_errorstate_set(err_state);

        if (!(*gdas_profile1) && *gdas_profile2) {
            /* Any profile exist! */
            cpl_msg_warning(
                cpl_func,
                "(mf_gdas      ) It could not find GDAS profile before MDJ-OBS on the server -> using first after"
            );
            *average_used  = CPL_TRUE;
            *gdas_profile1 = cpl_table_duplicate(*gdas_profile2);
        }
        else if (*gdas_profile1 && !(*gdas_profile2)) {
            /* Any profile exist! */
            cpl_msg_warning(
                cpl_func,
                "(mf_gdas      ) It could not find GDAS profile after MDJ-OBS on the server -> using first before"
            );
            *average_used  = CPL_TRUE;
            *gdas_profile2 = cpl_table_duplicate(*gdas_profile1);
        }
        else if (!(*gdas_profile1) && !(*gdas_profile2)) {
            /* No GDAS profile was found by grib -> Retrieve best matching profile from library */
            int lmonth = (month % 6) + 1;

            char *gdas_default_file = cpl_sprintf(
                "%s/%s/GDAS_t0_s%1i.fits", params->config->directories.telluriccorr_data_path, MF_PROFILES_LIB_PATH,
                lmonth
            );

            *average_used = CPL_TRUE;
            cpl_msg_warning(
                cpl_func,
                "(mf_gdas      ) It could not find GDAS profile on server -> Using average profile by telluriccorr "
                "data : %s",
                gdas_default_file
            );
            *gdas_profile1 = cpl_table_load(gdas_default_file, 1, 0);

            if (!(*gdas_profile1)) {
                cpl_error_set_message(cpl_func, CPL_ERROR_FILE_IO, "Could not load profile: %s", gdas_default_file);
                cpl_free(gdas_default_file);
                cpl_array_delete(gdas_files);
                return NULL;
            }


            /* Cleanup */
            cpl_free(gdas_default_file);

            /* Replace gdas_profile */
            cpl_table_delete(*gdas_profile2);
            *gdas_profile2 = cpl_table_duplicate(*gdas_profile1);
        }

    }  //end if GDAS_PROF != NONE/none

    /* Get elevation of telescope in m */
    double elevation = params->config->ambient.elevation.value;

    /* Compare lowest height of GDAS profiles and geoelevation */
    if (cpl_table_get(*gdas_profile1, MF_COL_GDAS_HEIGHT, 0, NULL) > elevation) {
        const char *gdas_file = cpl_array_get_string(gdas_files, 0);
        cpl_array_delete(gdas_files);
        cpl_error_set_message(
            cpl_func, CPL_ERROR_BAD_FILE_FORMAT,
            "Unexpected file structure: %s (lowest height %g > geoelev %g of mf_parameters *params)", gdas_file,
            cpl_table_get(*gdas_profile1, MF_COL_GDAS_HEIGHT, 0, NULL), elevation
        );
        return NULL;
    }

    if (cpl_table_get(*gdas_profile2, MF_COL_GDAS_HEIGHT, 0, NULL) > elevation) {
        const char *gdas_file = cpl_array_get_string(gdas_files, 1);
        cpl_array_delete(gdas_files);
        cpl_error_set_message(
            cpl_func, CPL_ERROR_BAD_FILE_FORMAT,
            "Unexpected file structure: %s (lowest height %g > geoelev %g of mf_parameters *params)", gdas_file,
            cpl_table_get(*gdas_profile2, MF_COL_GDAS_HEIGHT, 0, NULL), elevation
        );
        return NULL;
    }

    /* Cleanup */
    cpl_array_delete(gdas_files);

    /* Interpolate profiles linearly between two points in time */
    cpl_table *gdas_interpolate = mf_gdas_interpolate_profile(*gdas_profile1, *gdas_profile2, timestamps);
    cpl_array_delete(timestamps);

    return gdas_interpolate;
}

/** @cond PRIVATE */

/* ---------------------------------------------------------------------------*/
/**
 * @brief Brake up a fractional year into year, month, day.
 *
 * @param year               output: Year.
 * @param month              output: Month.
 * @param day                output: Day.
 * @param hh                 output: Hour.
 * @param mm                 output: Minute.
 * @param ss                 output: Second.
 * @param fracyear           Fractional year (e.g. 2002.345).
 *
 * @return double            Fractional day (optionally broken up into hh/mm/ss).
 *
 * @description This routine brakes up a fractional year into year, month, day,
 *              e.g. 2002.380821917808219 corresponds to 20/05/2002, optionally
 *              including the time of day.
 *              The return value is the rest after braking up the date,
 *              i.e. the time of day.
 *
 */
/* ---------------------------------------------------------------------------*/
double mf_gdas_frac_year_to_date(int *year, int *month, int *day, int *hh, int *mm, double *ss, const double *fracyear)
{
    /* Calculate the year */
    *year = floor(*fracyear);

    /* Calculate remaining fractional year */
    double rest = *fracyear - *year;

    /* Corresponds to an accuracy of 0.01s */
    double acc = 1e-10;

    /* Round to desired accuracy */
    rest = floor(rest / acc + 1) * acc;

    int days_in_year    = 365;
    int days_in_month[] = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

    if (*year % 4 == 0 && *year % 100 != 0) {
        /* Leap year (February has 29 days) */
        days_in_month[1]++;
        days_in_year++;
    }

    /* Calculate remaining fractional days */
    rest *= days_in_year;

    /* Count months and subtract days from total */
    int i = 0;
    for (; i < 12 && (rest - days_in_month[i]) >= 0; i++) {
        rest -= days_in_month[i];
    }

    /* Set month */
    *month = i + 1;

    /* Calculate days */
    *day = (int)floor(rest) + 1;

    /* Calculate remaining fractional day */
    rest -= *day - 1;

    /* If requested calculate time hh:mm:ss */
    if (hh != NULL) {
        rest *= 24;
        *hh = floor(rest);
        rest -= *hh;

        rest *= 60;
        *mm = floor(rest);
        rest -= *mm;

        rest *= 60;
        *ss = rest;
    }

    double frac_day = rest;

    return frac_day;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Check local GDAS database for two closest in time data files.
 *
 * @param gdasdir            Path of the GDAS DB.
 * @param lon                Longitude.
 * @param lat                Latitude.
 * @param year               Year.
 * @param month              Month.
 * @param day                Day.
 * @param utch               Hour in UTC.
 * @param timestamps         output: Updated array of time stamps of found files.
 * @param gdas_files         output: Updated array of found filenames, empty on failure.
 *
 * @return error code        1 on success, 0 on failure.
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_boolean mf_gdas_extract_files(
    const char  *gdas_file_search,
    const char  *gdas_path_tarball,
    const char  *local_gdas_path,
    const double lon,
    const double lat,
    const int    year,
    const int    month,
    const int    day,
    int          utch,
    cpl_array   *timestamps,
    cpl_array   *gdas_files
)
{
    /* Check if exist the *.tar.gz */
    cpl_error_code err = access(gdas_path_tarball, F_OK);
    if (err != CPL_ERROR_NONE) {
        return CPL_FALSE;
    }

    /* Get first GDAS file */
    double      utc_1;
    char       *gdas_file1          = NULL;
    cpl_boolean positive_direction1 = CPL_FALSE;
    cpl_boolean check1              = mf_gdas_closest_file(
        &gdas_file1, &utc_1, gdas_path_tarball, local_gdas_path, lon, lat, year, month, day, utch, positive_direction1
    );

    /* Get second GDAS file */
    double      utc_2;
    char       *gdas_file2          = NULL;
    cpl_boolean positive_direction2 = CPL_TRUE;
    cpl_boolean check2              = mf_gdas_closest_file(
        &gdas_file2, &utc_2, gdas_path_tarball, local_gdas_path, lon, lat, year, month, day, utch + 1,
        positive_direction2
    );

    /* Check GDAS files */
    cpl_boolean found = (check1 && check2);
    if (found) {
        cpl_msg_info(
            cpl_func, "(mf_gdas      ) Search the GDAS files more closest to the image [%s] :", gdas_file_search
        );
        cpl_msg_info(cpl_func, "(mf_gdas      )   Local ESO GDAS DB [%s]", gdas_path_tarball);
        cpl_msg_info(cpl_func, "(mf_gdas      )   1. %s", gdas_file1);
        cpl_msg_info(cpl_func, "(mf_gdas      )   2. %s", gdas_file2);

        cpl_array_set_double(timestamps, 0, utc_1);
        cpl_array_set_double(timestamps, 1, utc_2);

        if (cpl_array_get_double(timestamps, 1, NULL) < cpl_array_get_double(timestamps, 0, NULL)) {
            cpl_array_set_double(timestamps, 1, cpl_array_get_double(timestamps, 1, NULL) + 24.);
        }

        cpl_array_set_string(gdas_files, 0, gdas_file1);
        cpl_array_set_string(gdas_files, 1, gdas_file2);
    }

    /* Cleanup */
    if (gdas_file1) {
        cpl_free(gdas_file1);
    }
    if (gdas_file2) {
        cpl_free(gdas_file2);
    }

    return found;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Hour closest in the file.
 *
 * @param filestr            File name including path.
 * @param found_hour         output: Hour found.
 * @param gdasdir            Path of the GDAS DB.
 * @param lon                Longitude.
 * @param lat                Latitude.
 * @param year               Year.
 * @param month              Month.
 * @param day                Day.
 * @param utch               Hour in UTC.
 * @param direction          Direction of the position.
 *
 * @return error code        1 on success, 0 on failure.
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_boolean mf_gdas_closest_file(
    char       **gdas_file,
    double      *utc,
    const char  *global_gdas_tarball,
    const char  *local_gdas_path,
    const double lon,
    const double lat,
    const int    year,
    const int    month,
    const int    day,
    int          utch,
    cpl_boolean  positive_direction
)
{
    /* Check inputs */
    cpl_error_ensure(!(*gdas_file), CPL_ERROR_ILLEGAL_INPUT, return CPL_FALSE, "input GDAS file name not null");

    int direction = positive_direction ? 1 : -1;

    /* Search the GDAS file more near to the image hour (Every GDAS file cover 3h) */
    cpl_error_code err;
    cpl_boolean    found = CPL_FALSE;
    for (int i = 0; i < MF_GDAS_FILE_INVERVAL * MF_GDAS_FILE_SECURITY_SEARCH && !found; i++) {
        int lyear  = year;
        int lmonth = month;
        int lday   = day;

        /* Adjust UTC time for the name in next GDAS file to search */
        if (utch < 0 || utch > 23) {
            long tmpday;

            err = mf_gdas_greg_to_julian(&tmpday, year, month, day);
            if (err != CPL_ERROR_NONE) {
                cpl_error_set_message(cpl_func, err, "Error convert data from Gregorian to Julian");
                return CPL_FALSE;
            }

            tmpday += direction;

            err = mf_gdas_julian_to_greg(&lyear, &lmonth, &lday, tmpday);
            if (err != CPL_ERROR_NONE) {
                cpl_error_set_message(cpl_func, err, "Error convert data from Julian to Gregorian");
                return CPL_FALSE;
            }

            utch += -24 * direction;
        }

        /* GDAS filename */
        char *gdas_file_extracted =
            cpl_sprintf("C%+.1f%+.1fD%04i-%02i-%02iT%02i.gdas", lon, lat, lyear, lmonth, lday, utch);

        /* Storage filename */
        *gdas_file = cpl_sprintf("%s/%s", local_gdas_path, gdas_file_extracted);

        /* Extract GDAS filename from the 'gdas_tarball' in the temporary directory */
        char *tar_syscall_command =
            cpl_sprintf("tar -C %s -zxf %s %s 2>/dev/null", local_gdas_path, global_gdas_tarball, gdas_file_extracted);
        mf_io_system(tar_syscall_command, NULL, NULL);
        cpl_free(tar_syscall_command);

        /* Check if the GDAS file was download (if it was untar and exist locally). */
        if (access(*gdas_file, F_OK) == CPL_ERROR_NONE) {
            /* Found : Set UTC hour and return */
            cpl_msg_info(cpl_func, "(mf_gdas      ) GDAS file : %s --> Found     in the tarball", gdas_file_extracted);
            found = CPL_TRUE;
            *utc  = utch;
        }
        else {
            /* Not found : Update UTC hour */
            cpl_msg_info(cpl_func, "(mf_gdas      ) GDAS file : %s --> Not found in the tarball", gdas_file_extracted);
            cpl_free(*gdas_file);
            *gdas_file = NULL;
            utch += direction;
        }

        /* Cleanup */
        cpl_free(gdas_file_extracted);
    }

    return found;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Convert Gregorian to Julian date.
 *
 * @param jd                 output: Julian date.
 * @param year               Gregorian year.
 * @param month              Gregorian month.
 * @param day                Gregorian day.
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *
 * @description This function converts a sequence of year/month/day into a Julian date..
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code mf_gdas_greg_to_julian(long *jd, const int year, const int month, const int day)
{
    /* Check inputs */
    cpl_ensure(
        year >= 1 && month >= 1 && month <= 12 && day >= 1 && day <= 31, CPL_ERROR_ILLEGAL_INPUT,
        CPL_ERROR_ILLEGAL_INPUT
    );

    long jd1 = day - 32075 + 1461 * (year + 4800 + (month - 14) / 12) / 4;
    long jd2 = 367 * (month - 2 - (month - 14) / 12 * 12) / 12;
    long jd3 = 3 * ((year + 4900 + (month - 14) / 12) / 100) / 4;

    *jd = jd1 + jd2 - jd3;

    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief This function converts a Julian date into a sequence of year/month/day.
 *
 * @param year               output: Gregorian year.
 * @param month              output: Gregorian month.
 * @param day                output: Gregorian day.
 * @param jd                 Julian date.
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code mf_gdas_julian_to_greg(int *year, int *month, int *day, const long jd)
{
    long l = jd + 68569;
    long n = 4 * l / 146097;

    l -= (146097 * n + 3) / 4;

    long y = 4000 * (l + 1) / 1461001;

    l += -1461 * y / 4 + 31;

    long j = 80 * l / 2447;

    /* Set day */
    *day = (int)(l - 2447 * j / 80);

    l = j / 11;

    /* Set year */
    *year = (int)(100 * (n - 49) + y + l);

    j += 2 - 12 * l;

    /* Set month */
    *month = (int)j;

    /* Check outputs */
    cpl_ensure(
        *year >= 1 && *month >= 1 && *month <= 12 && *day >= 1 && *day <= 31, CPL_ERROR_ILLEGAL_OUTPUT,
        CPL_ERROR_ILLEGAL_OUTPUT
    );

    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Attempt to update gdas tarball from eso ftp server.
 *
 * @param lon                Longitude.
 * @param lat                Latitude.
 *
 * @return error code        CPL_ERROR_NONE or a specific error code in other case.
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code
mf_gdas_update_db(const char *gdas_tarball_name, const char *global_gdas_tarball, const char *local_gdas_tarball)
{
    /* Check the number of entries in the local ESO GDAS DB */
    long nentries_old = -1;

    /* Check if exist the *.tar.gz */
    cpl_error_code err = access(global_gdas_tarball, F_OK);
    if (err != CPL_ERROR_NONE) {
        cpl_msg_info(cpl_func, "(mf_gdas      ) Not local version found of ESO GDAS DB [%s]", global_gdas_tarball);
    }
    else {
        nentries_old = mf_io_tarball_nfiles(global_gdas_tarball);
        cpl_msg_info(
            cpl_func, "(mf_gdas      ) The old version of ESO GDAS DB contained (%ld) files in [%s]", nentries_old,
            global_gdas_tarball
        );
    }

    /* Update local database */
    char *url_gdas_file = cpl_sprintf("%s/%s", MF_GDAS_URL, gdas_tarball_name);
    cpl_msg_info(
        cpl_func, "(mf_gdas      ) Downloading from FTP ESO GDAS DB [ftp://%s%s] ...", MF_GDAS_FTP_ABSOLUTE,
        url_gdas_file
    );
    err = mf_io_curl(MF_GDAS_FTP_ABSOLUTE, url_gdas_file, local_gdas_tarball);
    /*
  char *fake_file = cpl_sprintf("%s.data", gdas_tarball_name);
  err = mf_io_curl(MF_GDAS_FTP_ABSOLUTE, "/pub/dfs/pipelines/gravity/finals2000A.data", fake_file);
  cpl_free(fake_file);
  err = mf_io_curl(MF_GDAS_FTP_ABSOLUTE, "/pub/dfs/pipelines/gravity/finals2000A.data", gdas_tarball_name);
*/
    cpl_free(url_gdas_file);

    if (err != CPL_ERROR_NONE) {
        cpl_msg_warning(cpl_func, "(mf_gdas      ) Download and update the new GDAS data failed.");
        cpl_msg_warning(
            cpl_func, "(mf_gdas      )   GDAS data could not be transferred in their entirety within 10 min."
        );
        cpl_msg_warning(cpl_func, "(mf_gdas      )   Consider to restart the fit to complete the transfer.");
    }
    else {
        /* Report on changes on success */
        long nentries_new = mf_io_tarball_nfiles(local_gdas_tarball);

        if (nentries_new == 0) {
            cpl_msg_info(cpl_func, "(mf_gdas      ) Problems with the download ESO GDAS DB (files entries == 0) !");
            err = CPL_ERROR_FILE_IO;
        }
        else if (nentries_new <= nentries_old) {
            cpl_msg_info(cpl_func, "(mf_gdas      ) Current database is up to date");
            err = CPL_ERROR_ILLEGAL_INPUT;
        }
        else {
            cpl_msg_info(
                cpl_func,
                "(mf_gdas      ) The new version of ESO GDAS DB contains (%ld) new files --> Changing general ESO GDAS "
                "database!",
                nentries_new - CPL_MAX(nentries_old, 0)
            );
            if (mf_io_rm(global_gdas_tarball) != CPL_ERROR_NONE) {
                cpl_msg_warning(
                    cpl_func, "(mf_gdas      ) User without enough permissions to change the general ESO GDAS database!"
                );
                err = CPL_ERROR_FILE_NOT_CREATED;
            }
            else {
                /* Move local tarball to global tarball */
                cpl_msg_info(cpl_func, "(mf_gdas      ) Move : %s -> %s", local_gdas_tarball, global_gdas_tarball);
                err = mf_io_mv(local_gdas_tarball, global_gdas_tarball);
            }
        }
    }

    return err;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Interpolate profiles linearly between two points in time.
 *
 * @param outprofile         out: interpolated output atmospheric profile.
 * @param profile1           first input atmospheric profile.
 * @param profile2           second input atmospheric profile.
 * @param timestamps         time stamps for input and output profiles.
 *
 * @return cpl_error_code    Invalid objects value(s) or structure.
 *
 * @description This function calculates an interpolated profile from two input
 *              profiles taken at different times. profile1 and profile2 are
 *              expected to be sampled over the same geographic elevation grid and to contain
 *              the same data columns (molecules). The output profile outprofile is
 *              linearly interpolated in time. If the number of layers in both profiles
 *              differs, the profile closest to the requested time is taken as output
 *              profile. In this case, a warning message is printed.
 *
 * @note No checks are performed whether the input profiles contain the same column information.
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_table *
mf_gdas_interpolate_profile(const cpl_table *profile1, const cpl_table *profile2, const cpl_array *timestamps)
{
    /* Get time values */
    double t1   = cpl_array_get_double(timestamps, 0, NULL);
    double t2   = cpl_array_get_double(timestamps, 1, NULL);
    double tout = cpl_array_get_double(timestamps, 2, NULL);

    /* Check time values */
    if (t1 > t2 || tout < t1 || tout > t2) {
        cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Invalid object value(s): cpl_array *timestamps (invalid time(s))"
        );
        return NULL;
    }

    /* Ensure that input tables have the same structure */
    if (cpl_table_compare_structure(profile1, profile2) == 1) {
        cpl_error_set_message(
            cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
            "Invalid object structure: cpl_table *profile1 != *profile2 (columns)"
        );
        return NULL;
    }

    /* Get relative position ([0,1]) of output profile time */
    double dt;
    if (t1 == t2) {
        dt = 0;
    }
    else {
        dt = tout - t1;
        dt /= t2 - t1;
    }

    /* Get number of rows in both profiles */
    cpl_size nrow1 = cpl_table_get_nrow(profile1);
    cpl_size nrow2 = cpl_table_get_nrow(profile2);

    /* Check number of rows -> Difference: Do not interpolate and take profile closest to requested time */
    cpl_table *outprofile;
    if (nrow1 != nrow2) {
        cpl_msg_warning(
            cpl_func,
            "(mf_atm       ) GDAS profiles differ in # of rows -> no interpolation -> take profile closest in time"
        );

        if (dt < 0.5) {
            outprofile = cpl_table_duplicate(profile1);
        }
        else {
            outprofile = cpl_table_duplicate(profile2);
        }
    }
    else {
        /* Get number and names of columns */
        cpl_size   ncol = cpl_table_get_ncol(profile1);
        cpl_array *cols = cpl_table_get_column_names(profile1);

        /* Prepare output table structure */
        outprofile = cpl_table_new(1);
        cpl_table_copy_structure(outprofile, profile1);
        cpl_table_set_size(outprofile, nrow1);

        /* Create temporary arrays for column values */
        cpl_array *col1 = cpl_array_new(nrow1, CPL_TYPE_DOUBLE);
        cpl_array *col2 = cpl_array_new(nrow1, CPL_TYPE_DOUBLE);

        /* Interpolate data values of each column depending on time stamps */
        for (cpl_size i = 0; i < ncol; i++) {
            const char *col_name = cpl_array_get_string(cols, i);

            cpl_array_copy_data_double(col1, cpl_table_get_data_double_const(profile1, col_name));
            cpl_array_copy_data_double(col2, cpl_table_get_data_double_const(profile2, col_name));

            cpl_array_subtract(col2, col1);
            cpl_array_multiply_scalar(col2, dt);
            cpl_array_add(col2, col1);

            cpl_table_copy_data_double(outprofile, col_name, cpl_array_get_data_double(col2));
        }

        /* Delete temporary arrays */
        cpl_array_delete(cols);
        cpl_array_delete(col1);
        cpl_array_delete(col2);
    }

    return outprofile;
}

/** @endcond */


/**@}*/
