/*
 * This file is part of the ESO Telluric Correction Library
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "mf_kernel_synthetic.h"

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/

/*  */
static cpl_error_code mf_kernel_synthetic_create_table(
    cpl_table           *kerntab,
    const cpl_table     *spec,
    const mf_parameters *params,
    const cpl_array     *fitpar
);

/*  */
static cpl_error_code mf_kernel_synthetic_calculate(
    cpl_array        *kernel,
    const double      wbox,
    const double      wgauss,
    const double      wlorentz,
    const double      kernfac,
    const cpl_boolean kernmode
);

/*  */
static cpl_error_code mf_kernel_synthetic_boxcar(cpl_array *kernel, const double fwhm);

/*  */
static cpl_error_code
mf_kernel_synthetic_voigt(cpl_array *kernel, const double wgauss, const double wlorentz, const double kernfac);

/*  */
static cpl_error_code mf_kernel_synthetic_gauss(cpl_array *kernel, const double fwhm, const double kernfac);

/*  */
static cpl_error_code mf_kernel_synthetic_lorentz(cpl_array *kernel, const double fwhm, const double kernfac);

/* Convolution of two kernels. The output array has to exist */
static cpl_error_code
mf_kernel_synthetic_convolution(cpl_array *outkernel, const cpl_array *inkernel1, const cpl_array *inkernel2);

/*  */
static cpl_error_code mf_kernel_synthetic_invert_table(cpl_table *kerntab);

/*  */
static cpl_error_code mf_kernel_synthetic_convolve_table(cpl_table *spec, const cpl_table *kerntab);

/* Convolution of a flux array with given kernel for a fixed range of pixels */
static cpl_error_code mf_kernel_synthetic_convolve_window_inv(
    cpl_array       *convflux,
    const cpl_array *flux,
    const int        range[2],
    const cpl_array *kernel,
    const int        kerncen
);

/*----------------------------------------------------------------------------*/
/**
 *                 Functions
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup mf_kernel_synthetic       Tools to generate the input synthetic internal kernel.
 *
 * @brief
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/* ---------------------------------------------------------------------------*/
/**
 * @brief Apply synthetic kernel
 *
 * @param spec               input spectrum (CPL table)
 * @param params             mf_parameters parameter structure
 * @param fitpar             CPL array with fit parameters
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @note Applies synthetic kernels to a model spectrum as provided by a CPL
 *          table with columns MF_COL_LAMBDA and MF_COL_FLUX. Depending on the parameter
 *          varkern, either a constant or a variable kernel is applied in the
 *          case of a synthesis based on a boxcar, Gaussian, and Lorentzian component.
 *          The variable kernel increases linearly with wavelength, as it is expected
 *          for constant resolution. For reasonable results, the input wavelength
 *          grid should have a constant step size.
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_error_code mf_kernel_synthetic(cpl_table *spec, const mf_parameters *params, const cpl_array *fitpar)
{
    /*!
     * \callgraph
     *
     *
     *
     * \b INPUT:
     * \param spec
     * \param params
     * \param fitpar
     *
     * \b OUTPUT:
     * \param spec    convolved spectrum
     *
     * \b ERRORS:
     * - none
     */

    /* Create kernel table */
    cpl_table *kerntab = cpl_table_new(0);

    /* Calculate kernel table */
    mf_kernel_synthetic_create_table(kerntab, spec, params, fitpar);

    /* Invert kernel table */
    mf_kernel_synthetic_invert_table(kerntab);

    /* Convolve flux column of spectrum with inverted kernel(s) */
    mf_kernel_synthetic_convolve_table(spec, kerntab);

    /* Cleanup */
    cpl_table_delete(kerntab);

    return CPL_ERROR_NONE;
}


/** @cond PRIVATE */

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code
mf_kernel_synthetic_convolution(cpl_array *outkernel, const cpl_array *inkernel1, const cpl_array *inkernel2)
{
    /*!
     * Convolution of two kernels. The output array has to exist. Its size
     * is derived from the input kernel arrays.
     *
     * \note The center of the convolution function (kernel 2) is shifted by
     *       -0.5 pixels for an even number of kernel pixels.
     *
     * \b INPUT:
     * \param inkernel1  first kernel as CPL array
     * \param inkernel2  second kernel as CPL array
     *
     * \b OUTPUT:
     * \param outkernel  output array with convolved kernel
     *
     * \b ERRORS:
     * - No data
     * - Invalid object value(s)
     */

    /* Check for existence of kernel data */
    cpl_size n1 = cpl_array_get_size(inkernel1);
    cpl_size n2 = cpl_array_get_size(inkernel2);
    if (n1 <= 0 || n2 <= 0) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND, "No data: cpl_array *inkernel1 or *inkernel2");
    }

    /* Calculate size of output kernel and initialize it */
    cpl_size n = n1 + n2 - 1;
    cpl_array_set_size(outkernel, n);
    cpl_array_fill_window_double(outkernel, 0, n, 0.);

    /* Get pointers to CPL arrays */
    const double *inkern1 = cpl_array_get_data_double_const(inkernel1);
    const double *inkern2 = cpl_array_get_data_double_const(inkernel2);
    double       *outkern = cpl_array_get_data_double(outkernel);

    /* Check kernel 1 */
    double sum1 = 0;
    for (cpl_size k = 0; k < n1; k++) {
        if (inkern1[k] < 0 || inkern1[k] > 1) {
            return cpl_error_set_message(
                cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                "Invalid object value(s): cpl_array *inkernel1 (kernel element(s) < 0 or > 1)"
            );
        }
        sum1 += inkern1[k];
    }

    if (sum1 < 1 - MF_TOL || sum1 > 1 + MF_TOL) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT,
            "Invalid object value(s): cpl_array *inkernel1 (sum of kernel elements != 1)"
        );
    }

    /* Check kernel 2 */
    double sum2 = 0;
    for (cpl_size k = 0; k < n2; k++) {
        if (inkern2[k] < 0 || inkern2[k] > 1) {
            return cpl_error_set_message(
                cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                "Invalid object value(s): cpl_array *inkernel2 (kernel element(s) < 0 or > 1)"
            );
        }
        sum2 += inkern2[k];
    }

    if (sum2 < 1 - MF_TOL || sum2 > 1 + MF_TOL) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT,
            "Invalid object value(s): cpl_array *inkernel2 (sum of kernel elements != 1)"
        );
    }

    /* Skip convolution if number of pixels is one for one or both kernels */
    if (n1 == 1) {
        for (cpl_size i = 0; i < n2; i++) {
            outkern[i] += inkern2[i];
        }
        return CPL_ERROR_NONE;
    }

    if (n2 == 1) {
        for (cpl_size i = 0; i < n1; i++) {
            outkern[i] += inkern1[i];
        }
        return CPL_ERROR_NONE;
    }

    /* Kernel 2 with even or odd pixel number? Note: center of kernel at -0.5 pixels for even pixel number */
    cpl_size kmin = (n2 % 2 == 0) ? -n2 / 2 : -(n2 - 1) / 2;
    cpl_size kmax = kmin + n2 - 1;

    /* Set range of kernel 1 in output array */
    cpl_size jmin = (int)ceil(0.5 * (n - n1));
    cpl_size jmax = jmin + n1 - 1;

    /* Convolve kernel 1 with kernel 2 */
    for (cpl_size j = jmin; j <= jmax; j++) {
        for (cpl_size k = kmin; k <= kmax; k++) {
            cpl_size i = j + k;
            outkern[i] += inkern1[j - jmin] * inkern2[k - kmin];
        }
    }

    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code mf_kernel_synthetic_create_table(
    cpl_table           *kerntab,
    const cpl_table     *spec,
    const mf_parameters *params,
    const cpl_array     *fitpar
)
{
    /*!
     * \callgraph
     *
     * Calculates convolution kernels for different pixel ranges of an input
     * spectrum and stores them in an output table. Each row of this table
     * gives a pixel range and the correponding kernel as array.
     *
     * The kernels are calculated as a combination of a boxcar, Gaussian, and
     * Lorentzian kernel. Depending on the parameter kernmode, the latter
     * two components can be computed either independently or via a Voigt
     * profile approximation. The width of the boxcar is computed from the
     * width of the spectrograph slit in pixels times a correction factor
     * provided by the fitpar CPL array. The width of the Gaussian and the
     * Lorentzian are directly taken from the fitpar CPL array. The number of
     * pixels of the latter and the Voigt profile depends on the parameter
     * kernfac times FWHM.
     *
     * If the varkern parameter is set to 1, the FWHM of all kernel
     * components linearly increase with wavelength, which results in a
     * constant resolution. In this case, the FWHM from the fitpar array are
     * related to the center of the full wavelength range (considering the
     * data of all chips). The number of kernels provided by the output table
     * depends on mf_LIMRELLAMVAR, which provides the relative wavelength
     * change that causes a recalculation. If varkern is set to 0, only a
     * single kernel is calculated, i.e. the output table has only one row.
     *
     * For a correct application of the routine, it is required that the
     * wavelength difference of adjacent pixels is a constant for the entire
     * spectrum.
     *
     * \b INPUT:
     * \param kerntab   empty CPL table
     * \param spec      input spectrum (CPL table)
     * \param params    mf_parameters parameter structure
     * \param fitpar    CPL array with fit parameters
     *
     * \b OUTPUT:
     * \param kerntab   table with kernels for different pixel ranges
     *
     * \b ERRORS:
     * - No data
     * - Invalid object value(s)
     * - Invalid object structure
     */

    /* Check number of data points in spectrum */
    cpl_size m = cpl_table_get_nrow(spec);
    if (m <= 0) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND, "No data: cpl_table *spec");
    }

    /* Get wavelength range, central wavelength, and wavelength step of input spectrum */
    double wspec[3] = { cpl_table_get(spec, MF_COL_IN_LAMBDA, 0, NULL),
                        cpl_table_get(spec, MF_COL_IN_LAMBDA, m - 1, NULL), 0.5 * (wspec[0] + wspec[1]) };

    double dlam = (wspec[1] - wspec[0]) / (m - 1);

    /* Kernel: linearly increasing with wavelength or constant? */
    cpl_boolean var_kern = params->config->fitting.var_kern;

    /* Get central wavelength */
    cpl_size nchip = cpl_table_get_nrow(params->chiptab);

    double limlam[2] = { cpl_table_get(params->chiptab, MF_COL_WL_MIN, 0, NULL),
                         cpl_table_get(params->chiptab, MF_COL_WL_MAX, nchip - 1, NULL) };

    double reflam = 0.5 * (limlam[0] + limlam[1]);


    /*** Boxcar kernel ***/

    /* Get slit width in pixels (width of boxcar) */
    double slit_width  = params->config->instrumental.slit_width.value;
    double pixel_scale = params->config->instrumental.pixel_scale.value;
    if (pixel_scale == 0.) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Invalid object value(s): pixsc of mf_parameters *params = 0"
        );
    }

    double wbox = slit_width / pixel_scale;

    /* Get relative width of boxcar from CPL array of fit parameters and correct slit width */
    cpl_size nres = cpl_array_get_size(fitpar) - 3;
    if (params->config->inputs.transmission == MF_PARAMETERS_TRANSMISSION_FALSE) {
        nres--;
    }

    wbox *= cpl_array_get(fitpar, nres, NULL);


    /*** Gaussian, Lorentzian, and Voigt profile kernels ***/

    /* Get kernel size in FWHM */
    double kern_fac = params->config->fitting.kern_fac;

    /* Get FWHM of Gaussian in pixels from CPL array of fit parameters */
    nres++;
    double wgauss = cpl_array_get(fitpar, nres, NULL);

    /* Get FWHM of Lorentzian in pixels from CPL array of fit parameters */
    nres++;
    double wlorentz = cpl_array_get(fitpar, nres, NULL);

    /* Independent Gaussian and Lorentzian kernels or Voigt profile approximation? */
    cpl_boolean kern_mode = params->config->fitting.kern_mode;

    /* Create CPL array for central kernel */
    cpl_array *kernel0 = cpl_array_new(0, CPL_TYPE_DOUBLE);

    /* Calculate synthetic kernel for central wavelength */
    mf_kernel_synthetic_calculate(kernel0, wbox, wgauss, wlorentz, kern_fac, kern_mode);

    /* Get number of kernel pixels for central wavelength */
    cpl_size np0 = cpl_array_get_size(kernel0);

    /* Get central pixel of input spectrum */
    int p0c = floor(0.5 * m);

    /* Derive maximum kernel size and number of kernels */
    double x0    = 0.;
    double c     = 0.;
    int    n1r   = 0;
    int    npmax = np0;
    int    nkern = 1;
    if (var_kern) {
        /* Analytical solution for a kernel that linearly increases with wavelength and that is only calculated for positions
         * differing by a constant factor to an integer power; a constant step size of the wavelength grid is required */

        x0 = wspec[2];
        c  = 1 + MF_LIMRELLAMVAR;

        double dx0 = 0.5 * (wspec[2] / reflam) * np0 * dlam;
        double r0  = wspec[2] - wspec[0];

        double xmin = (x0 - r0) * x0 / (x0 + dx0);
        double xmax = (x0 + r0) * x0 / (x0 - dx0);

        n1r     = ceil(log(x0 / xmin) / log(c) - 0.5);
        int n2r = floor(log(xmax / x0) / log(c) + 0.5);

        double dxmaxr = dx0 * pow(c, n2r);

        npmax = ceil(2 * dxmaxr / dlam) + 4;
        nkern = 1 + n1r + n2r;
    }


    /*** Create CPL array for kernel of maximum size ***/

    cpl_array *kernel = cpl_array_new(npmax, CPL_TYPE_DOUBLE);
    cpl_array_fill_window_double(kernel, 0, npmax, 0.);

    cpl_array *tabkernel = cpl_array_new(npmax, CPL_TYPE_DOUBLE);
    cpl_array_fill_window_double(tabkernel, 0, npmax, 0.);

    /* Get pointers to kernel arrays */
    double *kern0   = cpl_array_get_data_double(kernel0);
    double *tabkern = cpl_array_get_data_double(tabkernel);

    /* Create output table columns if required */
    if (cpl_table_has_column(kerntab, MF_COL_PIX_MIN) != 1) {
        cpl_table_new_column(kerntab, MF_COL_PIX_MIN, CPL_TYPE_INT);
    }
    if (cpl_table_has_column(kerntab, MF_COL_PIX_MAX) != 1) {
        cpl_table_new_column(kerntab, MF_COL_PIX_MAX, CPL_TYPE_INT);
    }
    if (cpl_table_has_column(kerntab, MF_COL_PIX_0) != 1) {
        cpl_table_new_column(kerntab, MF_COL_PIX_0, CPL_TYPE_INT);
    }
    if (cpl_table_has_column(kerntab, MF_COL_N_PIX) != 1) {
        cpl_table_new_column(kerntab, MF_COL_N_PIX, CPL_TYPE_INT);
    }
    if (cpl_table_has_column(kerntab, MF_COL_KERNEL) != 1) {
        cpl_table_new_column_array(kerntab, MF_COL_KERNEL, CPL_TYPE_DOUBLE, npmax);
    }

    /* Initialize output table */
    cpl_table_set_size(kerntab, nkern);
    cpl_table_fill_column_window_int(kerntab, MF_COL_PIX_MIN, 0, nkern, 0);
    cpl_table_fill_column_window_int(kerntab, MF_COL_PIX_MAX, 0, nkern, 0);
    cpl_table_fill_column_window_int(kerntab, MF_COL_PIX_0, 0, nkern, 0);
    cpl_table_fill_column_window_int(kerntab, MF_COL_N_PIX, 0, nkern, 0);
    cpl_table_fill_column_window_array(kerntab, MF_COL_KERNEL, 0, nkern, kernel);


    /*** Get pointers to table columns ***/

    cpl_array **kernarr = cpl_table_get_data_array(kerntab, MF_COL_KERNEL);

    int *pmin = cpl_table_get_data_int(kerntab, MF_COL_PIX_MIN);
    int *pmax = cpl_table_get_data_int(kerntab, MF_COL_PIX_MAX);
    int *p0   = cpl_table_get_data_int(kerntab, MF_COL_PIX_0);
    int *np   = cpl_table_get_data_int(kerntab, MF_COL_N_PIX);


    /* Fill output table for constant kernel and return */
    if (!var_kern) {
        pmin[0] = -floor(0.5 * (npmax - 1));
        pmax[0] = m - 1 + floor(0.5 * npmax);

        p0[0] = p0c;
        np[0] = npmax;

        cpl_array_copy_data_double(kernarr[0], kern0);

        cpl_array_delete(kernel0);
        cpl_array_delete(kernel);
        cpl_array_delete(tabkernel);

        return CPL_ERROR_NONE;
    }

    /* Calculate kernels for different pixel ranges */
    for (int j = 0; j < nkern; j++) {
        /* Get central, lower, and upper wavelength of range */
        double clam = x0 * pow(c, j - n1r);
        double llam = clam / sqrt(c);
        double ulam = clam * sqrt(c);

        /* Convert wavelengths into pixels by assuming a constant step size of the wavelength grid */
        int pminj = p0c + floor((llam - x0) / dlam + 0.5);
        int pmaxj = p0c + floor((ulam - x0) / dlam + 0.5);

        /* Skip kernel if it is valid for the same pixels as the previous loop */
        if (j > 0) {
            if (pminj == pmax[j - 1]) {
                pminj++;
            }
            if (pminj > pmaxj) {
                continue;
            }
            if (pmaxj == pmax[j - 1]) {
                continue;
            }
        }

        /* Write pixel ranges into output table */
        pmin[j] = pminj;
        pmax[j] = pmaxj;

        p0[j] = floor(0.5 * (pminj + pmaxj));


        /* Scale FWHM of kernel components */
        double cwbox     = wbox * clam / reflam;
        double cwgauss   = wgauss * clam / reflam;
        double cwlorentz = wlorentz * clam / reflam;

        /* Calculate synthetic kernel */
        mf_kernel_synthetic_calculate(kernel, cwbox, cwgauss, cwlorentz, kern_fac, kern_mode);

        /* Get kernel size and check whether it is not too large for the array column in the output table */
        cpl_size npj = cpl_array_get_size(kernel);
        if (npj > npmax) {
            cpl_array_delete(kernel0);
            cpl_array_delete(kernel);
            cpl_array_delete(tabkernel);
            return cpl_error_set_message(
                cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                "Invalid object structure: cpl_array *kernel (size too large for column 'kernel' in cpl_table *kerntab)"
            );
        }

        np[j] = npj;

        /* Write kernel and its size into output table */
        double *kern = cpl_array_get_data_double(kernel);
        for (int i = 0; i < npj; i++) {
            tabkern[i] = kern[i];
        }
        cpl_array_copy_data_double(kernarr[j], tabkern);
    }

    /* Remove possible empty rows in output table */
    cpl_table_select_all(kerntab);
    cpl_table_and_selected_int(kerntab, MF_COL_N_PIX, CPL_EQUAL_TO, 0);
    cpl_table_erase_selected(kerntab);

    /* Cleanup */
    cpl_array_delete(kernel0);
    cpl_array_delete(kernel);
    cpl_array_delete(tabkernel);

    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code mf_kernel_synthetic_calculate(
    cpl_array        *kernel,
    const double      wbox,
    const double      wgauss,
    const double      wlorentz,
    const double      kernfac,
    const cpl_boolean kernmode
)
{
    /*!
     * Calculates a synthetic kernel consisting of a boxcar, a Gaussian, and a
     * Lorentzian of given FWHM. The convolution of the latter two components
     * can also be substituted by a Voigt profile approximation (parameter
     * kernmode). The size of the Gaussian, Lorentzian, or Voigtian kernels
     * are determined by kernfac, which gives the size in units of FWHM.
     * The sum of the output kernel values is normalised to 1.
     *
     * \b INPUT:
     * \param wbox      FWHM of boxcar in pixels
     * \param wgauss    FWHM of Gaussian in pixels
     * \param wlorentz  FWHM of Lorentzian in pixels
     * \param kernfac   size of Gaussian/Lorentzian kernel in FWHM
     * \param kernmode  kernel mode (approx. for Voigtian profile = 1,
     *                  separate Gaussian and Lorentzian kernels = 0)
     *
     * \b OUTPUT:
     * \param kernel    CPL array containing the kernel elements
     *
     * \b ERRORS:
     * - Invalid input parameter(s)
     */

    /*** Check inputs ***/

    /* Invalid FWHM */
    if (wbox < 0. || wgauss < 0. || wlorentz < 0.) {
        kernel = NULL;
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Invalid input parameter(s): wbox < 0 || wgauss < 0 || wlorentz < 0"
        );
    }

    /* Invalid kernel size */
    if (kernfac < 0.) {
        kernel = NULL;
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Invalid input parameter(s): kernfac < 0");
    }

    /* Create temporary kernel arrays */
    cpl_array *bkernel = cpl_array_new(0, CPL_TYPE_DOUBLE);
    cpl_array *gkernel = cpl_array_new(0, CPL_TYPE_DOUBLE);
    cpl_array *lkernel = cpl_array_new(0, CPL_TYPE_DOUBLE);
    cpl_array *vkernel = cpl_array_new(0, CPL_TYPE_DOUBLE);

    /* Calculate boxcar kernel */
    mf_kernel_synthetic_boxcar(bkernel, wbox);

    /* Independent Gaussian and Lorentzian kernels or Voigt profile approximation? */

    if (kernmode) {
        /* Calculate Voigt profile kernel */
        mf_kernel_synthetic_voigt(vkernel, wgauss, wlorentz, kernfac);
    }
    else {
        /* Calculate Gaussian kernel */
        mf_kernel_synthetic_gauss(gkernel, wgauss, kernfac);

        /* Calculate Lorentzian kernel */
        mf_kernel_synthetic_lorentz(lkernel, wlorentz, kernfac);

        /* Convolve Gaussian with Lorentzian kernel to get Voigtian kernel */
        mf_kernel_synthetic_convolution(vkernel, gkernel, lkernel);
    }

    /* Convolve boxcar with Voigtian kernel to get output kernel */
    mf_kernel_synthetic_convolution(kernel, bkernel, vkernel);

    /* Cleanup */
    cpl_array_delete(bkernel);
    cpl_array_delete(gkernel);
    cpl_array_delete(lkernel);
    cpl_array_delete(vkernel);


    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code mf_kernel_synthetic_boxcar(cpl_array *kernel, const double fwhm)
{
    /*!
     * Calculates boxcar kernel.
     * The sum of the kernel values is normalised to 1.
     *
     * \b INPUT:
     * \param fwhm    width of boxcar in pixels
     *
     * \b OUTPUT:
     * \param kernel  CPL array containing the kernel elements
     *
     * \b ERRORS:
     * - Invalid input parameter(s)
     */

    /* Check invalid FWHM */
    if (fwhm < 0.) {
        kernel = NULL;
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Invalid input parameter(s): fwhm < 0");
    }

    /* Number of kernel pixels */
    double term  = (fwhm - 1.) / 2.;
    int    nkpix = 2 * ceil(term) + 1;
    cpl_array_set_size(kernel, nkpix);

    /* Kernel with one pixel only */
    if (nkpix == 1) {
        cpl_array_set(kernel, 0, 1.);
        return CPL_ERROR_NONE;
    }

    /* Set kernel values */
    double mval = fmod(term, 1.) / fwhm;
    double fval = 1. / fwhm;

    for (cpl_size k = 0; k < nkpix; k++) {
        if (mval > 0 && (k == 0 || k == nkpix - 1)) {
            cpl_array_set(kernel, k, mval);
        }
        else {
            cpl_array_set(kernel, k, fval);
        }
    }


    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code
mf_kernel_synthetic_voigt(cpl_array *kernel, const double wgauss, const double wlorentz, const double kernfac)
{
    /*!
     * Calculates kernel based on a Voigt profile approximation.
     * The sum of the kernel values is normalised to 1.
     * The number of kernel pixels is determined by the product of the Voigt
     * FWHM derived from the input FWHM of Gaussian and Lorentzian and \e
     * kernfac (size of kernel in units of FWHM). Then the upper odd integer
     * number is taken.
     *
     * \b INPUT:
     * \param wgauss    FWHM of Gaussian in pixels
     * \param wlorentz  FWHM of Lorentzian in pixels
     * \param kernfac   size of kernel in FWHM
     *
     * \b OUTPUT:
     * \param kernel    CPL array containing the kernel elements
     *
     * \b ERRORS:
     * - Invalid input parameter(s)
     */

    /* Invalid FWHM */
    if (wgauss < 0. || wlorentz < 0.) {
        kernel = NULL;
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Invalid input parameter(s): wgauss < 0 || wlorentz < 0"
        );
    }

    /* Invalid kernel size */
    if (kernfac < 0.) {
        kernel = NULL;
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Invalid input parameter(s): kernfac < 0");
    }

    /* Gamma of Lorentzian */
    double gam = wlorentz / 2;

    /* FWHM of Voigt profile in pixels */
    double wvoigt = gam + sqrt(gam * gam + wgauss * wgauss);

    /* Ratio of Lorentzian and Voigt profile FWHM */
    double wlwv = wlorentz / wvoigt;

    /* Number of kernel pixels */
    int nkpix = 2 * ceil(wvoigt * kernfac / 2 - 0.5) + 1;
    cpl_array_set_size(kernel, nkpix);

    /* Kernel with one pixel only */
    if (nkpix == 1) {
        cpl_array_set(kernel, 0, 1.);
        return CPL_ERROR_NONE;
    }

    /* Get pointer to CPL array */
    cpl_array_fill_window_double(kernel, 0, nkpix, 0.);
    double *kern = cpl_array_get_data_double(kernel);

    /* Integration limits for upper bin */
    double xmax = 0.5 * nkpix;
    double xmin = xmax - 1;

    /* Number of points per pixel for integration of Voigt profile */
    int nx = ceil(MF_BINS_PER_FWHM / wvoigt);

    /* Step in pixels for integration of Voigt profile */
    double dx = 1. / nx;

    /* Reference pixel for mirroring of kernel values */
    int refpix = floor(nkpix / 2.);

    /* Calculate kernel up to reference pixel */
    for (cpl_size k = nkpix - 1; k >= refpix; k--) {
        if (xmax <= 0.) {
            /* Skip integration */
            kern[k] = 1.;
        }
        else {
            /* First point in pixels for integration */
            double x = xmin + dx / 2;

            /* Perform integration */
            kern[k] = 0.;
            for (cpl_size i = 0; i < nx; i++) {
                /* Get variables of approximation formula */
                double xv    = x / wvoigt;
                double xv2   = xv * xv;
                double xv225 = pow(fabs(xv), 2.25);

                /* Calculate Voigt approximation for integration point */
                kern[k] += (1. - wlwv) * exp(-2.772 * xv2) + wlwv / (1. + 4. * xv2) +
                           0.016 * (1. - wlwv) * wlwv * (exp(-0.4 * xv225) - 10. / (10. + xv225));

                /* Get next integration point */
                x += dx;
            }
            kern[k] /= (double)nx;
        }

        /* Shift integration limits for next bin */
        xmax = xmin;
        xmin = xmax - 1;
    }

    /* Mirror right wing of kernel */
    for (cpl_size k = refpix - 1; k >= 0; k--) {
        kern[k] = kern[nkpix - k - 1];
    }

    /* Add all kernel values */
    double sum = 0.;
    for (cpl_size k = 0; k < nkpix; k++) {
        if (kern[k] < 0.) {
            kern[k] = 0.;
        }
        sum += kern[k];
    }

    /* Normalise kernel values */
    for (cpl_size k = 0; k < nkpix; k++) {
        kern[k] /= sum;
    }

    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code mf_kernel_synthetic_gauss(cpl_array *kernel, const double fwhm, const double kernfac)
{
    /*!
     * Calculates Gaussian kernel.
     * The sum of the kernel values is normalised to 1.
     * The number of kernel pixels is determined by the product of the input
     * parameters FWHM and kernfac (size of kernel in units of FWHM).
     * Then the upper odd integer number is taken.
     *
     * \b INPUT:
     * \param fwhm     FWHM of Gaussian in pixels
     * \param kernfac  size of kernel in FWHM
     *
     * \b OUTPUT:
     * \param kernel   CPL array containing the kernel elements
     *
     * \b ERRORS:
     * - Invalid input parameter(s)
     */

    /* Invalid FWHM */
    if (fwhm < 0.) {
        kernel = NULL;
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Invalid input parameter(s): fwhm < 0");
    }

    /* Invalid kernel size */
    if (kernfac < 0.) {
        kernel = NULL;
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Invalid input parameter(s): kernfac < 0");
    }

    /* sigma of Gaussian */
    double sigma = fwhm / CPL_MATH_FWHM_SIG;

    /* Number of kernel pixels */
    int nkpix = 2 * ceil(fwhm * kernfac / 2 - 0.5) + 1;
    cpl_array_set_size(kernel, nkpix);

    /* Kernel with one pixel only */
    if (nkpix == 1) {
        cpl_array_set(kernel, 0, 1.);
        return CPL_ERROR_NONE;
    }

    /* Get pointer to CPL array */
    cpl_array_fill_window_double(kernel, 0, nkpix, 0.);
    double *kern = cpl_array_get_data_double(kernel);

    /* Integration limits for upper bin */
    double xmax = 0.5 * nkpix;
    double xmin = xmax - 1;

    /* Number of points per pixel for integration of Gaussian */
    int nx = ceil(MF_BINS_PER_FWHM / fwhm);

    /* Step in pixels for integration of Gaussian */
    double dx = 1. / nx;

    /* Reference pixel for mirroring of kernel values */
    int refpix = floor(nkpix / 2.);

    /* Calculate kernel up to reference pixel */
    for (cpl_size k = nkpix - 1; k >= refpix; k--) {
        if (xmax <= 0.) {
            /* Skip integration */
            kern[k] = 1.;
        }
        else {
            /* First point in pixels for integration */
            double x = xmin + dx / 2;

            /* Perform integration */
            kern[k] = 0.;
            for (cpl_size i = 0; i < nx; i++) {
                kern[k] += exp(-0.5 * pow(x / sigma, 2));
                x += dx;
            }
            kern[k] /= (double)nx;
        }

        /* Shift integration limits for next bin */
        xmax = xmin;
        xmin = xmax - 1;
    }

    /* Mirror right wing of kernel */
    for (cpl_size k = refpix - 1; k >= 0; k--) {
        kern[k] = kern[nkpix - k - 1];
    }


    /* Add all kernel values */
    double sum = 0;
    for (cpl_size k = 0; k < nkpix; k++) {
        if (kern[k] < 0.) {
            kern[k] = 0.;
        }
        sum += kern[k];
    }

    /* Normalise kernel values */
    for (cpl_size k = 0; k < nkpix; k++) {
        kern[k] /= sum;
    }

    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code mf_kernel_synthetic_lorentz(cpl_array *kernel, const double fwhm, const double kernfac)
{
    /*!
     * Calculates Lorentzian kernel.
     * The sum of the kernel values is normalised to 1.
     * The number of kernel pixels is determined by the product of the input
     * parameters FWHM and kernfac (size of kernel in units of FWHM).
     * Then the upper odd integer number is taken.
     *
     * \b INPUT:
     * \param fwhm     FWHM of Lorentzian in pixels
     * \param kernfac  size of kernel in FWHM
     *
     * \b OUTPUT:
     * \param kernel   CPL array containing the kernel elements
     *
     * \b ERRORS:
     * - Invalid input parameter(s)
     */

    /* Invalid FWHM */
    if (fwhm < 0.) {
        kernel = NULL;
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Invalid input parameter(s): fwhm < 0");
    }

    /* Invalid kernel size */
    if (kernfac < 0.) {
        kernel = NULL;
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Invalid input parameter(s): kernfac < 0");
    }

    /* Gamma of Lorentzian */
    double gam = fwhm / 2;

    /* Number of kernel pixels */
    int nkpix = 2 * ceil(fwhm * kernfac / 2 - 0.5) + 1;
    cpl_array_set_size(kernel, nkpix);

    /* Kernel with one pixel only */
    if (nkpix == 1) {
        cpl_array_set(kernel, 0, 1.);
        return CPL_ERROR_NONE;
    }

    /* Get pointer to CPL array */
    cpl_array_fill_window_double(kernel, 0, nkpix, 0.);
    double *kern = cpl_array_get_data_double(kernel);

    /* Integration limits for upper bin */
    double xmax = 0.5 * nkpix;
    double xmin = xmax - 1;

    /* Number of points per pixel for integration of Lorentzian */
    int nx = ceil(MF_BINS_PER_FWHM / fwhm);

    /* Step in pixels for integration of Lorentzian */
    double dx = 1. / nx;

    /* Reference pixel for mirroring of kernel values */
    int refpix = floor(nkpix / 2.);

    /* Calculate kernel up to reference pixel */
    for (cpl_size k = nkpix - 1; k >= refpix; k--) {
        if (xmax <= 0.) {
            /* Skip integration */
            kern[k] = 1.;
        }
        else {
            /* First point in pixels for integration */
            double x = xmin + dx / 2;

            /* Perform integration */
            kern[k] = 0.;
            for (cpl_size i = 0; i < nx; i++) {
                kern[k] += gam / (x * x + gam * gam);
                x += dx;
            }
            kern[k] /= (double)nx;
        }

        /* Shift integration limits for next bin */
        xmax = xmin;
        xmin = xmax - 1;
    }

    /* Mirror right wing of kernel */
    for (cpl_size k = refpix - 1; k >= 0; k--) {
        kern[k] = kern[nkpix - k - 1];
    }


    /* Add all kernel values */
    double sum = 0;
    for (cpl_size k = 0; k < nkpix; k++) {
        if (kern[k] < 0.) {
            kern[k] = 0.;
        }
        sum += kern[k];
    }

    /* Normalise kernel values */
    for (cpl_size k = 0; k < nkpix; k++) {
        kern[k] /= sum;
    }

    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code mf_kernel_synthetic_invert_table(cpl_table *kerntab)
{
    /*!
     * Inverts kernels in a table that are valid for different pixel ranges.
     * The resulting kernels provide the contributions of the different
     * pixels to each output pixel. The applied approximation derives the
     * inverted kernels only for the central pixel of each range and uses
     * these kernels for the full range. This approach is consistent with the
     * kernel creation in mf_kernel_calculate_table, which also calculates
     * the kernels only for the central range pixels. The resulting kernels
     * are composed of contributions from different input kernels if their
     * size is larger than the corresponding pixel ranges. Only in the latter
     * case, the output kernels differ from the input kernels and become
     * (slightly) asymmetric. The output kernels are normalised, i.e. the sum
     * of the kernel values equals 1.
     *
     * \b INPUT:
     * \param kerntab   table with kernels for different pixel ranges
     *
     * \b OUTPUT:
     * \param kerntab   table with inverted kernels
     *
     * \b ERRORS:
     * - Invalid object structure
     * - No data
     */

    /* Check existence of required columns in kernel table */
    if (cpl_table_has_column(kerntab, MF_COL_PIX_MIN) != 1 || cpl_table_has_column(kerntab, MF_COL_PIX_MAX) != 1 ||
        cpl_table_has_column(kerntab, MF_COL_PIX_0) != 1 || cpl_table_has_column(kerntab, MF_COL_N_PIX) != 1 ||
        cpl_table_has_column(kerntab, MF_COL_KERNEL) != 1) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT, "Invalid object structure: cpl_table *kerntab (missing column(s))"
        );
    }

    /* Get number of kernels */
    cpl_size nkern = cpl_table_get_nrow(kerntab);
    if (nkern == 0) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND, "No data: cpl_table *kerntab");
    }

    /* Get maximum number of kernel elements */
    cpl_size npmax = cpl_table_get_column_depth(kerntab, MF_COL_KERNEL);
    if (npmax == 0) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT, "Invalid object structure: cpl_table *kerntab (empty kernel arrays)"
        );
    }

    /* Duplicate kernel size column */
    if (cpl_table_has_column(kerntab, MF_COL_T_N_PIX) == 1) {
        cpl_table_erase_column(kerntab, MF_COL_T_N_PIX);
    }
    cpl_table_duplicate_column(kerntab, MF_COL_T_N_PIX, kerntab, MF_COL_N_PIX);

    /* Duplicate kernel array column */
    if (cpl_table_has_column(kerntab, MF_COL_T_KERNEL) == 1) {
        cpl_table_erase_column(kerntab, MF_COL_T_KERNEL);
    }
    cpl_table_duplicate_column(kerntab, MF_COL_T_KERNEL, kerntab, MF_COL_KERNEL);

    /* Create new column for central kernel pixel if required */
    if (cpl_table_has_column(kerntab, MF_COL_KERNCEN) != 1) {
        cpl_table_new_column(kerntab, MF_COL_KERNCEN, CPL_TYPE_INT);
    }
    cpl_table_fill_column_window_int(kerntab, MF_COL_KERNCEN, 0, nkern, 0);

    /* Get pointers to table columns */
    int *pmin = cpl_table_get_data_int(kerntab, MF_COL_PIX_MIN);
    int *pmax = cpl_table_get_data_int(kerntab, MF_COL_PIX_MAX);
    int *p0   = cpl_table_get_data_int(kerntab, MF_COL_PIX_0);
    int *np   = cpl_table_get_data_int(kerntab, MF_COL_N_PIX);
    int *tnp  = cpl_table_get_data_int(kerntab, MF_COL_T_N_PIX);

    /* Get pointer to kernel table columns */
    cpl_array **kernarr  = cpl_table_get_data_array(kerntab, MF_COL_KERNEL);
    cpl_array **tkernarr = cpl_table_get_data_array(kerntab, MF_COL_T_KERNEL);
    int        *kcen     = cpl_table_get_data_int(kerntab, MF_COL_KERNCEN);

    /* Invert kernel(s) */
    for (cpl_size i = 0; i < nkern; i++) {
        /* Get pointer to kernel in table column to be modified */
        double *outkern = cpl_array_get_data_double(kernarr[i]);

        int    kimax = -1;
        double sum   = 0.;

        for (cpl_size j = nkern - 1; j >= 0; j--) {
            /* Derive contributing range of kernel elements */
            int kjcen = floor(0.5 * tnp[j]);
            int kjmin = p0[i] - pmax[j] + kjcen;
            int kjmax = p0[i] - pmin[j] + kjcen;

            /* Skip non-contributing kernels */
            if (kjmax < 0 || kjmin >= tnp[j]) {
                continue;
            }

            /* Avoid invalid kernel elements */
            if (kjmin < 0) {
                kjmin = 0;
            }
            if (kjmax >= tnp[j]) {
                kjmax = tnp[j] - 1;
            }

            /* Make sure that output kernels are not cut due to too a small range of pixels valid for the first or last kernel in the table */
            if (j == 0 && kjmax < tnp[j] - 1) {
                kjmax = tnp[j] - 1;
            }
            if (j == nkern - 1 && kjmin > 0) {
                kjmin = 0;
            }

            /* Elements of output kernel to be filled */
            int nk = kjmax - kjmin + 1;

            int kimin = kimax + 1;
            kimax     = kimin + nk - 1;

            /* Identify central element of output kernel */
            if (j == i) {
                kcen[i] = kimin + kjcen - kjmin;
            }

            /* Make sure that maximum kernel size is not exceeded */
            if (kimax >= npmax) {
                kimax = npmax - 1;
                nk    = kimax - kimin + 1;

                if (nk < 1) {
                    break;
                }
            }

            /* Get pointer to kernel in temporary table column */
            double *inkern = cpl_array_get_data_double(tkernarr[j]);

            /* Copy selected kernel elements from input to output kernel and sum up kernel values*/
            for (cpl_size k = 0; k < nk; k++) {
                outkern[kimin + k] = inkern[kjmin + k];
                sum += outkern[kimin + k];
            }
        }

        /* Set size of inverted kernel */
        np[i] = kimax + 1;

        /* Normalise kernel values */
        for (cpl_size k = 0; k < np[i]; k++) {
            outkern[k] /= sum;
        }
    }

    /* Erase temporary kernel array column */
    cpl_table_erase_column(kerntab, MF_COL_T_N_PIX);
    cpl_table_erase_column(kerntab, MF_COL_T_KERNEL);

    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code mf_kernel_synthetic_convolve_table(cpl_table *spec, const cpl_table *kerntab)
{
    /*!
     * Convolves the flux column of a spectrum with kernels from a table. Each
     * input kernel has to be valid for a fixed pixel range, which is also
     * provided by the kernel table. The convolution is performed in an
     * inverted way, i.e. the kernel gives the contributions of the different
     * pixels to each output pixel.
     *
     * \b INPUT:
     * \param spec      input spectrum (CPL table)
     * \param kerntab   table with kernels for different pixel ranges
     *
     * \b OUTPUT:
     * \param spec      input spectrum with convolved flux
     *
     * \b ERRORS:
     * - No data
     * - Invalid object structure
     */

    /* Check number of data points in spectrum */
    cpl_size m = cpl_table_get_nrow(spec);
    if (m <= 0) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND, "No data: cpl_table *spec");
    }

    /* Check existence of 'flux' column in spectrum table */
    if (cpl_table_has_column(spec, MF_COL_IN_FLUX) != 1) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT, "Invalid object structure: cpl_table *spec (no 'flux' column)"
        );
    }

    /* Check existence of required columns in kernel table */
    if (cpl_table_has_column(kerntab, MF_COL_PIX_MIN) != 1 || cpl_table_has_column(kerntab, MF_COL_PIX_MAX) != 1 ||
        cpl_table_has_column(kerntab, MF_COL_N_PIX) != 1 || cpl_table_has_column(kerntab, MF_COL_KERNEL) != 1) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT, "Invalid object structure: cpl_table *kerntab (missing column(s))"
        );
    }

    /* Get number of kernels */
    cpl_size nkern = cpl_table_get_nrow(kerntab);
    if (nkern == 0) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND, "No data: cpl_table *kerntab");
    }

    /* Get maximum number of kernel elements */
    cpl_size npmax = cpl_table_get_column_depth(kerntab, MF_COL_KERNEL);
    if (npmax == 0) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT, "Invalid object structure: cpl_table *kerntab (empty kernel arrays)"
        );
    }

    /* Get pointers to kernel table columns */
    const cpl_array **kernarr = cpl_table_get_data_array_const(kerntab, MF_COL_KERNEL);
    const int        *kcen    = cpl_table_get_data_int_const(kerntab, MF_COL_KERNCEN);
    const int        *pmin    = cpl_table_get_data_int_const(kerntab, MF_COL_PIX_MIN);
    const int        *pmax    = cpl_table_get_data_int_const(kerntab, MF_COL_PIX_MAX);
    const int        *np      = cpl_table_get_data_int_const(kerntab, MF_COL_N_PIX);

    /* Initialize CPL array for kernel */
    cpl_array *kernel = cpl_array_new(npmax, CPL_TYPE_DOUBLE);
    cpl_array_fill_window_double(kernel, 0, npmax, 0.);

    /* Copy flux column into CPL array */
    double    *vec  = cpl_table_get_data_double(spec, MF_COL_IN_FLUX);
    cpl_array *flux = cpl_array_new(m, CPL_TYPE_DOUBLE);
    cpl_array_copy_data_double(flux, vec);

    /* Create CPL array for convolved flux */
    cpl_array *convflux = cpl_array_new(m, CPL_TYPE_DOUBLE);
    cpl_array_fill_window_double(convflux, 0, m, 0.);

    /* Convolve spectrum with kernel(s) */
    for (cpl_size j = nkern - 1; j >= 0; j--) {
        /* Set pixel range */
        int range[2] = { pmin[j], pmax[j] };

        /* Make sure that the pixel ranges are valid */
        if (range[1] >= 0 && range[0] < m) {
            /* if it's outside the range set to min or max */
            if (range[0] < 0) {
                range[0] = 0;
            }
            if (range[1] >= m) {
                range[1] = m - 1;
            }

            /* Adapt size of kernel array and get pointer */
            cpl_array_set_size(kernel, np[j]);
            double *kern = cpl_array_get_data_double(kernel);

            /* Get pointer to kernel in table */
            const double *tabkern = cpl_array_get_data_double_const(kernarr[j]);

            /* Copy kernel elements into array of correct length */
            for (cpl_size k = 0; k < np[j]; k++) {
                kern[k] = tabkern[k];
            }

            /* Convolve spectrum with inverted kernel */
            mf_kernel_synthetic_convolve_window_inv(convflux, flux, range, kernel, kcen[j]);
        }
    }

    /* Copy resulting flux array into MF_COL_FLUX column of input CPL table */
    vec = cpl_array_get_data_double(convflux);
    cpl_table_copy_data_double(spec, MF_COL_IN_FLUX, vec);

    /* Cleanup */
    cpl_array_delete(kernel);
    cpl_array_delete(flux);
    cpl_array_delete(convflux);

    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code mf_kernel_synthetic_convolve_window_inv(
    cpl_array       *convflux,
    const cpl_array *flux,
    const int        range[2],
    const cpl_array *kernel,
    const int        kerncen
)
{
    /*!
     * Convolution of a flux array with given kernel for a fixed range of
     * pixels. The output array has to exist and have the same number of
     * pixels as the input array. Technically, the convolution is carried out
     * by adding the contributions of all pixels that are linked to a given
     * array pixel via the input kernel. This is performed for all pixels in
     * the selected window. For negative kerncen, the center of the
     * convolution function is the median pixel. For an even number of kernel
     * pixels, it is the higher one of the two central pixels. A positive
     * kerncen directly provides the center of the convolution function.
     * The latter can be important for wide inverted kernels, which can show
     * an asymmetry in the number of elements of both wings.
     *
     * \b INPUT:
     * \param convflux  CPL array of convolved flux values
     * \param flux      CPL array of unconvolved input flux
     * \param range     range of pixels to be considered (minimum and maximum)
     * \param kernel    kernel as CPL array
     *                  (required: sum of all values = 1)
     * \param kerncen   central kernel pixel (-1 = central position)
     *
     * \b OUTPUT:
     * \param convflux  output array with added convolved flux for given range
     *
     * \b ERRORS:
     * - Invalid object structure
     * - Invalid input parameter(s)
     * - Invalid object value(s)
     */

    /* No data points -> no convolution */
    cpl_size n = cpl_array_get_size(flux);
    if (n <= 0) {
        return CPL_ERROR_NONE;
    }

    /* Check correspondence of data points */
    if (cpl_array_get_size(convflux) != n) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
            "Invalid object structure: cpl_array *convflux != cpl_array *flux (size)"
        );
    }

    /* Check range values */
    if (range[0] > range[1]) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Invalid input parameter(s): range[2] (min. > max.)"
        );
    }

    if (range[0] < 0 || range[1] >= n) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Invalid input parameter(s): range[2] (invalid pixel)"
        );
    }

    /* Check validity of kernel center */
    cpl_size nkpix = cpl_array_get_size(kernel);
    if (kerncen >= nkpix) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Invalid input parameter(s): kerncen (invalid kernel element)"
        );
    }

    /* Initialize output array if not done so far */
    if (cpl_array_has_invalid(convflux) == 1) {
        cpl_array_fill_window_double(convflux, 0, n, 0.);
    }

    /* Get pointers to CPL arrays */
    const double *influx  = cpl_array_get_data_double_const(flux);
    double       *outflux = cpl_array_get_data_double(convflux);

    /* No kernel data -> no convolution */
    if (nkpix == 0) {
        for (cpl_size i = range[0]; i <= range[1]; i++) {
            outflux[i] += influx[i];
        }
        return CPL_ERROR_NONE;
    }

    /* Get pointer to CPL array */
    const double *kern = cpl_array_get_data_double_const(kernel);

    /* Check kernel */
    double sum = 0.;
    for (cpl_size k = 0; k < nkpix; k++) {
        if (kern[k] < 0 || kern[k] > 1) {
            for (cpl_size i = range[0]; i <= range[1]; i++) {
                outflux[i] += influx[i];
            }
            return cpl_error_set_message(
                cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                "Invalid object value(s): cpl_array *kernel (kernel element(s) < 0 or > 1)"
            );
        }
        sum += kern[k];
    }

    if (sum < 1 - MF_TOL || sum > 1 + MF_TOL) {
        for (cpl_size i = range[0]; i <= range[1]; i++) {
            outflux[i] += influx[i];
        }
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT,
            "Invalid object value(s): cpl_array *kernel (sum of kernel elements != 1)"
        );
    }

    /* Skip convolution if number of kernel pixels is one */
    if (nkpix == 1) {
        for (cpl_size i = range[0]; i <= range[1]; i++) {
            outflux[i] += influx[i];
        }
        return CPL_ERROR_NONE;
    }

    /* Get range of kernel elements relative to central pixel */
    cpl_size kmin;
    if (kerncen >= 0) {
        /* Use input kernel center if valid */
        kmin = -kerncen;
    }
    else {
        /* Kernel with even or odd pixel number? Note: center of kernel at -0.5 pixels for even pixel number */
        if (nkpix % 2 == 0) {
            kmin = -nkpix / 2;
        }
        else {
            kmin = -(nkpix - 1) / 2;
        }
    }
    cpl_size kmax = kmin + nkpix - 1;

    /* Set flux of virtual input pixels */
    double in0   = influx[0];
    double innm1 = influx[n - 1];

    /* Convolve array with kernel */
    for (cpl_size i = range[0]; i <= range[1]; i++) {
        /* Calculate output flux for each kernel element and add it to the corresponding output pixel */
        for (cpl_size k = kmin; k <= kmax; k++) {
            cpl_size j = i - k;

            /* Flux of real and virtual input pixels */
            double in;
            if (j < 0) {
                in = in0;
            }
            else if (j >= n) {
                in = innm1;
            }
            else {
                in = influx[j];
            }

            outflux[i] += in * kern[k - kmin];
        }
    }

    return CPL_ERROR_NONE;
}

/** @endcond */


/**@}*/
